#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <omp.h>
#include <papi.h>

#define N 1000000
#define N_ITER 10000
#define NUM_EVENTS 2
#define MAX_THREADS 32

int main(int argc, char *argv[])
{
    int i, j, nb_hwc, retval;
    float x[N], y[N];
    int events[NUM_EVENTS] = {PAPI_TOT_INS, PAPI_TOT_CYC};
    char event_name[PAPI_MAX_STR_LEN];
    long_long values[NUM_EVENTS];

    retval = PAPI_library_init(PAPI_VER_CURRENT);
    if (retval != PAPI_VER_CURRENT) {
        fprintf(stderr, "This program was compiled with PAPI %d.%d\n",
                PAPI_VERSION_MAJOR(PAPI_VER_CURRENT),
                PAPI_VERSION_MINOR(PAPI_VER_CURRENT));
        return 1;
    }

    retval= PAPI_thread_init(pthread_self);
    if (retval != PAPI_OK) {
        perror("PAPI_thread_init");
        return 1;
    }

    nb_hwc = PAPI_num_counters();
    if (nb_hwc < PAPI_OK) {
        PAPI_perror("PAPI_num_counters");
        return 1;
    }

    printf("%d available hardware counter(s).\n", nb_hwc);

    for (i = 0; i < N; i++) {
        x[i] = 1.0;
        y[i] = 2.0;
    }

#pragma omp parallel private(retval)
{  
    retval = PAPI_start_counters(events, NUM_EVENTS);
    if (retval != PAPI_OK) {
        PAPI_perror("PAPI_start_counters");
        exit(EXIT_FAILURE);
    }
}

#pragma omp parallel
{
    int my_thread_num = omp_get_thread_num();
    int n = N / (my_thread_num + 1);
 
#pragma omp for private(i)
    for (j = 0; j < N_ITER; j++) {
        for (i = 0; i < n; i++) {
            y[i] = 3.0 * x[i] + y[i];
        }
    }
}

#pragma omp parallel private(values, event_name, i, retval)
{      
    int my_thread_num = omp_get_thread_num();

    retval = PAPI_stop_counters(values, NUM_EVENTS);
    if (retval != PAPI_OK) {
        PAPI_perror("PAPI_stop_counters");
        exit(EXIT_FAILURE);
    }

#pragma omp critical
    {
        for (i = 0; i < NUM_EVENTS; i++) {
            PAPI_event_code_to_name(events[i], event_name);
            printf("Thread %d:%s:%lld\n", my_thread_num, event_name, values[i]);
        }
    }
}

    return 0;
}
