#include <sollya.h>
#include <stdarg.h>

sollya_obj_t stupid_wrapper(sollya_obj_t arg1, sollya_obj_t arg2, sollya_obj_t arg3, ...) {
  va_list va;
  sollya_obj_t a;
  va_start(va, arg3);
  a = sollya_lib_v_remez(arg1, arg2, arg3, va);
  va_end(va);
  return a;
}

int boolean_not_eq(sollya_obj_t a, sollya_obj_t b) {
  sollya_obj_t tmp;
  int res;

  tmp = sollya_lib_cmp_not_equal(a,b);
  res = sollya_lib_is_true(tmp);
  sollya_lib_clear_obj(tmp);
  return res;
}

int callback(sollya_msg_t msg, void *data) {
  (void)msg; /* Avoiding "unused parameter" warning */
  (void)data; /* Avoiding "unused parameter" warning */
  return 0;
}

int main(void) {
  sollya_obj_t f, n, I, w, quality, p, res, pold32, pold64, psvn, prec;
  sollya_lib_init();
  sollya_lib_install_msg_callback(callback, NULL);

  /* This set of tests the problem mentioned in the commit message
     of revision 1270 */

  sollya_lib_printf("=============== Check remez #1 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_EXP(SOLLYA_X_);
  n = SOLLYA_CONST(5);
  I = sollya_lib_parse_string("[-1,1]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("11692536319630510474283968587625525849426869787677b-163 + x * (11692461446776396870604093448366028590354143070281b-163 + x * (11673235319402898388606243422629423700138678461245b-164 + x * (31133428140297179651102037915750686232945795875449b-167 + x * (16385167097562211512230476137789746035022726429977b-168 + x * 3269345410694656957446976530521438920874314212397b-168))))");
  pold64 = sollya_lib_parse_string("11692536319630510474283968587625525849426869787677b-163 + x * (11692461446776396870604093448366028590354143070281b-163 + x * (11673235319402898388606243422629423700138678461245b-164 + x * (31133428140297179651102037915750686232945795875449b-167 + x * (16385167097562211512230476137789746035022726429977b-168 + x * 3269345410694656957446976530521438920874314212397b-168))))");
  psvn = sollya_lib_parse_string("5846268159844744362743250522846987862429367629517b-162 + x * (11692461446497641124271694526853099581211451868505b-163 + x * (46692941277687010312354789479617646356383574093505b-166 + x * (31133428154210205207475891032719442864509304264413b-167 + x * (32770334189338630118937388326942173859103583586297b-169 + x * 26154763134310317818688286279981231602507327130357b-171))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);

  sollya_lib_printf("=============== Check remez #2 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_SIN(SOLLYA_X_);
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[8602502489799048500699352110124879245308378241707b-164;7945546686039470376946513431105230628758836096031b-161]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("2214847272544331422982959495339200583251295560623b-168 + x * (22570554785080561636146441606989328655486420601777b-164 + x * (14059659517842917521277573489957716260107438465453b-167 + x * (-45540662462117273602262338480450403246420473478025b-167 + x * 28992086169774180186447811867835608633187677753845b-169)))");
  pold64 = sollya_lib_parse_string("2214847272544331422982959495339200583251295560623b-168 + x * (22570554785080561636146441606989328655486420601777b-164 + x * (14059659517842917521277573489957716260107438465453b-167 + x * (-45540662462117273602262338480450403246420473478025b-167 + x * 28992086169774180186447811867835608633187677753845b-169)))");
  psvn = sollya_lib_parse_string("35437554945294880980632330593543073201088752576427b-172 + x * (45141109621726235334917783839125819818331107328775b-165 + x * (7029829603586427223675545515975596612915570331385b-166 + x * (-11385165569957834689007715202379973379744083428327b-165 + x * 14496043012911941041556939044573938262006021182555b-168)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #3 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_EXP(SOLLYA_X_);
  n = SOLLYA_CONST(5);
  I = sollya_lib_parse_string("[-1,1]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("127320445892597569233192084573816508818883486078679099554120447022081703359819068009205547b-296 + x * (1018557044797143666503767416322855867839491631117818135205555253332674508098291366773725959b-299 + x * (508441106512784560521103888523310670303733435407883686814016285688113354381980961005656571b-299 + x * (1356052046777263815902350693610623816222104351491496878169889792980744642403155011929747871b-302 + x * (1427349359299711414482673569219142091501984527550101477773398194525418017639951550600094011b-304 + x * 142400076760410063419384360841976625466855068476814073302706323729266395387747260369551141b-303))))");
  pold64 = sollya_lib_parse_string("23385072639206744003725553236226897950127307962451b-164 + x * (365389420211574099793611713376448329829065556119b-158 + x * (46692941283719626881488690119204107293177946206343b-166 + x * (31133428132962382466270199289173155642175257945163b-167 + x * (32770334150519966250277800665629495916849322658691b-169 + x * 13077381702228291991132998535625300995139355424413b-170))))");
  psvn = sollya_lib_parse_string("1018563567135987915012032258711626357126926844785584405249158988500045361779294524522137609b-299 + x * (63659815291914294289514095925732401040544551614636617651518742466446015741886727574488441b-295 + x * (2033764425613398113005508465663637496484258379622733425468204193164419557706436715536867277b-301 + x * (1356052049838784195049149578638111221779213254032530089671765628325726082534242762143843701b-302 + x * (713674681588234494371962064568899835020713913740079267560613729649927766384468681112736739b-303 + x * 142400072661608631070456694965201033405098147932863694799040634951915614775231369251933923b-303))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #4 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_SIN(SOLLYA_X_);
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[374691828310027073697551143672108152896494840720292515058859891294604926811223524123212345b-299;346077367399210202326012086853215365101795134298550595521910142083912300379007927274911891b-296]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("771761635395839434377129531728754385222063722650731773025337381349894526273800455320994827b-306 + x * (491543155523301332510096977904466770120443466902627277040602137207242486357322286394764839b-298 + x * (306192269961748776289513828518754452203023757960747199912707501846371742102754825130961253b-301 + x * (-991787802459613064655442347525062928740559494434023642211380691777391678934404916892326387b-301 + x * 1262783450062636101159847119423691169466818338834735686194803407195747225125865497190175227b-304)))");
  pold64 = sollya_lib_parse_string("4429694543892476662758950169267086222983904426445b-169 + x * (705329837039797126620403885467797756064876572665b-159 + x * (14059659515839393799760722484884113456401859506117b-167 + x * (-22770331230580585570448475640636477599253588269445b-166 + x * 14496043084575263078863254256762631012805774503031b-168)))");
  psvn = sollya_lib_parse_string("771761604782391317266041851785945726969630324975846680133806343191685414408397696546623025b-306 + x * (983086312139836775818762885502129911710245659966259444915127631295702907925608027660771709b-299 + x * (1224769054074531902982422536607908492630645593827460997931865849686410542941640013242217675b-303 + x * (-1983575597502081965714353024113746965208457974615584377507146126565280394079564572724184361b-302 + x * 1262783444340237354442532109902524854573062009580400181046879466613800524190097257252128189b-304)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #5 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_SUB(SOLLYA_POW(SOLLYA_CONST(2), SOLLYA_X_), SOLLYA_CONST(1)),SOLLYA_X_);
  n = SOLLYA_CONST(9);
  I = sollya_lib_parse_string("[-1/16, 1/16]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("32417143657589090276342392320386542212634719195519b-165 + x * (22469851728064591059583753980795080887258113935177b-166 + x * (10383276248605324150931761963547736824435408824845b-167 + x * (28788554626783304799277995343333212138822726194415b-171 + x * (31927528755121801828772085911906676930045165407763b-174 + x * (29507302051812361584763105995068885830047872617559b-177 + x * (46749493035990890820236833700755803008100799330757b-181 + x * (4050534911275574634871329395893993407613062655611b-181 + x * (39932254070374564388452160149914943140225571249661b-188 + x * 44286286888723497574154492217999447288717640509613b-192))))))))");
  pold64 = sollya_lib_parse_string("32417143657589090276342392320386542212634719195519b-165 + x * (22469851728064591059583753980795080887258113935177b-166 + x * (10383276248605324150931761963547736824435408824845b-167 + x * (28788554626783304799277995343333212138822726194415b-171 + x * (31927528755121801828772085911906676930045165407763b-174 + x * (29507302051812361584763105995068885830047872617559b-177 + x * (46749493035990890820236833700755803008100799330757b-181 + x * (4050534911275574634871329395893993407613062655611b-181 + x * (39932254070374564388452160149914943140225571249661b-188 + x * 44286286888723497574154492217999447288717640509613b-192))))))))");
  psvn = sollya_lib_parse_string("16208571828794545138171196160198665353392027726519b-164 + x * (1404365733004036941223984624239966955522849707087b-162 + x * (20766552497210648301863523826884866697474843319139b-168 + x * (28788554626783304799277546456383385755588541941189b-171 + x * (7981882188780450457193021387137975228227525851703b-172 + x * (29507302051812361607733914252846404885311057946759b-177 + x * (23374746517995445671169652317254474972656939309419b-180 + x * (16202139645101822893887622168399062673636756653911b-183 + x * (19966127035178946160378341872570680908647845093767b-187 + x * 22143143457750640206904485509370313176196151922715b-191))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #6 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_EXP(SOLLYA_X_);
  n = SOLLYA_CONST(5);
  I = sollya_lib_parse_string("[0, 1]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("11691999891702459232513188345837804387460473133465b-163 + x * (23385884215836431320692282794470362044234264759653b-164 + x * (23341752494481389797661158971731272634607100367573b-165 + x * (31877473745927764039755782208650650896265284145499b-167 + x * (13020439476544364685564705598260199073785811131065b-168 + x * 10404004557414297823136671010327185596581741545881b-169))))");
  pold64 = sollya_lib_parse_string("11691999891702459232513188345837804387460473133465b-163 + x * (23385884215836431320692282794470362044234264759653b-164 + x * (23341752494481389797661158971731272634607100367573b-165 + x * (31877473745927764039755782208650650896265284145499b-167 + x * (13020439476544364685564705598260199073785811131065b-168 + x * 10404004557414297823136671010327185596581741545881b-169))))");
  psvn = sollya_lib_parse_string("46767999566809558881308044468442218032103592655757b-165 + x * (11692942107914186071032289323545495195098625849317b-163 + x * (1458859530921322116893682516178375390461742133049b-161 + x * (15938736871253002006918388195626326961019565596185b-166 + x * (26040878969624894003265505832639851578483944977057b-169 + x * 41616018202667367394235807063903500749064273666975b-171))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #7 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_COS(SOLLYA_EXP(SOLLYA_X_));
  n = SOLLYA_CONST(5);
  I = sollya_lib_parse_string("[-1b-7, 1b-7]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("25268886549758389380340876080809564841766313745633b-165 + x * (-19676979553010954045662456941477966428585006577331b-164 + x * (-1009731963712275556396770797362269719145679957307b-160 + x * (-25268886572680378753378093437862180139058073796075b-166 + x * (30111938956773510352709131042243854571830406972593b-169 + x * 12980576291156057011865982562724840642811102794567b-166))))");
  pold64 = sollya_lib_parse_string("25268886549758389380340876080809564841766313745633b-165 + x * (-19676979553010954045662456941477966428585006577331b-164 + x * (-1009731963712275556396770797362269719145679957307b-160 + x * (-25268886572680378753378093437862180139058073796075b-166 + x * (30111938956773510352709131042243854571830406972593b-169 + x * 12980576291156057011865982562724840642811102794567b-166))))");
  psvn = sollya_lib_parse_string("25268886549758389380340964094845911511083285927847b-165 + x * (-39353959106021908091317784228352609542993628947795b-165 + x * (-8077855709698204451264858061909024193753897104777b-163 + x * (-25268886572680378560233411601948682207329758068657b-166 + x * (30111938956773161130666968507397664953869736937191b-169 + x * 12980576291149064825229651770061989288082684342499b-166))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #8 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_CONST(1);
  n = sollya_lib_build_list(SOLLYA_CONST(0), SOLLYA_CONST(2), SOLLYA_CONST(4), SOLLYA_CONST(6), SOLLYA_CONST(8), NULL);
  I = sollya_lib_parse_string("[0;18365771228192875685539220798973244513816620097673b-164]");
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_COS(SOLLYA_X_));
  res = stupid_wrapper(f,n,I,w, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("46768052391966959108624215231152264196631503496533b-165 + x^2 * (-46768051993840437443783072518668792259606171959491b-166 + x^2 * (31178661612951199203931279724159830154661317721883b-169 + x^2 * (-16625815622415464147749762639614753800875397430565b-173 + x^2 * 37351022179333407291017870331792433578970727797399b-180)))");
  pold64 = sollya_lib_parse_string("46768052391966959108624215231152264196631503496533b-165 + x^2 * (-46768051993840437443783072518668792259606171959491b-166 + x^2 * (31178661612951199203931279724159830154661317721883b-169 + x^2 * (-16625815622415464147749762639614753800875397430565b-173 + x^2 * 37351022179333407291017870331792433578970727797399b-180)))");
  psvn = sollya_lib_parse_string("23384026195983475555764083291413192636174326996971b-164 + x^2 * (-5846006499229983362883814289569946057071030537265b-163 + x^2 * (31178661612930653056505238164343085429561431777217b-169 + x^2 * (-33251631243914015813578984958842684370701439463593b-174 + x^2 * 37351022158050956678139896134233747310006229285203b-180)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #9 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_EXP(SOLLYA_X_);
  n = SOLLYA_CONST(5);
  I = sollya_lib_parse_string("[0, 1]");
  w = sollya_lib_default();
  quality = sollya_lib_parse_string("1e-5");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("11691999891702459232513188345837804387460473133465b-163 + x * (23385884215836431320692282794470362044234264759653b-164 + x * (23341752494481389797661158971731272634607100367573b-165 + x * (31877473745927764039755782208650650896265284145499b-167 + x * (13020439476544364685564705598260199073785811131065b-168 + x * 10404004557414297823136671010327185596581741545881b-169))))");
  pold64 = sollya_lib_parse_string("11691999891702459232513188345837804387460473133465b-163 + x * (23385884215836431320692282794470362044234264759653b-164 + x * (23341752494481389797661158971731272634607100367573b-165 + x * (31877473745927764039755782208650650896265284145499b-167 + x * (13020439476544364685564705598260199073785811131065b-168 + x * 10404004557414297823136671010327185596581741545881b-169))))");
  psvn = sollya_lib_parse_string("46767999566809558881308044468442218032103592655757b-165 + x * (11692942107914186071032289323545495195098625849317b-163 + x * (1458859530921322116893682516178375390461742133049b-161 + x * (15938736871253002006918388195626326961019565596185b-166 + x * (26040878969624894003265505832639851578483944977057b-169 + x * 41616018202667367394235807063903500749064273666975b-171))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);


  f = SOLLYA_EXP(SOLLYA_X_);
  n = SOLLYA_CONST(5);
  I = sollya_lib_parse_string("[0, 1]");
  w = sollya_lib_default();
  quality = sollya_lib_parse_string("1e-10");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("11691999891702299312469444817572169311673890781507b-163 + x * (5846471053959190028880485661894891163596000435605b-162 + x * (2917719061827678043813769415697570710862370049051b-162 + x * (31877473743673721131764284633492051013566249115055b-167 + x * (26040878964869758854248710691789107998476450084373b-169 + x * 20808009104796731383700609709219405830023419999013b-170))))");
  pold64 = sollya_lib_parse_string("11691999891702299312469444817572169311673890781507b-163 + x * (5846471053959190028880485661894891163596000435605b-162 + x * (2917719061827678043813769415697570710862370049051b-162 + x * (31877473743673721131764284633492051013566249115055b-167 + x * (26040878964869758854248710691789107998476450084373b-169 + x * 20808009104796731383700609709219405830023419999013b-170))))");
  psvn = sollya_lib_parse_string("5845999945851149656234435605422211132352886535717b-162 + x * (23385884215836760115545970932838216522984359588165b-164 + x * (5835438123655356087555656737495441519792436822103b-163 + x * (31877473743673721133327937242850399199174475687415b-167 + x * (6510219741217439713495165031525459507010189269817b-167 + x * 10404004552398365689695293864167514071470455315299b-169))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);


  f = SOLLYA_EXP(SOLLYA_X_);
  n = SOLLYA_CONST(5);
  I = sollya_lib_parse_string("[0, 1]");
  w = sollya_lib_default();
  quality = sollya_lib_parse_string("1e-15");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("11691999891702299312469444817572169311673890781507b-163 + x * (5846471053959190028880485661894891163596000435605b-162 + x * (2917719061827678043813769415697570710862370049051b-162 + x * (31877473743673721131764284633492051013566249115055b-167 + x * (26040878964869758854248710691789107998476450084373b-169 + x * 20808009104796731383700609709219405830023419999013b-170))))");
  pold64 = sollya_lib_parse_string("11691999891702299312469444817572169311673890781507b-163 + x * (5846471053959190028880485661894891163596000435605b-162 + x * (2917719061827678043813769415697570710862370049051b-162 + x * (31877473743673721131764284633492051013566249115055b-167 + x * (26040878964869758854248710691789107998476450084373b-169 + x * 20808009104796731383700609709219405830023419999013b-170))))");
  psvn = sollya_lib_parse_string("5845999945851149656234435605422211132352886535717b-162 + x * (23385884215836760115545970932838216522984359588165b-164 + x * (5835438123655356087555656737495441519792436822103b-163 + x * (31877473743673721133327937242850399199174475687415b-167 + x * (6510219741217439713495165031525459507010189269817b-167 + x * 10404004552398365689695293864167514071470455315299b-169))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #10 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_ATAN(SOLLYA_X_);
  n = SOLLYA_CONST(10);
  I = sollya_lib_parse_string("[-1, 1]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("38372544245022954071593576343710401499b-180 + x * (23380900442023710590629720276282789164224450528167b-164 + x * (-33609089175829249495416553130562627902380011341963b-215 + x * (-7723855757337398739828398738169459706253746872539b-164 + x * (14798371425256328739831059603503168717188731259649b-211 + x * (33702797323732429414901179632614772732191491283097b-167 + x * (-20026589901676225916120391461096241995653144141359b-210 + x * (-7965193355045203212490647632529296700704259316517b-166 + x * (11492309484232489173482584564890302837739876433267b-209 + x * (31196332541693159565915304978636341240498090752081b-170 + x * (-18696265896642848807310482596711689481317128284659b-211))))))))))");
  pold64 = sollya_lib_parse_string("38372544245022954071593576343710401499b-180 + x * (23380900442023710590629720276282789164224450528167b-164 + x * (-33609089175829249495416553130562627902380011341963b-215 + x * (-7723855757337398739828398738169459706253746872539b-164 + x * (14798371425256328739831059603503168717188731259649b-211 + x * (33702797323732429414901179632614772732191491283097b-167 + x * (-20026589901676225916120391461096241995653144141359b-210 + x * (-7965193355045203212490647632529296700704259316517b-166 + x * (11492309484232489173482584564890302837739876433267b-209 + x * (31196332541693159565915304978636341240498090752081b-170 + x * (-18696265896642848807310482596711689481317128284659b-211))))))))))");
  psvn = sollya_lib_parse_string("-299518546557879520665255352509681884284993b-181 + x * (46761800881643993210141123783443178910382594686091b-165 + x * (15626932760797014273458339403729579988713039201177b-197 + x * (-30895422984424026393941474369915254415267721041803b-166 + x * (-37843049683583409517658396527903377324449648118839b-195 + x * (33702797050918309664309509947748141237482896518641b-167 + x * (14340405569685532672251459301997641076318498173369b-192 + x * (-31860772758098094375292352991903847201619947720321b-168 + x * (-34116843989195391948704554536758220176631032690829b-193 + x * (974885357098662804386862783336303497880652218429b-165 + x * 13921338801137066666041733737929563262809760263031b-193)))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #11 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_EXP(SOLLYA_X_);
  n = SOLLYA_CONST(3);
  I = sollya_lib_parse_string("[0, 1]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("46742573553811911796274412764391966160112581418361b-165 + x * (185720745566008984665440218961959381728519963271b-157 + x * (2465278576003452056620348681425508255554707916727b-162 + x * 26187910218233432242621629490193931328425524460129b-166))");
  pold64 = sollya_lib_parse_string("46742573553811911796274412764391966160112581418361b-165 + x * (185720745566008984665440218961959381728519963271b-157 + x * (2465278576003452056620348681425508255554707916727b-162 + x * 26187910218233432242621629490193931328425524460129b-166))");
  psvn = sollya_lib_parse_string("5842821694226678901701403157817208990903244674607b-162 + x * (11886127716219130633708414511730040683306080916975b-163 + x * (19722228608052746177156080281278926582286515128695b-165 + x * 26187910218226727873273123845127088231281394729319b-166))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #12 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_SIN(SOLLYA_X_);
  n = SOLLYA_CONST(10);
  I = sollya_lib_parse_string("[-1, 1]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("-1678574854834777132325060817371809361210735b-195 + x * (5846006547781923868641738892908189653030528304081b-162 + x * (5186811493742158415982843490921456281458058504887b-212 + x * (-31178700608529053074795655135118567830794991320849b-167 + x * (-17684054797287521526384201972347004604281755888439b-211 + x * (24942872635780650464194733939244297067048845011387b-171 + x * (22016393755287390814121400437369894118518307246709b-210 + x * (-4749415908268213847793669454772779765296429152301b-174 + x * (-45380107924050076762231472980106618449214871580875b-211 + x * (32950198265521156768718284738219033131726498607409b-183 + x * 33095953091156378813051015035960771162990683757215b-212)))))))))");
  pold64 = sollya_lib_parse_string("-1678574854834777132325060817371809361210735b-195 + x * (5846006547781923868641738892908189653030528304081b-162 + x * (5186811493742158415982843490921456281458058504887b-212 + x * (-31178700608529053074795655135118567830794991320849b-167 + x * (-17684054797287521526384201972347004604281755888439b-211 + x * (24942872635780650464194733939244297067048845011387b-171 + x * (22016393755287390814121400437369894118518307246709b-210 + x * (-4749415908268213847793669454772779765296429152301b-174 + x * (-45380107924050076762231472980106618449214871580875b-211 + x * (32950198265521156768718284738219033131726498607409b-183 + x * 33095953091156378813051015035960771162990683757215b-212)))))))))");
  psvn = sollya_lib_parse_string("-501469618235094668449707646317635919279433b-200 + x * (11692013095563848406205506363824123741381520090139b-163 + x * (35620680907110926409952304136880869962315563985327b-221 + x * (-31178700608529090497338468003375391002953276015721b-167 + x * (-1128778635011195860269996678779233379792952318519b-213 + x * (24942872635784374822466512879970305036022055002917b-171 + x * (11978333137537591148120847557596099349826247948291b-215 + x * (-9498831816665181898799354229729984976210026113385b-175 + x * (-25536101833773061591914574416528636823559165769255b-216 + x * (32950198285237698762568735027923570972010884588359b-183 + x * 19058765346681769783426726535203697116534992847419b-217)))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #13 ================\n");
  prec = SOLLYA_CONST(400); sollya_lib_set_prec(prec);
  f = SOLLYA_CONST(1);
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_EXP(SOLLYA_X_));
  n = SOLLYA_CONST(25);
  I = sollya_lib_parse_string("[-1/8, 1/8]");
  quality = sollya_lib_parse_string("1e-40");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("161390617380431786853494948250188242145606612051826469551955379941434960076203657512239965672175150832369427924571327933b-396 + x * (2582249878086908589655919172003011874329705792829223512815518586697713929528373906645973178921345938870139944690922674719b-400 + x * (2582249878086908589655919172003011874329705792829223485685832626402974434496485981425174368061267812180286898481509725671b-401 + x * (1721499918724605726437279448002007916219803861886149410406732804556694110264268799366882772800656159764861209972164484205b-402 + x * (430374979681151431609319862000501979054950965471732012462885573145148550524342099381399012383180900245523589700170524133b-402 + x * (1377199934979684581149823558401606332975843089496293355610634191387301345954222763890552010541004557848043938741178151699b-406 + x * (918133289986456387433215705601070888650562042107828893577378984985802017340780034420418039342184959156809188297142744975b-408 + x * (2098590377111900314133064469945304888344142586006558021664462294500550566006715853849729548334078513627267405790955317995b-412 + x * (2098590377111900314133064469945304888345787506900621321180727067543258071161052823401917796984757736187655589072408299063b-415 + x * (466353417135977847585125437765623308514931894662593265387822335180764499496513126666357989921279120928715711788963003853b-416 + x * (1492330934835129112272401400849994541401556850483751940809697007101719533588456093085148424645368218472711086153465369745b-421 + x * (542665794485501495371782327581816454357534185948580690447803027072292049780254099877219461967508882519283166690502899997b-423 + x * (723554392647335327162376436776575188728320134851889625237160943201761614313054851000351657881568088833377803218059300269b-427 + x * (1781056966516517728399695844345844215372471556174952462382582280718355455253866004121406344397890554175851263316922607569b-432 + x * (508873419004719350971341650169098890399521214954235811771143731737284929667556652712937386546656949766148389278605087449b-434 + x * (2171193254420135897477725344179594326124834570679287455570164950528675466483358002551258133738010625366840419395002970151b-440 + x * (542798313605033974370075337706401928002793026768980861612649913973919428294315062471434161634783741775104301987170180913b-442 + x * (2043476004160127903487494000509588015818708867395743746978714423885089063524101669876222772309992862092677104256708028515b-448 + x * (454105778702250630730985753828746431279635370264398739548474403059075817085065739554350706233574401054757346846349616091b-450 + x * (1529619465102318290396191335466982351668824576419822769979570691541195457819322839083484971691416193070080717307820096573b-456 + x * (19120243313792727007427770378223565295754614486585379700107158437774929323650748483979182572831002782497789306950108115b-454 + x * (1864678966975461754778135403670357609027436049075542111503616505395034722356341051966639526803224699875841442314088899867b-465 + x * (678065070284798591971107193700619246254693193913266466079001229674269542931115396375932917051666799806461803260495692375b-468 + x * (1886789853182169718590609346799899921135706970254622139162469534096421624662899054210479361093776718465635395892988821967b-474 + x * (2516112710027923243369908368888976324181415757063417914592398528680717333744550145476637666909737722020354051992072953817b-479 + x * 50307115488146658682733933302724747268538227244369668577358664240031360337093741929589647579922106753044861061577239247b-478))))))))))))))))))))))))");
  pold64 = sollya_lib_parse_string("40347654345107946713373737062547060536401653012956617387988844985358740019050914378059991418043787708092356981303897685b-394 + x * (2582249878086908589655919172003011874329705792829223512815518586697713929528373906645973178921345938870139926226572530419b-400 + x * (2582249878086908589655919172003011874329705792829223485685832626402974434496485981425174368061267812180282125000422255911b-401 + x * (860749959362302863218639724001003958109901930943074705203366402278347055132134399683441386400328079882648031171242699173b-401 + x * (860749959362302863218639724001003958109901930943464024925771146290297101048684198762798024766361800519426800804325423261b-403 + x * (2689843623007196447558249137503137369093443534172447960177019905053322941316841335723734395587899493844083791919392107b-397 + x * (1836266579972912774866431411202141777301124084215657787154757969971604034681560068840836078684369856252482870680526569565b-409 + x * (524647594277975078533266117486326222086035646501639505416115573625137641501678963462432387083804078551349852471424366881b-410 + x * (2098590377111900314133064469945304888345787506900621321180727067543258071161052823401917796924550191857500025280407601681b-415 + x * (932706834271955695170250875531246617029863789325186530775644670361528998993026253332715959587593584168853466498978493359b-417 + x * (746165467417564556136200700424997270700778425241875970404848503550859766794228046542575476250648530710852883838050956953b-420 + x * (2170663177942005981487129310327265817430136743794322761791212108289168199121016399510605899167723492392769350299425793467b-425 + x * (361777196323667663581188218388287594364160067425944812618580471600880807156527425474829065505737939544865331259422833885b-426 + x * (1781056966516517728399695844345844215372471556174952462382582280718355455253865956950153988746212242282738289838998624873b-432 + x * (1017746838009438701942683300338197780799042429908471623542287463474569859335115708752954312779336577592615316744664682969b-435 + x * (2171193254420135897477725344179594326124834570679287455570164950528675466485064591832988938624819005420325820282167690501b-440 + x * (542798313605033974370075337706401928002793026768980861612649913973919428258372130710028952129061378622338285405291189611b-442 + x * (2043476004160127903487494000509588015818708867395743746978714423885089022119822083619278399482467899458468117240299862735b-448 + x * (1816423114809002522923943015314985725118541481057594958193897612236306014723991503049169276327319997307976033324504731927b-452 + x * (1529619465102318290396191335466982351668824576419822769979570691541861911335333959863748886364482689496284252774938835763b-456 + x * (611847786041367264237688652103154089464147663570732150403429069992530290695438702459754703049302351095233153303462760107b-459 + x * (116542435435966359673633462729397350564214753067221381968976030733466566923737898464146533011998759208748515650262757833b-461 + x * (678065070284798591971107193700619246254693193913266466079001447615324869593187070765855363583452756240464620094460862779b-468 + x * (943394926591084859295304673399949960567853485127311069581315469253949592743326197255792712101563256946658640743668942465b-473 + x * (1258056355013961621684954184444488162090707878531708957293674068630135446803760583617837399906318732340257693807559576789b-478 + x * 804913847810346538923742932843595956296611635909914696819171177188150929320002028022779677696622256806586637731183250781b-482))))))))))))))))))))))))");
  psvn = sollya_lib_parse_string("322781234760863573706989896500376484291213224103652939103910759882869920152407315024479931344350301664738855877716953233b-397 + x * (2582249878086908589655919172003011874329705792829223512815518586697713929528373906645973178921345938870139917496983436547b-400 + x * (322781234760863573706989896500376484291213224103652935710729078300371804312060747678146796007658476522534400639048326257b-398 + x * (215187489840575715804659931000250989527475482735768676300841600569586763783033599920860346600082019970677345421220989365b-399 + x * (215187489840575715804659931000250989527475482735866006231442786572574275262171049690699506191590450147611339571114332875b-401 + x * (86074995936230286321863972400100395810990193093518334725664636961706334122138922743159500658812783583670405968683128629b-402 + x * (229533322496614096858303926400267722162640510526957223394344746246450504335195008605104509835545644409092901087604544675b-406 + x * (524647594277975078533266117486326222086035646501639505416115573625137641501678963462432387083862534640861338611109593713b-410 + x * (1049295188555950157066532234972652444172893753450310660590363533771629035580526411700958898544937412080047978435348712663b-414 + x * (932706834271955695170250875531246617029863789325186530775644670361528998993026253332715955100711707168416780337610877655b-417 + x * (1492330934835129112272401400849994541401556850483751940809697007101719533588456093085147444937191584321384263876383340095b-421 + x * (2170663177942005981487129310327265817430136743794322761791212108289168199121016399511022765792218903652988214513312856103b-425 + x * (1447108785294670654324752873553150377456640269703779250474321886403523228626109701995372192338698312632456211997150128931b-428 + x * (55658030203641179012490495135807631730389736130467264449455696272448607976683310769804731474744691911992680097598424237b-427 + x * (2035493676018877403885366600676395561598084859816943247084574926949139718670227902558813571256713100640996854633281035007b-436 + x * (2171193254420135897477725344179594326124834570679287455570164950528675466485542447945601193415112392184189585676454336315b-440 + x * (542798313605033974370075337706401928002793026768980861612649913973919428280091492437959294874814210466569463784562330835b-442 + x * (2043476004160127903487494000509588015818708867395743746978714423885089009799059654815955419940954308858052258556151366221b-448 + x * (1816423114809002522923943015314985725118541481057594958193897612236304581434438367655186328396155237501320381759356490615b-452 + x * (1529619465102318290396191335466982351668824576419822769979570691542070957164975815069679148246384821231067099343136098321b-456 + x * (2447391144165469056950754608412616357856590654282928601613716280000378390117407187338735367488534615108410922687299482825b-461 + x * (932339483487730877389067701835178804513718024537771055751808243625667767026354230066961864474108971019815002730940286519b-464 + x * (169516267571199647992776798425154811563673298478316616519750338800836477027932354552473274172640840616069899166552838595b-466 + x * (471697463295542429647652336699974980283926742563655534790671511774808513886873528374826427763415526242932575396729201841b-472 + x * (2516112710027923243369908368888976324181415757063417914589335963128256773414910370734693737723384103402834997935138370453b-479 + x * 1609827695620693077847485865687191912593223271819829393342670614997429234517774491398970587717345702913258027881874277425b-483))))))))))))))))))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);
  sollya_lib_clear_obj(prec);

  sollya_lib_printf("=============== Check remez #14 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_EXP(SOLLYA_X_);
  n = SOLLYA_CONST(10);
  I = sollya_lib_parse_string("[0, 1]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("23384026197294909850223840992809340481141545456185b-164 + x * (23384026197183194624978796257378849130918970768719b-164 + x * (11692013103063455329995041984738330602777159874659b-164 + x * (31178701049749408969300159772616151147056225354243b-167 + x * (31178718928032908530740508502853005869373615788143b-169 + x * (24942641274774123930888945415406111016671104869359b-171 + x * (33264627827944578300293076057231916380765491375043b-174 + x * (18950258324398416999410271318831779761843002307777b-176 + x * (39016347215689526123052797965843565732733417180009b-180 + x * (14014192292950540150415375020363290646996307648069b-182 + x * 2802308148667058909292393693969468428746054200227b-182)))))))))");
  pold64 = sollya_lib_parse_string("23384026197294909850223840992809340481141545456185b-164 + x * (23384026197183194624978796257378849130918970768719b-164 + x * (11692013103063455329995041984738330602777159874659b-164 + x * (31178701049749408969300159772616151147056225354243b-167 + x * (31178718928032908530740508502853005869373615788143b-169 + x * (24942641274774123930888945415406111016671104869359b-171 + x * (33264627827944578300293076057231916380765491375043b-174 + x * (18950258324398416999410271318831779761843002307777b-176 + x * (39016347215689526123052797965843565732733417180009b-180 + x * (14014192292950540150415375020363290646996307648069b-182 + x * 2802308148667058909292393693969468428746054200227b-182)))))))))");
  psvn = sollya_lib_parse_string("11692013098647454925479766594890768402020389016685b-163 + x * (2923003274647899314022989689912663654170960733739b-161 + x * (23384026206126916717621326407453992440024639124631b-165 + x * (3897337631218641563117791179495583256345586131443b-164 + x * (31178718928039669757532735707023766163548760260885b-169 + x * (24942641274676881997211624204256299106764248859825b-171 + x * (33264627829661543681005336437047562199181910619221b-174 + x * (1184391144687711538704423090650991631165709423125b-172 + x * (4877043417471098209911175115217626323630091258985b-177 + x * (28028384133527983626040751345914065460389879764863b-183 + x * 44836931077334323181376267238444603759641933172525b-186)))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #15 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_CONST(1);
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_COS(SOLLYA_X_));
  n = SOLLYA_CONST(15);
  I = sollya_lib_parse_string("[-1/2, 1/4]");
  quality = sollya_lib_parse_string("1e-40");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("2037035976334486086268445369435520635669474321562577451975655157097896464243257676249067023b-300 + x * (-147022966028724218954630457508230947486371372871749025252274007219232544034284648139415787b-373 + x * (-2037035976334486086267813534277772106898100862459457575607571406931639199411180914730434671b-301 + x * (1644629116256985976487964471064420022272893859535101909440375675266852682766040066939757477b-368 + x * (1358023984222990723447194731402895074478493689559655697573367338981199184886376418609600393b-304 + x * (-1410502123645648389482990176064909235877375828877296670824273606862592472726771255810394497b-361 + x * (-1448558916504522421008699188242670201543249049661405841998293770230912599051296200711718225b-309 + x * (539099188072322128769712711403419973541337473075122764966536485997508305218365952629146421b-354 + x * (827747952288080278806805188627854434960592102720762386536796656170780754344747602124202003b-314 + x * (-797047321997901203825499478107502791492353540573826383432166459902911767841610788036548779b-350 + x * (-1177241534201309254225497658165796336282985949571010147067884030194722915344294071302142951b-321 + x * (2000920995024742776650603097530534335966489792343502135715895342850001531040565868127767b-338 + x * (1141573461565483381928707239907860736890956525739802194008371659491759020811660632918762481b-328 + x * (-1569482484524507248868121030716379419120450572564469525806037183646653943119665927925997873b-347 + x * (-1614534504445457486506762534809318663084383593349434912234239957430267008365062703268491847b-336 + x * (-428026523256046596374630313117538331924801089558422800532774851557691869752368542107371539b-341)))))))))))))))");
  pold64 = sollya_lib_parse_string("23384026197294446691258953661820219905513142999997b-164 + x * (-27003854063734750849884821333808741158467473742595b-241 + x * (-23384026197294446691251700549919982557379607705321b-165 + x * (37758832721834729315051124292478278364221823357325b-233 + x * (1948668849774537223221542671047764362309720717625b-165 + x * (-32383540589224596878145415657450161861333033785079b-226 + x * (-33257281702818745252782110178958829782540667558185b-174 + x * (6188555176873990055241491435716663097447123323431b-218 + x * (38008321946068544570195875158242165410103806748873b-180 + x * (-36598618138537720042729778331398363620481658419391b-216 + x * (-3378517512224125484327326252982467949824603981687b-183 + x * (23520713260932955827973162823434838850936138699919b-212 + x * (13104620655457709642089971023765909010985616378681b-192 + x * (-36033550669396221651928873829838168286171617182909b-212 + x * (-9266973579996565980194798904182086061308550848519b-199 + x * (-39308028131999003166237923968395842357350102726509b-208)))))))))))))))");
  psvn = sollya_lib_parse_string("2037035976334486086268445369435520635669474321562577451975655157097896464243257671397097913b-300 + x * (-1176183728229793751637043660065847579890970982973992202010414015204696627262467139379764029b-376 + x * (-254629497041810760783476691784721513362262607807432196950946425866454899926397153399607781b-298 + x * (1644629116256985976487964471064420022272893859535101909434031953675864992648891286038789435b-368 + x * (1358023984222990723447194731402895074478493689559655697573367338981199184883856076285964445b-304 + x * (-705251061822824194741495088032454617938687914438648335410722373975668856478821775383473093b-360 + x * (-724279458252261210504349594121335100771624524830702920999146885115456297982352023632567717b-308 + x * (539099188072322128769712711403419973541337473075122764966008962096033302633773492746513937b-354 + x * (413873976144040139403402594313927217480296051360381193268398328085389343770403986416032953b-313 + x * (-778366525388575394360839334089358194816751504466627327570123773750386092643938586073693b-340 + x * (-1177241534201309254225497658165796336282985949571010147067884030191605631226200688044978549b-321 + x * (512235774726334150822554392967816790007421386839936546743185078349352360000905812636728755b-346 + x * (1141573461565483381928707239907860736890956525739802194008371657122260546600652960078317033b-328 + x * (-784741242262253624434060515358189709560225286282234762903248657515610266886473073544093209b-346 + x * (-201816813055682185813345316851164832885547949168679364029279847737577066398917512602560307b-333 + x * (-1712106093024186385498521252470153327699204358233691202130959727264412424137740563374441917b-343)))))))))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #16 ================\n");
  prec = SOLLYA_CONST(200); sollya_lib_set_prec(prec);
  f = SOLLYA_SIN(SOLLYA_X_);
  w = SOLLYA_CONST(1);
  n = SOLLYA_CONST(9);
  I = sollya_lib_parse_string("[-1/2, 1/2]");
  quality = sollya_lib_parse_string("1e-40");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("-86947901283126667b-244 + x * (1606938044258570093798407961449753679676162673259975802224191b-200 + x * (13636820948132726187694106438306066516729639249701307318935b-373 + x * (-1071292029371495160966220394362561448878316413905740061801861b-202 + x * (-111990244421936242081011350812209431978491985227156656276615b-371 + x * (857033575383108922734866118457470597964184028156001905494615b-206 + x * (549321510198610685060667636750153381459576297857810582292629b-370 + x * (-652963883135120024306270284879808076963840233212115246631809b-211 + x * (-1596641875356444035698494903347006028859398613195275802297331b-370 + x * 1153615303801817099389035339506468998354123628680403050940707b-218))))))))");
  pold64 = sollya_lib_parse_string("1101702397987b-245 + x * (1606938044258570093798407961449753679676162673259975802192261b-200 + x * (-1391143701206469558761975841650146109945844716683463572892595b-394 + x * (-1071292029371495160966220394362561448878316413905740060751103b-202 + x * (1099410985292744598667779066844465641116416435838202025648363b-390 + x * (857033575383108922734866118457470597964184028156004135919475b-206 + x * (-1161799305280053672790319311141790392778132728191469440939649b-388 + x * (-1305927766270240048612540569759616153927680466426312867749207b-212 + x * (95785462876615352149535634672062757573412322268306607987879b-384 + x * 576807651900908549694517669753234499177061814565628571151813b-217))))))))");
  psvn = sollya_lib_parse_string("5119600653300209b-246 + x * (803469022129285046899203980724876839838081336629987901096367b-199 + x * (-201900659293567833992865902103684434843931295566368931355751b-383 + x * (-1071292029371495160966220394362561448878316413905740060766587b-202 + x * (825124598372502403051371387459200482219416407231744562965875b-380 + x * (428516787691554461367433059228735298982092014078002051962743b-205 + x * (-1009147081261682703749474816682212409401787097671013031504583b-377 + x * (-326481941567560012153135142439904038481920116606570763522007b-210 + x * (366558600702259702423292890113706752781180393271089391738525b-374 + x * 576807651900908549694517669753234499177061814562399320345937b-217))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #17 ================\n");
  prec = SOLLYA_CONST(200); sollya_lib_set_prec(prec);
  f = SOLLYA_CONST(1);
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_EXP(SOLLYA_ADD(SOLLYA_POW(SOLLYA_COS(SOLLYA_X_), SOLLYA_CONST(2)), SOLLYA_CONST(1))));
  n = SOLLYA_CONST(15);
  I = sollya_lib_parse_string("[1,2]");
  quality = sollya_lib_parse_string("1e-40");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1429129772344635854127522980319603763315581962207637816481495b-197 + x * (1258278747211590164438565899307058663065266450700796107832479b-198 + x * (-1202878917246296858587934180920738608442450055679313080135115b-195 + x * (336837707566227764251712761526301661125280453907774843967313b-192 + x * (-1411317735908160715166251154609539762583930751245877998128451b-193 + x * (1180739419570963338745592334849712845368799300827296718761131b-192 + x * (-1400829552020409071501690715419925186599874201865469197848967b-192 + x * (73774156815960633806864131107864831639587911807241292641453b-188 + x * (-720401733120572588026235911018525487050411202136172115523201b-192 + x * (80348824392035439630284753878261997158325598597726556408827b-190 + x * (-416085524045616662367389437501662300792235411427007456053439b-194 + x * (1516396334943608786362280214735888311251638266120825634313019b-198 + x * (-906020503784862624475910695302721150708070846898444362431877b-200 + x * (1187929837442894851249315789697675967541152927109058015663159b-204 + x * (-1304662018163848750887736589024039856901512670169419818894241b-210 + x * (-729069044992515022585488029954641599182740016769331847317347b-212)))))))))))))))");
  pold64 = sollya_lib_parse_string("714564886172317927063761490159801881657790981010465303668201b-196 + x * (1258278747211590164438565899307058663065266454696322711478861b-198 + x * (-18794983081973388415436471576886540756913282158790900538475b-189 + x * (1347350830264911057006851046105206644501121819436710187690425b-194 + x * (-1411317735908160715166251154609539762583930755266304814879647b-193 + x * (590369709785481669372796167424856422684399651964417662529631b-191 + x * (-700414776010204535750845357709962593299937102737693735725465b-191 + x * (1180386509055370140909826097725837306233406592143442790143967b-192 + x * (-720401733120572588026235911018525487050411204371202279457173b-192 + x * (1285581190272567034084556062052191954533209582358572230577769b-194 + x * (-832171048091233324734778875003324601584470826805302456706021b-195 + x * (379099083735902196590570053683972077812909568986721325878589b-196 + x * (-906020503784862624475910695302721150708070855821307509854977b-200 + x * (1187929837442894851249315789697675967541152949455838970639555b-204 + x * (-163082752270481093860967073628004982112689101024071081715501b-207 + x * (-1458138089985030045170976059909283198365479984009101751117121b-213)))))))))))))))");
  psvn = sollya_lib_parse_string("714564886172317927063761490159801881657790980941038093582955b-196 + x * (157284843401448770554820737413382332883158307209856051503421b-195 + x * (-601439458623148429293967090460369304221225030011602564440427b-194 + x * (1347350830264911057006851046105206644501121822298593252104795b-194 + x * (-1411317735908160715166251154609539762583930758300920035042625b-193 + x * (295184854892740834686398083712428211342199826569628293662483b-190 + x * (-350207388005102267875422678854981296649968552055066361959517b-190 + x * (1180386509055370140909826097725837306233406594606596965076293b-192 + x * (-180100433280143147006558977754631371762602801520776769328315b-190 + x * (1285581190272567034084556062052191954533209586044565215103713b-194 + x * (-832171048091233324734778875003324601584470829853727191930371b-195 + x * (758198167471804393181140107367944155625819141777467611978757b-197 + x * (-906020503784862624475910695302721150708070862754694236638027b-200 + x * (148491229680361856406164473712209495942644120860204666721477b-201 + x * (-1304662018163848750887736589024039856901512916200236511645811b-210 + x * (-1458138089985030045170976059909283198365479945115367833194563b-213)))))))))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);
  sollya_lib_clear_obj(prec);

  sollya_lib_printf("=============== Check remez #18 ================\n");
  prec = SOLLYA_CONST(200); sollya_lib_set_prec(prec);
  f = SOLLYA_CONST(1);
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_TAN(SOLLYA_X_));
  n = SOLLYA_CONST(10);
  I = sollya_lib_parse_string("[1/4,1/2]");
  quality = sollya_lib_parse_string("1e-40");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1094653698906394439537746354636570119587248281854588161151645b-219 + x * (803437057336964936961730558645726525457791801950137635572287b-199 + x * (902021694368808954079103247381093078629104460444437390593653b-210 + x * (1056889635883223009274070235439924443715342210868409313160023b-201 + x * (1239694968725463472760134669503154963255295977288145886845797b-205 + x * (282532854374805521596477948893958861358073979772606588154531b-202 + x * (745643752889790461146602566612321479694038184243781141844107b-201 + x * (-1171551728340426070339628381549725358789127811584844764092781b-201 + x * (813974777057437275620501698223620379521525143974703084506525b-200 + x * (-573353867651326030295462629700963357884059020692651314615537b-200 + x * 909290520151942024628769491295084023281483044413698733753071b-202)))))))))");
  pold64 = sollya_lib_parse_string("1094653698906394439537746354636570119587248281853335503513195b-219 + x * (100429632167120617120216319830715815682223975243767208610777b-196 + x * (112752711796101119259887905922636634828638057555448406782225b-207 + x * (528444817941611504637035117719962221857671105434210743859435b-200 + x * (1239694968725463472760134669503154963255295977287215108448943b-205 + x * (282532854374805521596477948893958861358073979772985177962475b-202 + x * (372821876444895230573301283306160739847019092121678155896023b-200 + x * (-146443966042553258792453547693715669848640976448024424165389b-198 + x * (813974777057437275620501698223620379521525143974379564626113b-200 + x * (-1146707735302652060590925259401926715768118041384923094249915b-201 + x * 113661315018992753078596186411885502910185380551687459548631b-199)))))))))");
  psvn = sollya_lib_parse_string("547326849453197219768873177318285059793624140926213615540933b-218 + x * (803437057336964936961730558645726525457791801950137694214707b-199 + x * (902021694368808954079103247381093078629104460442909284056059b-210 + x * (1056889635883223009274070235439924443715342210868431674482813b-201 + x * (1239694968725463472760134669503154963255295977286397777664587b-205 + x * (141266427187402760798238974446979430679036989886667072346625b-201 + x * (372821876444895230573301283306160739847019092121472590934753b-200 + x * (-1171551728340426070339628381549725358789127811583535484887981b-201 + x * (406987388528718637810250849111810189760762571987017133695939b-199 + x * (-573353867651326030295462629700963357884059020692248788608963b-200 + x * 909290520151942024628769491295084023281483044413265206466819b-202)))))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);
  sollya_lib_clear_obj(prec);

  sollya_lib_printf("=============== Check remez #19 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_CONST(1);
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_POW(SOLLYA_X_, SOLLYA_CONST(2.5)));
  n = SOLLYA_CONST(7);
  I = sollya_lib_parse_string("[1,2]");
  quality = sollya_lib_parse_string("1e-40");
  res = stupid_wrapper(f,n,I,w,quality, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("30384532462332540461916109637629717389849715629829b-171 + x * (-3988089244372918017948764449242449608854351178857b-165 + x * (12811690968338316880563803186322867131597579395639b-164 + x * (7556488996758112745826292894088723697015390350955b-163 + x * (-28279885329592063848877875417848304249497177042921b-167 + x * (28332008519536597633619239409928869027803601428177b-169 + x * (-581679350863052459602370172446562830505796444253b-166 + x * 22636516313646583057384577954643601031050496316849b-175))))))");
  pold64 = sollya_lib_parse_string("30384532462332540461916109637629717389849715629829b-171 + x * (-3988089244372918017948764449242449608854351178857b-165 + x * (12811690968338316880563803186322867131597579395639b-164 + x * (7556488996758112745826292894088723697015390350955b-163 + x * (-28279885329592063848877875417848304249497177042921b-167 + x * (28332008519536597633619239409928869027803601428177b-169 + x * (-581679350863052459602370172446562830505796444253b-166 + x * 22636516313646583057384577954643601031050496316849b-175))))))");
  psvn = sollya_lib_parse_string("30384532462332540461916109637629717389849717946267b-171 + x * (-7976178488745836035897528898484899217708702741833b-166 + x * (12811690968338316880563803186322867131597579611375b-164 + x * (30225955987032450983305171576354894788061560871829b-165 + x * (-28279885329592063848877875417848304249497175487639b-167 + x * (14166004259768298816809619704964434513901799366283b-168 + x * (-4653434806904419676818961379572502644046370912779b-169 + x * 45273032627293166114769155909287202062100984362831b-176))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(quality);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #20 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[1;32769b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("519116931250074582514616920029812964033041191732166366208315683368591845444321107391435481b-299 + x * (461432969284079015555143407908324792312068012734235836652935285714259885748478589922847529b-299 + x * (346082648236258546966032322765771598479520265273144739387947665823565730921236322003050251b-299 + x * (-1845774007687049558287217203353987444440450373590070022669568374961280228727054054051744831b-318 + x * 1845774123550678624355073867555708750724188033665817979913979575189149376462087383277483735b-304)))");
  pold64 = sollya_lib_parse_string("519116931250074582514616920029812964033041191732166366208315683368591845444321107391435481b-299 + x * (461432969284079015555143407908324792312068012734235836652935285714259885748478589922847529b-299 + x * (346082648236258546966032322765771598479520265273144739387947665823565730921236322003050251b-299 + x * (-1845774007687049558287217203353987444440450373590070022669568374961280228727054054051744831b-318 + x * 1845774123550678624355073867555708750724188033665817979913979575189149376462087383277483735b-304)))");
  psvn = sollya_lib_parse_string("1038233862500138128475875956868800306632428862675682390630488920717334717811797516870946673b-300 + x * (1845731877136404353345197824363965360441000443429813536018330496370929583427539223063362555b-301 + x * (1384330592944901753130528148077906569022099276087325621882837928258673245249321784698887013b-301 + x * (-1845773996114896639580051046710666959523710035868209257249170813067311098077820745452470793b-318 + x * 461443530887625512479768353802665452066045555725266067356194961948251810773107744094813571b-302)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #21 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32769b-15;16385b-14]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038237383518400955977367009839350382183594881758703803206614495107092340077049399411473967b-300 + x * (115356481865619084755480970440415065182680519289900564187797650531309907130834307553810629b-297 + x * (346093210646321553188501224312760439606720851854109105733752355133531620668552095227966953b-299 + x * (-1384372810757865863246870774013473664314069086622551199217497615222312154655846982330854047b-316 + x * 922915226483104741726827502416104539849150495728611277321715440071753905846249319975793783b-303)))");
  pold64 = sollya_lib_parse_string("1038237383518400955977367009839350382183594881758703803206614495107092340077049399411473967b-300 + x * (115356481865619084755480970440415065182680519289900564187797650531309907130834307553810629b-297 + x * (346093210646321553188501224312760439606720851854109105733752355133531620668552095227966953b-299 + x * (-1384372810757865863246870774013473664314069086622551199217497615222312154655846982330854047b-316 + x * 922915226483104741726827502416104539849150495728611277321715440071753905846249319975793783b-303)))");
  psvn = sollya_lib_parse_string("519118691759194958869938336993917153759287839591458508418456567188501184298624872726100021b-299 + x * (922851854924996828995230076840742652224275493891088804975529618856025119380957142467717349b-300 + x * (1384372842585153765895329298580219550625999434847837181868479524834573737784352062675124987b-301 + x * (-21630825122885172715474116247404241411834884929307264625155299577493640947606454518049021b-310 + x * 461457613241508225907237045562485154919767748953769588607075164270235967889768812953893733b-302)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);

  sollya_lib_printf("=============== Check remez #22 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[16385b-14;32771b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038240905073938809564185146674779242326355205294558935393649861808689793476646895917998267b-300 + x * (1845675539125013381333631624704875717580705390911799805070033763505954280685015813184580603b-301 + x * (346103774023416837754962784984162191045292920130593796757336225574631447590348180153720899b-299 + x * (-576839612789582407852511079083681823850070896509088950771773952643899923582045367370743271b-314 + x * 1845886784100803912445798173029574729693497483658677219270101727506998653158623117662166741b-304)))");
  pold64 = sollya_lib_parse_string("1038240905073938809564185146674779242326355205294558935393649861808689793476646895917998267b-300 + x * (1845675539125013381333631624704875717580705390911799805070033763505954280685015813184580603b-301 + x * (346103774023416837754962784984162191045292920130593796757336225574631447590348180153720899b-299 + x * (-576839612789582407852511079083681823850070896509088950771773952643899923582045367370743271b-314 + x * 1845886784100803912445798173029574729693497483658677219270101727506998653158623117662166741b-304)))");
  psvn = sollya_lib_parse_string("519120452536963884821173244848899789501807316772478296222446759249397538745336823375809823b-299 + x * (1845675539125101694015605706461280777337860878884343298135109298542343529268042667699390915b-301 + x * (1384415096093534892035237675001300304985047505891202043397431733335781664070540161603927133b-301 + x * (-72104951508279319815327385157588544245642023780291058217791044568868088235988999943887227b-311 + x * 922943392050313663616110263716550440504329578501301689158501826794379977010195644488727179b-303)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #23 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32771b-15;8193b-13]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("259561106791707898293039710633779037559822312544256220625440469079039924536763394805587147b-298 + x * (922823682480649549181635253331914708804520679107862064077300309229451099522323876193792447b-300 + x * (692228676735226522485987916737252642808432088179842657020342965630357539103964428753876415b-300 + x * (-1615200212850920429100564708444107526276999411623835378169119182171976132997095979778353947b-315 + x * 1845943116954514373788808938428499536418675482610443460189101910631877163996812552600989749b-304)))");
  pold64 = sollya_lib_parse_string("259561106791707898293039710633779037559822312544256220625440469079039924536763394805587147b-298 + x * (922823682480649549181635253331914708804520679107862064077300309229451099522323876193792447b-300 + x * (692228676735226522485987916737252642808432088179842657020342965630357539103964428753876415b-300 + x * (-1615200212850920429100564708444107526276999411623835378169119182171976132997095979778353947b-315 + x * 1845943116954514373788808938428499536418675482610443460189101910631877163996812552600989749b-304)))");
  psvn = sollya_lib_parse_string("1038244427166820551565755973067147360355241963809994452142709091116644179810577841611351723b-300 + x * (1845647364961387421825523150084311418659650045680119146295042340946491798656971546590638849b-301 + x * (1384457353470320573860561042760847856837132747863114232386055905953473938377168004599301501b-301 + x * (-1615200211404136432467443495420435952306923966226610031809054827327191469221677447918295401b-315 + x * 922971558477168891593098006011029689206256949520522791428989183983865797157679647198709371b-303)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #24 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[8193b-13;32773b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("519123974898574090744797403189736402875820343090487170676158955496764340486780611707469221b-299 + x * (922809593679210720180621035369069149157607579947128535502043286129863499054654321549929891b-300 + x * (692249807357959376265836253883008438741613503989811188430199387556659219846343306612461139b-300 + x * (-129796835530224738633653365732089398163841549341692837949460264183292272644176377903552073b-311 + x * 922999725763696665770522218372845778462978965972769992666437562934620922571048924718626613b-303)))");
  pold64 = sollya_lib_parse_string("519123974898574090744797403189736402875820343090487170676158955496764340486780611707469221b-299 + x * (922809593679210720180621035369069149157607579947128535502043286129863499054654321549929891b-300 + x * (692249807357959376265836253883008438741613503989811188430199387556659219846343306612461139b-300 + x * (-129796835530224738633653365732089398163841549341692837949460264183292272644176377903552073b-311 + x * 922999725763696665770522218372845778462978965972769992666437562934620922571048924718626613b-303)))");
  psvn = sollya_lib_parse_string("1038247949797137138198411810152236042027150025677663036108739190315224689959792247512160749b-300 + x * (1845619187358509774604842467695937623435744055179726882540505213863424591596696213891671669b-301 + x * (346124903678946567323148220705739125716697802928868252801660951027451205747159679274351253b-299 + x * (-129796835439795219968125879277623388697425667763867935773182028537886821315832571839139797b-311 + x * 461499862881804183887307941678648489033912962903390891925428506090593169530838666607860305b-302)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #25 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32773b-15;16387b-14]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("519125736482478728255637096057110002668939858540359785187786624133365569476475713671526253b-299 + x * (1845591006316039313290133964477145081689851856783638624326715045513173982902030827577833881b-301 + x * (173067734978792493403998750245008224483827818145325564230859424100484935217540396100814837b-298 + x * (-1269163346883319747703475308401267430788360043492450542435418710232211881459685542953257601b-314 + x * 923027893909746625680981117047890403034071303800802551819413990779907053305630158715234517b-303)))");
  pold64 = sollya_lib_parse_string("519125736482478728255637096057110002668939858540359785187786624133365569476475713671526253b-299 + x * (1845591006316039313290133964477145081689851856783638624326715045513173982902030827577833881b-301 + x * (173067734978792493403998750245008224483827818145325564230859424100484935217540396100814837b-298 + x * (-1269163346883319747703475308401267430788360043492450542435418710232211881459685542953257601b-314 + x * 923027893909746625680981117047890403034071303800802551819413990779907053305630158715234517b-303)))");
  psvn = sollya_lib_parse_string("129781434120618301441886891494335257230404040267358123905067326858051091215008648316707077b-297 + x * (1845591006316127658316151392181817495790692827365561158476937662377994536076814765442483925b-301 + x * (1384541879830207451864381985396784724052974165647827955280948849702069351845888573147632721b-301 + x * (-1269163346159839445350523220045430584331992463269364606309266712655742118702639730565130127b-314 + x * 230756973477414581247598169075781406036468241818904491493725969017677964460473556657443327b-301)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #26 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[16387b-14;32775b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("129781874583791038442386371655888238747214651568295499870462632615589784132048754166817559b-297 + x * (1845562821833811595888707917735803085968761361790239982389832893025681836759066641151458825b-301 + x * (1384584148813992118218502092824141150820996752581358234459863858555017483845838529217963523b-301 + x * (-1499966094994776220877820593486293830101759281321418876570529901431945302547470745474971957b-314 + x * 923056062915433300256084594722183984064011594047292071874020456477159231091344608769404751b-303)))");
  pold64 = sollya_lib_parse_string("129781874583791038442386371655888238747214651568295499870462632615589784132048754166817559b-297 + x * (1845562821833811595888707917735803085968761361790239982389832893025681836759066641151458825b-301 + x * (1384584148813992118218502092824141150820996752581358234459863858555017483845838529217963523b-301 + x * (-1499966094994776220877820593486293830101759281321418876570529901431945302547470745474971957b-314 + x * 923056062915433300256084594722183984064011594047292071874020456477159231091344608769404751b-303)))");
  psvn = sollya_lib_parse_string("519127498335158630438849944537739003235788144925780928440246986880208839775007090726675597b-299 + x * (1845562821833899951698211738738811196398297128357832936381554474912053986902403755733150637b-301 + x * (1384584148813859610721502078307685229594954837084035891935610394605149725267561036620748249b-301 + x * (-749983047135625881737490579257942200594490085861072813969764878125190845156216961172183267b-313 + x * 1846112125830689993741463623643395385818060771384246846684899278004784835999453500264629919b-304)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #27 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32775b-15;4097b-12]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038258520913329631182690364241114017060837400037458356533335658790054537690003841798971393b-300 + x * (1845534633911397139670115973835376772897004392321237091077394093224806285780259610993209833b-301 + x * (692313210833575385125038578893698920052248173732735304293505140886091558303904658423931033b-300 + x * (-1730782929220944054148482567938745159575983219937856205356812931349500903196307292525312495b-314 + x * 923084232780782923926735159276129795162580209945750248756228435482593334497531568751473455b-303)))");
  pold64 = sollya_lib_parse_string("1038258520913329631182690364241114017060837400037458356533335658790054537690003841798971393b-300 + x * (1845534633911397139670115973835376772897004392321237091077394093224806285780259610993209833b-301 + x * (692313210833575385125038578893698920052248173732735304293505140886091558303904658423931033b-300 + x * (-1730782929220944054148482567938745159575983219937856205356812931349500903196307292525312495b-314 + x * 923084232780782923926735159276129795162580209945750248756228435482593334497531568751473455b-303)))");
  psvn = sollya_lib_parse_string("1038258520913318582835871273068850476933856497077008462622245762799976468802191980219453751b-300 + x * (230691829238935688283021954504460225955832164377297712195680683883387600919481882172279827b-298 + x * (1384626421667018250622821490040263781999764206537059040333196049055481337520408035865245597b-301 + x * (-1730782928497375439674518395023876303183168123788598217714567069661455459900593836351181579b-314 + x * 461542116390347308923267897641860796326923886221496814784474481383857636583464998032289687b-302)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #28 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[4097b-12;32777b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038262045694030331360107288571033481310845936055396192569911648051621914710101903264082523b-300 + x * (1845506442548454768920116619308727687991614007806277523964786136086245050686512829575540843b-301 + x * (173083587298761427962455984082138996639756712278337144683760477571124473370811720671440019b-298 + x * (-1961613850206626209203908481303935475641262125973990140990420781913512806725438551266796249b-314 + x * 28847262609556929122639338428295863535850904646269632285682528473324343941472711236120323b-298)))");
  pold64 = sollya_lib_parse_string("1038262045694030331360107288571033481310845936055396192569911648051621914710101903264082523b-300 + x * (1845506442548454768920116619308727687991614007806277523964786136086245050686512829575540843b-301 + x * (173083587298761427962455984082138996639756712278337144683760477571124473370811720671440019b-298 + x * (-1961613850206626209203908481303935475641262125973990140990420781913512806725438551266796249b-314 + x * 28847262609556929122639338428295863535850904646269632285682528473324343941472711236120323b-298)))");
  psvn = sollya_lib_parse_string("1038262045694019281327644184498830223124624538903533316526000381856241934726227296243277141b-300 + x * (461376610637135786574950413989595397954553370890732717751691452128820892831594675084109125b-299 + x * (692334349194979445970636423727417256143496794350725629890609626014440059141562269050675299b-300 + x * (-980806924741506717818749967271003360851474142455928734971270525619689801236608458317349921b-313 + x * 461556201752866711574665321587582831953372202433881041376295584330220815148852534671601019b-302)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #29 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32777b-15;16389b-14]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("519132785506249659649202446742857299970401223489329249721092671862612630646516898175534471b-299 + x * (922739123872321640347645406112487210240542424232485865309181772337452142304813401978565157b-300 + x * (1384710978983089614554980944010000553236544926984416074420180493723390468154027020533019691b-301 + x * (-274057357324581485608201308026878795188066463784519077329794224330771414587511991256451199b-311 + x * 923140575090575960281429569584646159509857496569850532443973947641407848668443675573329097b-303)))");
  pold64 = sollya_lib_parse_string("519132785506249659649202446742857299970401223489329249721092671862612630646516898175534471b-299 + x * (922739123872321640347645406112487210240542424232485865309181772337452142304813401978565157b-300 + x * (1384710978983089614554980944010000553236544926984416074420180493723390468154027020533019691b-301 + x * (-274057357324581485608201308026878795188066463784519077329794224330771414587511991256451199b-311 + x * 923140575090575960281429569584646159509857496569850532443973947641407848668443675573329097b-303)))");
  psvn = sollya_lib_parse_string("1038265571012488267580081747701507502552864186319462514593895218257388262663254136335223063b-300 + x * (1845478247744731668861670841934712268962889917703749792296961594361126466257897183888722895b-301 + x * (1384710978982957070664622808178739493641569277585846232694551334615901588430012311059927601b-301 + x * (-548114714468248737546338914998330532715841724616354653209882304138110057649247627183294433b-312 + x * 1846281150180975297622580633121261037106757793781048506247239550278406622496601732122485233b-304)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #30 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[16389b-14;32779b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("519134548434402756767156555739491223749783992943450543735006853556479254975332814832586831b-299 + x * (1845450049499621444821257909088314960970174514175996642815635808483444745999950335490924487b-301 + x * (692376631723210447209723438129936623162107424838590538473236247198461508299268646645182651b-300 + x * (-1211658977517938259044519799457477565612229358925289453861820187775265473657297948543474075b-313 + x * 1846337495070143691660987442551713494094189875111533870425503384713134072129251923138744189b-304)))");
  pold64 = sollya_lib_parse_string("519134548434402756767156555739491223749783992943450543735006853556479254975332814832586831b-299 + x * (1845450049499621444821257909088314960970174514175996642815635808483444745999950335490924487b-301 + x * (692376631723210447209723438129936623162107424838590538473236247198461508299268646645182651b-300 + x * (-1211658977517938259044519799457477565612229358925289453861820187775265473657297948543474075b-313 + x * 1846337495070143691660987442551713494094189875111533870425503384713134072129251923138744189b-304)))");
  psvn = sollya_lib_parse_string("519134548434397230064956936127972782382520850925244827929853218836000799005904648968273143b-299 + x * (230681256187463730471925329479143461564111131822598486908688243409777897014209001777846291b-298 + x * (1384753263446288338396241827674000401222192703237048413873503052479299741967864085811668019b-301 + x * (-1211658977156087710136727201386554177595012980707457086826797163585841616876353357638132717b-313 + x * 1846337495069967063330522310584765768262905119753111305766662117014275030040591328883128009b-304)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #31 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32779b-15;8195b-13]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("129784077907877229989358408559022727015506092336158554639793545988337383876342394802174983b-297 + x * (1845421847813048003890891490305799586237781940993530060924167527705002243397401652292399469b-301 + x * (346198887945090207628197443350771030623367818935766056578607553395243263683162740596574271b-299 + x * (-1327095570084590892482248321617848024550507093692816293544367981215387497436366034675858795b-313 + x * 923196920839335626205194440883751587784554165940231967259873678081231946926685373213603605b-303)))");
  pold64 = sollya_lib_parse_string("129784077907877229989358408559022727015506092336158554639793545988337383876342394802174983b-297 + x * (1845421847813048003890891490305799586237781940993530060924167527705002243397401652292399469b-301 + x * (346198887945090207628197443350771030623367818935766056578607553395243263683162740596574271b-299 + x * (-1327095570084590892482248321617848024550507093692816293544367981215387497436366034675858795b-313 + x * 923196920839335626205194440883751587784554165940231967259873678081231946926685373213603605b-303)))");
  psvn = sollya_lib_parse_string("129784077907875848103021982645840680645591515169750492797657603107699291758590525037607837b-297 + x * (1845421847813136413633870703430607052046900605464424242657808159061539467274966216924228471b-301 + x * (346198887945057065588968490245692536303137108974197916838210212886896320833374986466764927b-299 + x * (-331773892430679565249001833341009611065841487884530153968475059201120001178518278702171029b-311 + x * 1846393841678494618689572625930147418834828664945174741628459702119652349640192118655305321b-304)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #32 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[8195b-13;32781b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038276150195205231598046738711326787177258175938486395650936392494584531982139206239184115b-300 + x * (1845393642684581673262535084218842151923675663720046624798283948403833291890563117846027567b-301 + x * (692418921992592502835948587721625658789940441742669912865045829664790944938388346827005581b-300 + x * (-90158700457546112553689981541875410522556478042701804576870900283483380686066639012347575b-309 + x * 1846450190006787079679592267911737638039999300863927226721933815878236131218424338147248637b-304)))");
  pold64 = sollya_lib_parse_string("1038276150195205231598046738711326787177258175938486395650936392494584531982139206239184115b-300 + x * (1845393642684581673262535084218842151923675663720046624798283948403833291890563117846027567b-301 + x * (692418921992592502835948587721625658789940441742669912865045829664790944938388346827005581b-300 + x * (-90158700457546112553689981541875410522556478042701804576870900283483380686066639012347575b-309 + x * 1846450190006787079679592267911737638039999300863927226721933815878236131218424338147248637b-304)))");
  psvn = sollya_lib_parse_string("259569037548798543705211766440273847552185791158218877274658712831809194210870713833745673b-298 + x * (1845393642684670093795418658443966483959499499111436368344376226006811395102256883883741195b-301 + x * (86552365249065776586275418697003276474007004222425941606188960245812786516813868977909749b-297 + x * (-721269603479421542891763001724490699127240426599723243259503557550901344348923138863739103b-312 + x * 230806273750826305071032548154040822033001931080469550661156849783960847391861039573187439b-301)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #33 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32781b-15;16391b-14]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("259569919416359157263353411595457107631487782846370515248264225272015360527824594363826399b-298 + x * (115335339632117571316138611855828589074790000759090941028960355593604439013749368475305519b-297 + x * (173110017507646127293946241187089077523281184437658863294068384210744593210320374774470417b-298 + x * (-389497472387211515714083349816328582000148764030707683808210862949049454218666003932452401b-311 + x * 923253270027271825969308891928787425004862776651570518692548000594247064900609481273701205b-303)))");
  pold64 = sollya_lib_parse_string("259569919416359157263353411595457107631487782846370515248264225272015360527824594363826399b-298 + x * (115335339632117571316138611855828589074790000759090941028960355593604439013749368475305519b-297 + x * (173110017507646127293946241187089077523281184437658863294068384210744593210320374774470417b-298 + x * (-389497472387211515714083349816328582000148764030707683808210862949049454218666003932452401b-311 + x * 923253270027271825969308891928787425004862776651570518692548000594247064900609481273701205b-303)))");
  psvn = sollya_lib_parse_string("519139838832712785294744793651849205116744861032104915377745580281528028175234451240326511b-299 + x * (922682717056984786191037842560048361064150737445945643745434647385992378443837872514948331b-300 + x * (346220035015259106481160210084789300105860147286960056577337502664764130028140026088559053b-299 + x * (-778994944593464631590340766081149572734753525688109985050433618675051212447864257923529627b-312 + x * 923253270027183503718302883172076158284093998622393550288930910988039761075871500306073869b-303))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #34 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[16391b-14;32783b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("32446350177305655626960987935742946812325170002522767312791157638567102933964497546634605b-295 + x * (1845337222100605068161869797262611840822189092245241589797562358357269406524276475910977639b-301 + x * (86557652500536780164080755973761565595940061827114833473856294034809846659646798323076743b-297 + x * (-836723808545697939112234873358000218526157193537343518615496583468179871302592432518607973b-312 + x * 923281445910996724629512929320034672478053969908467355594159490598884955120472381156840557b-303)))");
  pold64 = sollya_lib_parse_string("32446350177305655626960987935742946812325170002522767312791157638567102933964497546634605b-295 + x * (1845337222100605068161869797262611840822189092245241589797562358357269406524276475910977639b-301 + x * (86557652500536780164080755973761565595940061827114833473856294034809846659646798323076743b-297 + x * (-836723808545697939112234873358000218526157193537343518615496583468179871302592432518607973b-312 + x * 923281445910996724629512929320034672478053969908467355594159490598884955120472381156840557b-303)))");
  psvn = sollya_lib_parse_string("1038283205673769919911887025331851626448194016385847841894509031764122005422656005340735635b-300 + x * (1845337222100693510277772059379214171997026956522498276381252238605922803714852220093772511b-301 + x * (1384922440008455878062060395599360175063199452216836309401558676876383623276463266751524873b-301 + x * (-1673447616729456992937773129116785790977599349155587035126314461539364498038188339220166517b-313 + x * 923281445910908399683084596640108535366700793165165294325102396864118622171225970267733811b-303)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #35 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32783b-15;2049b-11]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038286734220307239720704557190733547822740233150419403919615053957098975441209633569686853b-300 + x * (461327251661103022054384452204918703840591591979662598242558653096653916764392844993155701b-299 + x * (173120592978464878523250110885262068594635284888249060190772989330683453346211323951145019b-298 + x * (-894456195135440284141609327212459558817515685541830444876237823904741453274093716964232117b-312 + x * 1846619245309188953313966043331537677921113939942277956355175799511092774945261000205458275b-304)))");
  pold64 = sollya_lib_parse_string("1038286734220307239720704557190733547822740233150419403919615053957098975441209633569686853b-300 + x * (461327251661103022054384452204918703840591591979662598242558653096653916764392844993155701b-299 + x * (173120592978464878523250110885262068594635284888249060190772989330683453346211323951145019b-298 + x * (-894456195135440284141609327212459558817515685541830444876237823904741453274093716964232117b-312 + x * 1846619245309188953313966043331537677921113939942277956355175799511092774945261000205458275b-304)))");
  psvn = sollya_lib_parse_string("259571683555074044470670818178587812230825000699277365821145378951477594615160521312012945b-298 + x * (461327251661125135281638642126583135701747950391370487202780401118873146425858441195315429b-299 + x * (1384964743827586411485602529357234366644873145451430649939007038090334979300009801889793807b-301 + x * (-1788912389908919596374529568446813066503320970901186502380149207073680919363378497109230273b-313 + x * 461654811327253074507525053023055519156890774698540669390486172381238009591314241263639607b-302)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #36 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[2049b-11;32785b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("519145131652542185217707763517530853629482604390562710940905508700387992755729079147518445b-299 + x * (922640393872480403813800177925378752771650649922883158877492845194018341940238359134743875b-300 + x * (692503525759418150173428322959528988817757243557271875472398578625935986248675832975195243b-300 + x * (-1904384209409806576422326601306777292609996776581303101650200507548917112622001929760874319b-313 + x * 461668900129045661844556472102129996706923997672351301852902828249902414108550958025784073b-302)))");
  pold64 = sollya_lib_parse_string("519145131652542185217707763517530853629482604390562710940905508700387992755729079147518445b-299 + x * (922640393872480403813800177925378752771650649922883158877492845194018341940238359134743875b-300 + x * (692503525759418150173428322959528988817757243557271875472398578625935986248675832975195243b-300 + x * (-1904384209409806576422326601306777292609996776581303101650200507548917112622001929760874319b-313 + x * 461668900129045661844556472102129996706923997672351301852902828249902414108550958025784073b-302))");
  psvn = sollya_lib_parse_string("1038290263305073306910021357436777192703295793131639015679471380799055303888403647377259899b-300 + x * (1845280787745049271330801829385601924012697162885023235578234014713148345855702891268450563b-301 + x * (692503525759351835754213767281560999066059958340277525211195856674418520000763995276439531b-300 + x * (-1904384209047823516880656194229500482163654355185846454456001023003869726586038172925283849b-313 + x * 1846675600516005986703186368235938935641456969689551398906012084701101900989550520438953497b-304)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #37 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32785b-15;16393b-14]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("519146896464090670964582799011339625361876312484969902057815620146269565557712154432556743b-299 + x * (922626282700954902775491508103862824059066665916649646534456723782791265130430024881922389b-300 + x * (1385049363082215960042012852378943648217286866499398277925205949319793323173306680123576807b-301 + x * (-2019863074830693468602081973853455808051455628892417786481781217095389893397674055267503279b-313 + x * 1846731957443027016328279822610383179119522193788621019425224466359106076002887389125705337b-304)))");
  pold64 = sollya_lib_parse_string("519146896464090670964582799011339625361876312484969902057815620146269565557712154432556743b-299 + x * (922626282700954902775491508103862824059066665916649646534456723782791265130430024881922389b-300 + x * (1385049363082215960042012852378943648217286866499398277925205949319793323173306680123576807b-301 + x * (-2019863074830693468602081973853455808051455628892417786481781217095389893397674055267503279b-313 + x * 1846731957443027016328279822610383179119522193788621019425224466359106076002887389125705337b-304)))");
  psvn = sollya_lib_parse_string("129786724116021284589522791389533938685366914467921888591421991207973332630177941352280335b-297 + x * (922626282700999140024719752636160971604564766039702962600835043546126010405661939156175315b-300 + x * (1385049363082083319064688842536214923187066318823787115934850164088246633478542653277439281b-301 + x * (-2019863074468688320416400256549979086423852482583986753156892163967545423708598910926249469b-313 + x * 923365978721425175130951042714906397326260406560816810752882959071052073486538252895994867b-303)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #38 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[16393b-14;32787b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038297323089667131239012803580298341305713834793419452962509467851074752565013307757925211b-300 + x * (461306084903729408475343382353118217458409264416593990336519465374065385491184797547928089b-299 + x * (692545839259066841913693084826024353050072653235181386446947448934842309391203558978010225b-300 + x * (-133459311678504620865868670843504381990951036478161257103477502586783643932650946540974923b-309 + x * 923394158044887273321169016644805632854020760156510314516944026455765523834141879892028803b-303)))");
  pold64 = sollya_lib_parse_string("1038297323089667131239012803580298341305713834793419452962509467851074752565013307757925211b-300 + x * (461306084903729408475343382353118217458409264416593990336519465374065385491184797547928089b-299 + x * (692545839259066841913693084826024353050072653235181386446947448934842309391203558978010225b-300 + x * (-133459311678504620865868670843504381990951036478161257103477502586783643932650946540974923b-309 + x * 923394158044887273321169016644805632854020760156510314516944026455765523834141879892028803b-303)))");
  psvn = sollya_lib_parse_string("519148661544828032169112102124382491302320248973389841878107879960509083336343313940712093b-299 + x * (922612169807503059598077710548707697089552788794835145315237686434928383156922308443824261b-300 + x * (1385091678518001030710303067495711898717493641565016862821034871695097707892835720800233351b-301 + x * (-1067674493247023348006567289431697340072623351320326378673614505309301324650879612282858339b-312 + x * 923394158044798937592228773002626337647906422756899856455862690345226526602596370564158317b-303)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #39 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[32787b-15;8197b-13]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038300853789610722717431117998072253672745354894008819884081084657820842766590196380279323b-300 + x * (922598055191821408672718405244320303689195084515518394521613676725148488667342591269985053b-300 + x * (692566998913432581940713261525895859800522806638943732672249466782620232752379964505934067b-300 + x * (-1125420972904246892390459207958219498521812206768675803782281061308773968546984562494103223b-312 + x * 1846844676456477726400395348253828365600567546020264505003689125703323360661624398778675999b-304)))");
  pold64 = sollya_lib_parse_string("1038300853789610722717431117998072253672745354894008819884081084657820842766590196380279323b-300 + x * (922598055191821408672718405244320303689195084515518394521613676725148488667342591269985053b-300 + x * (692566998913432581940713261525895859800522806638943732672249466782620232752379964505934067b-300 + x * (-1125420972904246892390459207958219498521812206768675803782281061308773968546984562494103223b-312 + x * 1846844676456477726400395348253828365600567546020264505003689125703323360661624398778675999b-304)))");
  psvn = sollya_lib_parse_string("1038300853789599654128620928191626403462648427765647651718303542069532121112884285723288433b-300 + x * (57662378449491603544925455540598389841828254930253970683478709975367784588910161998384317b-296 + x * (173141749728341562327965010722690133162605762229874074625618492074160663896377606365502051b-298 + x * (-1125420972723222228136979778653159546641743330995910397970803424610079692501283999601450839b-312 + x * 230855584557037631193855946593581809239481891779533705777169923044462180995976007328272449b-301)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #40 ================\n");
  prec = SOLLYA_CONST(300); sollya_lib_set_prec(prec);
  f = SOLLYA_DIV(SOLLYA_EXP(SOLLYA_X_), SOLLYA_CONST(2));
  n = SOLLYA_CONST(4);
  I = sollya_lib_parse_string("[8197b-13;32789b-15]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("1038304385028081108032949485150642166821786168174196617731484907518431023000345463323466159b-300 + x * (1845167877707743853301029862315370528448617293542570314524618753478073554077018916083953161b-301 + x * (1385176321008686108005887213926430898412220922469336951557518747806594802627115704254710467b-301 + x * (-295792744001313994757328807870081597724043138056186589579893810117232836871800794189163391b-310 + x * 57715657454474657665133753721862497061635908198782446418079866976955603901528724645894937b-299)))");
  pold64 = sollya_lib_parse_string("1038304385028081108032949485150642166821786168174196617731484907518431023000345463323466159b-300 + x * (1845167877707743853301029862315370528448617293542570314524618753478073554077018916083953161b-301 + x * (1385176321008686108005887213926430898412220922469336951557518747806594802627115704254710467b-301 + x * (-295792744001313994757328807870081597724043138056186589579893810117232836871800794189163391b-310 + x * 57715657454474657665133753721862497061635908198782446418079866976955603901528724645894937b-299)))");
  psvn = sollya_lib_parse_string("129788048128508754719487677985588652217030582751023707667272355799437698858583880493596255b-297 + x * (922583938853916180095837031088441763457759698847839528028476278601053096850981462009406181b-300 + x * (1385176321008553430606693149765976057091289953765215499825608320368344363306656509825322385b-301 + x * (-147896371978027533617145218288141306016409404459395846496000360299907460107780973037514293b-309 + x * 1846901038543012363042900512235202156962321012189403619311425629162308800009952703983856039b-304)))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #41 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_SUB(SOLLYA_CONST(1), SOLLYA_DIV(SOLLYA_X_, SOLLYA_SIN(SOLLYA_X_)));
  n = sollya_lib_build_list(SOLLYA_CONST(3), SOLLYA_CONST(5), NULL);
  I = sollya_lib_parse_string("[-1/8; 1/8]");
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_SIN(SOLLYA_X_));
  res = stupid_wrapper(f,n,I,w, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("x^3 * (-31178697691997280151363863339059689087420918752665b-167 + x^2 * 24930042264710159851265878394398318040604898615191b-171)");
  pold64 = sollya_lib_parse_string("x^3 * (-31178697691997280151363863339059689087420918752665b-167 + x^2 * 24930042264710159851265878394398318040604898615191b-171)");
  psvn = sollya_lib_parse_string("x^3 * (-15589348845998640053441772179009544391934198160575b-166 + x^2 * 3116255283088763354828944887199725512210848109683b-168)");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #42 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_CONST(1);
  n = SOLLYA_CONST(6);
  I = sollya_lib_parse_string("[-1b-5; 1b-5]");
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_COS(SOLLYA_X_));
  res = stupid_wrapper(f,n,I,w, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("5846006549323611671784298104022718622318882177971b-162 + x * (37857459261849734002934198358497784854969213609141b-241 + x * (-23384026197294176595383359674367214160107584181507b-165 + x * (-9650697869631352226371375426638274030332473125873b-226 + x * (31178701574267686389540459856148935530651557981629b-169 + x * (28234185975650294540907970780664381466140252467453b-217 + x * (-33256121763484710174675823588818018912416025532385b-174))))))");
  pold64 = sollya_lib_parse_string("5846006549323611671784298104022718622318882177971b-162 + x * (37857459261849734002934198358497784854969213609141b-241 + x * (-23384026197294176595383359674367214160107584181507b-165 + x * (-9650697869631352226371375426638274030332473125873b-226 + x * (31178701574267686389540459856148935530651557981629b-169 + x * (28234185975650294540907970780664381466140252467453b-217 + x * (-33256121763484710174675823588818018912416025532385b-174))))))");
  psvn = sollya_lib_parse_string("46768052394588893374274383349804429530193101540037b-165 + x * (37828133831966473044668513721851715408518819224469b-244 + x * (-46768052394588353190751917165222241452885446611429b-166 + x * (-25427705009315713179984612760631972436763477326811b-230 + x * (1948668848391730452279130065687865733407030940519b-165 + x * (19097661627829399791264180875000091942897986598809b-219 + x * (-33256121763550137230701412675274816909367928737119b-174))))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #43 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = SOLLYA_SIN(SOLLYA_X_);
  n = SOLLYA_CONST(6);
  I = sollya_lib_parse_string("[-1b-5; 1b-5]");
  res = stupid_wrapper(f,n,I, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("2148263723907549819858926287983956687732819b-218 + x * (46768052394587948180729798097226792243112547550905b-165 + x * (-25507165977932281269808564943557481990277558638851b-228 + x * (-31178701565420095891159605334409136402204722409641b-167 + x * (25169562694197869368373056758332575493933394803211b-217 + x * (24941946363609599630853575496590807881363811448161b-171 + x * (-13542288596511770213422971032075701040404674585399b-207))))))");
  pold64 = sollya_lib_parse_string("2148263723907549819858926287983956687732819b-218 + x * (46768052394587948180729798097226792243112547550905b-165 + x * (-25507165977932281269808564943557481990277558638851b-228 + x * (-31178701565420095891159605334409136402204722409641b-167 + x * (25169562694197869368373056758332575493933394803211b-217 + x * (24941946363609599630853575496590807881363811448161b-171 + x * (-13542288596511770213422971032075701040404674585399b-207))))))");
  psvn = sollya_lib_parse_string("-31657351710771318264480101926512972684047b-216 + x * (11692013098646987045182253340122523086369332160129b-163 + x * (45142061063915304495253236000306896279355752461095b-232 + x * (-15589350782710047985431039322670567918156898471475b-166 + x * (-32638902245554706260123356836597206211247789846683b-220 + x * (12470973181805555448555253907066439009244525141935b-170 + x * 43769018414430920138875951265888862687962052230975b-211)))))");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(prec);


  sollya_lib_printf("=============== Check remez #44 ================\n");
  prec = sollya_lib_default(); sollya_lib_set_prec(prec);
  f = sollya_lib_parse_string("1-(1+x+x^2/2)/exp(x)");
  n = sollya_lib_build_list(SOLLYA_CONST(3), SOLLYA_CONST(4), NULL);
  I = sollya_lib_parse_string("[-1b-8;12167298183958086083256693241475396846566462796809b-171]");
  w = SOLLYA_DIV(SOLLYA_CONST(1), SOLLYA_EXP(SOLLYA_X_));
  res = stupid_wrapper(f,n,I,w, NULL);
  p = sollya_lib_horner(res);
  pold32 = sollya_lib_parse_string("x^3 * (7794680717168470900797342378780962700124730873177b-165 + x * 31179902867931546748200780881663744737011812548003b-169)");
  pold64 = sollya_lib_parse_string("x^3 * (7794680717168470900797342378780962700124730873177b-165 + x * 31179902867931546748200780881663744737011812548003b-169)");
  psvn = sollya_lib_parse_string("x^3 * (31178722868673883627864298068285024212112392034985b-167 + x * 7794975716982886342439431644725264558026848337299b-167)");
  if (boolean_not_eq(p, psvn) && boolean_not_eq(p, pold32) && boolean_not_eq(p, pold64))
    sollya_lib_printf("%b\n", p);
  else sollya_lib_printf("Check that mpfr_urandom has the expected behavior (either one of the old 32-bit or 64-bit behavior), or the new one. OK\n");
  sollya_lib_clear_obj(psvn);
  sollya_lib_clear_obj(pold32);
  sollya_lib_clear_obj(pold64);
  sollya_lib_clear_obj(p);
  sollya_lib_clear_obj(res);
  sollya_lib_clear_obj(f);
  sollya_lib_clear_obj(n);
  sollya_lib_clear_obj(I);
  sollya_lib_clear_obj(w);
  sollya_lib_clear_obj(prec);


  sollya_lib_close();
  return 0;
}
