module hydro_mpi
  use mpi

  implicit none

  integer, parameter :: NORTH=1
  integer, parameter :: SOUTH=2
  integer, parameter :: WEST=3
  integer, parameter :: EAST=4

  integer, parameter :: NDIMS=2

  integer, parameter                  :: etiquette=100

  integer :: ntx !Global nx
  integer :: nty !Global ny
  integer :: startx !Global x-position
  integer :: starty !Global y-position

  integer                             :: nb_procs, rang, code
  integer                             :: comm2D    ! 2D cartesian topology communicator
  integer                             :: bloc_dim1 ! Derived datatypes to send boundary blocks on dimension 1 (X)
  integer                             :: bloc_dim2 ! Derived datatypes to send boundary blocks on dimension 2 (Y)
  logical                             :: reorganize
  integer, dimension(NDIMS)           :: coords, dims
  logical, dimension(NDIMS)           :: periods
  integer, dimension(2*NDIMS)         :: voisins

contains

subroutine init_mpi
  use hydro_parameters

  integer :: type_temp, size_double
  integer(kind=MPI_ADDRESS_KIND) :: stride

  call MPI_COMM_SIZE(MPI_COMM_WORLD,nb_procs,code)

  ! Create 2D cartesian topology
  dims(:)=0
  call MPI_DIMS_CREATE(nb_procs,ndims,dims,code)
  periods(:)=.false.
  if(boundary_left==3 .AND. boundary_right==3) periods(1)=.true.
  if(boundary_up  ==3 .AND. boundary_down ==3) periods(2)=.true.
  reorganize=.true.
  call MPI_CART_CREATE(MPI_COMM_WORLD,ndims,dims,periods,reorganize,comm2d,code)
  call MPI_COMM_RANK(comm2d,rang,code)
  call MPI_CART_COORDS(comm2d,rang,ndims,coords,code)

  if (rang==0) then
     print *
     print *,' MPI Execution with ',nb_procs,' processes (',dims(1),'x',dims(2),')'
     print *,' Starting time integration, nx = ',nx,' ny = ',ny
     print *
  endif

  ! Calcul de la taille des sous-domaines locaux
  ntx = nx
  nty = ny

  startx=coords(1)*(ntx/dims(1))+min(coords(1),mod(ntx,dims(1)))
  if (coords(1)<mod(nx,dims(1))) then
     nx=nx/dims(1)+1
  else
     nx=nx/dims(1)
  endif

  starty=coords(2)*(nty/dims(2))+min(coords(2),mod(nty,dims(2)))
  if (coords(2)<mod(ny,dims(2))) then
     ny=ny/dims(2)+1
  else
     ny=ny/dims(2)
  endif

  ! Modify boundary conditions and determine neighbors list
  ! 0 stands for MPI boundary conditions
  voisins(:)=MPI_PROC_NULL
  call MPI_CART_SHIFT(comm2d,0,1,voisins(WEST), voisins(EAST), code)
  call MPI_CART_SHIFT(comm2d,1,1,voisins(SOUTH),voisins(NORTH),code)
  if(voisins(NORTH)/=MPI_PROC_NULL) boundary_up   =0
  if(voisins(SOUTH)/=MPI_PROC_NULL) boundary_down =0
  if(voisins(WEST) /=MPI_PROC_NULL) boundary_left =0
  if(voisins(EAST) /=MPI_PROC_NULL) boundary_right=0

  ! Create derived datatypes
  call MPI_TYPE_SIZE(MPI_DOUBLE_PRECISION,size_double,code)
  stride = (nx+4)*(ny+4)*size_double

  call MPI_TYPE_VECTOR(ny,2,nx+4,MPI_DOUBLE_PRECISION,type_temp,code)
  call MPI_TYPE_CREATE_HVECTOR(nvar,1,stride,type_temp,bloc_dim1,code)
  call MPI_TYPE_COMMIT(bloc_dim1,code)

  call MPI_TYPE_VECTOR(2,nx,nx+4,MPI_DOUBLE_PRECISION,type_temp,code)
  call MPI_TYPE_CREATE_HVECTOR(nvar,1,stride,type_temp,bloc_dim2,code)
  call MPI_TYPE_COMMIT(bloc_dim2,code)

end subroutine init_mpi

subroutine clean_mpi

  call MPI_TYPE_FREE(bloc_dim1,code)
  call MPI_TYPE_FREE(bloc_dim2,code)
  call MPI_COMM_FREE(comm2d,code)

end subroutine clean_mpi

end module hydro_mpi
