module hydro_pnetcdf
#ifdef PNETCDF
  use hydro_commons
  use hydro_mpi
  use hydro_parameters
  use hydro_utils
  use pnetcdf
  implicit none

  integer(kind=mpi_offset_kind),dimension(1),parameter,private::one1d=1

  integer,private :: fh, mode, status

contains

subroutine output_pnetcdf
  implicit none

  integer :: nout
  character(LEN=5)  :: charoutnum
  character(LEN=80) :: filename

  integer :: ntx_dim_id,nty_dim_id,nvar_dim_id
  integer :: gamma_id,nstep_id,ntx_id,nty_id,nvar_id,t_id,u_id
  integer(kind=MPI_OFFSET_KIND),dimension(3) :: counts,starts

  if (rang==0) print *,'Parallel netCDF: Outputting array of size=',ntx,nty,nvar

  call timer_start()

  !Determine filename
  nout=nstep/noutput
  call title(nout,charoutnum)
  filename='output_'//trim(charoutnum)//'.nc'

  !Create file
  mode = NF_64BIT_DATA
  status = nfmpi_create(comm2d,filename,mode,MPI_INFO_NULL,fh);call CHKERR(status)

  !Define dimensions
  status = nfmpi_def_dim(fh,'ntx', int(ntx, kind=mpi_offset_kind),ntx_dim_id)
  status = nfmpi_def_dim(fh,'nty', int(nty, kind=mpi_offset_kind),nty_dim_id)
  status = nfmpi_def_dim(fh,'nvar',int(nvar,kind=mpi_offset_kind),nvar_dim_id)

  !Define variables
  status = nfmpi_def_var(fh,'t',    NF_DOUBLE,0,(/0/),t_id)
  status = nfmpi_def_var(fh,'gamma',NF_DOUBLE,0,(/0/),gamma_id)
  status = nfmpi_def_var(fh,'ntx',  NF_INT,   0,(/0/),ntx_id)
  status = nfmpi_def_var(fh,'nty',  NF_INT,   0,(/0/),nty_id)
  status = nfmpi_def_var(fh,'nvar', NF_INT,   0,(/0/),nvar_id)
  status = nfmpi_def_var(fh,'nstep',NF_INT,   0,(/0/),nstep_id)

  status = nfmpi_def_var(fh,'u',NF_DOUBLE,3,(/ntx_dim_id,nty_dim_id,nvar_dim_id/),u_id)

  !Definition of dimensions and variables finished
  status = nfmpi_enddef(fh);call CHKERR(status)

  !Write run parameters
  status = nfmpi_begin_indep_data(fh);call CHKERR(status)
  if (rang==0) then
     status = nfmpi_put_var1_double(fh,t_id,    one1d,t);    call CHKERR(status)
     status = nfmpi_put_var1_double(fh,gamma_id,one1d,gamma);call CHKERR(status)
     status = nfmpi_put_var1_int   (fh,ntx_id,  one1d,ntx);  call CHKERR(status)
     status = nfmpi_put_var1_int   (fh,nty_id,  one1d,nty);  call CHKERR(status)
     status = nfmpi_put_var1_int   (fh,nvar_id, one1d,nvar); call CHKERR(status)
     status = nfmpi_put_var1_int   (fh,nstep_id,one1d,nstep);call CHKERR(status)
  end if

  status = nfmpi_end_indep_data(fh);call CHKERR(status)

  starts(1) = startx+1 ; starts(2) = starty+1 ; starts(3) = 1
  counts(1) = nx       ; counts(2) = ny       ; counts(3) = nvar
  status = nfmpi_put_vara_double_all(fh,u_id,starts,counts,uold(3:nx+2,3:ny+2,1:nvar));call CHKERR(status)

  status = nfmpi_close(fh);call CHKERR(status)

  call timer_stop("Parallel netCDF",.true.,int(ntx,kind=8)*nty*nvar*8)

end subroutine output_pnetcdf


subroutine input_pnetcdf
  implicit none

  integer :: ntx_prov,nty_prov,nvar_prov
  real(kind=prec_real) :: gamma_prov
  real(kind=8),dimension(1) :: td
  integer,dimension(1) :: ti
  integer :: var_id
  integer(kind=MPI_OFFSET_KIND),dimension(3) :: counts,starts

  if (rang==0) print *,'Parallel netCDF: Reading file ',trim(restart_file)

  call timer_start()

  mode = NF_NOWRITE
  status = nfmpi_open(comm2d,trim(restart_file),mode,MPI_INFO_NULL,fh)

  !Read run parameters
  status = nfmpi_inq_varid(fh,'t',    var_id);status = nfmpi_get_var_double_all(fh,var_id,td);t=td(1)
  status = nfmpi_inq_varid(fh,'gamma',var_id);status = nfmpi_get_var_double_all(fh,var_id,td);gamma_prov=td(1)
  status = nfmpi_inq_varid(fh,'ntx',  var_id);status = nfmpi_get_var_int_all(fh,var_id,ti);ntx_prov=ti(1)
  status = nfmpi_inq_varid(fh,'nty',  var_id);status = nfmpi_get_var_int_all(fh,var_id,ti);nty_prov=ti(1)
  status = nfmpi_inq_varid(fh,'nvar', var_id);status = nfmpi_get_var_int_all(fh,var_id,ti);nvar_prov=ti(1)
  status = nfmpi_inq_varid(fh,'nstep',var_id);status = nfmpi_get_var_int_all(fh,var_id,ti);nstep=ti(1)
  if(gamma_prov/=gamma .or. ntx_prov/=ntx .or. nty_prov/=nty .or. nvar_prov/=nvar)then
    print *,'ERROR: Invalid parameters in restart_file:'
    print *,rang,'In nml:          gamma=',gamma,' nx=',ntx,' ny=',nty,'nvar=',nvar
    print *,rang,'In restart_file: gamma=',gamma_prov,' nx=',ntx_prov,' ny=',nty_prov,'nvar=',nvar_prov
    call MPI_Abort(comm2d,1,code)
  endif

  starts(1) = startx+1 ; starts(2) = starty+1 ; starts(3) = 1
  counts(1) = nx       ; counts(2) = ny       ; counts(3) = nvar
  status = nfmpi_inq_varid(fh,'u',var_id)
  status = nfmpi_get_vara_double_all(fh,var_id,starts,counts,uold(3:nx+2,3:ny+2,1:nvar));call CHKERR(status)

  status = nfmpi_close(fh);call CHKERR(status)

  call timer_stop("Parallel netCDF",.false.,int(ntx,kind=8)*nty*nvar*8)

end subroutine input_pnetcdf


subroutine CHKERR(status)
  use hydro_mpi
  implicit none

  integer::status

  if ((status)/=NF_NOERR) then
    write(*,*) nfmpi_strerror(status)
    call MPI_Abort(comm2d,1,code)
  end if
end subroutine CHKERR

#endif
end module hydro_pnetcdf
