/*
*  C Implementation: mpi_funcs
*
* Description: MPI functions used by HYDRO MPI2D
*
*
* Author: Philippe Wautelet <philippe.wautelet@idris.fr>, (C) 2010
*
*/

#include "main.h"

void
init_mpi(int *argc, char ***argv, hydroparam_t * H)
{
  const int ndims = 2;
  const int reorder = 1;
  int periods[2] = { 0, 0 };

  int ierr;
  int coords[2];
  int dims[2];

  int          size_double;
  MPI_Aint     pas;
  MPI_Datatype type_temp;


  MPI_Comm *comm2d = &(mpi_vars.comm2d);
  int *neighb = mpi_vars.neighb;

  ierr = MPI_Init(argc,argv);

  MPI_Comm_size(MPI_COMM_WORLD,&(mpi_vars.nb_procs));

  // Create 2D cartesian topology
  dims[0]=0;dims[1]=0;
  MPI_Dims_create(mpi_vars.nb_procs,ndims,dims);
  if(H->boundary_left==3 && H->boundary_right==3) periods[0]=1;
  if(H->boundary_up  ==3 && H->boundary_down==3)  periods[1]=1;
  MPI_Cart_create(MPI_COMM_WORLD,ndims,dims,periods,reorder,comm2d);
  MPI_Comm_rank(*comm2d,&(mpi_vars.rank));
  MPI_Cart_coords(*comm2d,mpi_vars.rank,ndims,coords);

  if (mpi_vars.rank==0) {
    printf("\n");
    printf(" MPI execution with  %d processes (%d x %d)\n",mpi_vars.nb_procs,dims[0],dims[1]);
    printf("Starting time integration, nx = %d ny = %d\n",H->nx,H->ny);
    printf("\n");
  }

  // Compute subdomain sizes
  H->ntx=H->nx;
  H->nty=H->ny;

  H->startx=coords[0]*(H->ntx/dims[0]) + MIN(coords[0],H->ntx%dims[0]);
  H->starty=coords[1]*(H->ntx/dims[1]) + MIN(coords[1],H->ntx%dims[1]);

  if (coords[0]<H->nx%dims[0]) {
     H->nx=H->nx/dims[0]+1;
  } else {
     H->nx=H->nx/dims[0];
  }
  if (coords[1]<H->ny%dims[1]) {
     H->ny=H->ny/dims[1]+1;
  } else {
     H->ny=H->ny/dims[1];
  }

  // Modify boundary conditions and determine neighbors list
  // 0 stands for MPI boundary conditions
  neighb[0]=MPI_PROC_NULL;neighb[1]=MPI_PROC_NULL;
  neighb[2]=MPI_PROC_NULL;neighb[3]=MPI_PROC_NULL;
  MPI_Cart_shift(*comm2d,0,1,&(neighb[WEST]), &(neighb[EAST]));
  MPI_Cart_shift(*comm2d,1,1,&(neighb[SOUTH]),&(neighb[NORTH]));
  if(neighb[NORTH]!=MPI_PROC_NULL) H->boundary_up   =0;
  if(neighb[SOUTH]!=MPI_PROC_NULL) H->boundary_down =0;
  if(neighb[WEST] !=MPI_PROC_NULL) H->boundary_left =0;
  if(neighb[EAST] !=MPI_PROC_NULL) H->boundary_right=0;

  // Create derived datatypes
  MPI_Type_size(MPI_DOUBLE,&size_double);
  pas = (H->nx+ExtraLayerTot)*(H->ny+ExtraLayerTot)*size_double;

  MPI_Type_vector((int) H->ny,2,(int) (H->nx+ExtraLayerTot),MPI_DOUBLE,&type_temp);
  MPI_Type_create_hvector((int) H->nvar,1,pas,type_temp,&(mpi_vars.bloc_dim1));
  MPI_Type_commit(&(mpi_vars.bloc_dim1));

  MPI_Type_vector(2,(int) H->nx,(int) (H->nx+ExtraLayerTot),MPI_DOUBLE,&type_temp);
  MPI_Type_create_hvector((int) H->nvar,1,pas,type_temp,&(mpi_vars.bloc_dim2));
  MPI_Type_commit(&(mpi_vars.bloc_dim2));
}

void
clean_mpi()
{
  int ierr;

  MPI_Type_free(&(mpi_vars.bloc_dim1));
  MPI_Type_free(&(mpi_vars.bloc_dim2));
  ierr = MPI_Finalize();
}
