#ifdef PNETCDF

#include "main.h"
#include <pnetcdf.h>

static void chkerr(int);

void
output_pnetcdf(int noutput, const hydroparam_t H, hydrovar_t * Hv)
{
  const int ndim = 3;
  MPI_Datatype type_in;
  char filename[256];
  int dims[ndim], fh, mode, var;
  int ntx_dim_id, nty_dim_id, nvar_dim_id;
  int gamma_id, nstep_id, ntx_id, nty_id, nvar_id, t_id, u_id;
  int pos_noghost[ndim], uold_shape[ndim], uold_noghost_shape[ndim];
  MPI_Offset offset = 0;
  MPI_Offset counts[ndim], starts[ndim], strides[ndim];

  if (mpi_vars.rank==0) printf("Parallel-netCDF : Outputting array of size=%ld %ld %ld\n",H.ntx,H.nty,H.nvar);

  timer_start();

  //Determine filename
  snprintf(filename,sizeof(filename),"output_%05d.nc",noutput);

  //Open file
  mode = NC_64BIT_DATA;
  chkerr(ncmpi_create(mpi_vars.comm2d,filename,mode,MPI_INFO_NULL,&fh));

  //Define dimensions
  ncmpi_def_dim(fh,"ntx", H.ntx, &ntx_dim_id);
  ncmpi_def_dim(fh,"nty", H.nty, &nty_dim_id);
  ncmpi_def_dim(fh,"nvar",H.nvar,&nvar_dim_id);

  //Define variables
  dims[0]=0;
  ncmpi_def_var(fh,"t",    NC_DOUBLE,0,dims,&t_id);
  ncmpi_def_var(fh,"gamma",NC_DOUBLE,0,dims,&gamma_id);
  ncmpi_def_var(fh,"ntx",  NC_INT,   0,dims,&ntx_id);
  ncmpi_def_var(fh,"nty",  NC_INT,   0,dims,&nty_id);
  ncmpi_def_var(fh,"nvar", NC_INT,   0,dims,&nvar_id);
  ncmpi_def_var(fh,"nstep",NC_INT,   0,dims,&nstep_id);

  dims[0]=nvar_dim_id;
  dims[1]=nty_dim_id;
  dims[2]=ntx_dim_id;
  ncmpi_def_var(fh,"u",NC_DOUBLE,3,dims,&u_id);

  chkerr(ncmpi_enddef(fh));

  //Write run parameters
  chkerr(ncmpi_begin_indep_data(fh));
  if (mpi_vars.rank==0) {
     chkerr(ncmpi_put_var1_double(fh,t_id,    &offset,&(H.t)));
     chkerr(ncmpi_put_var1_double(fh,gamma_id,&offset,&(H.gamma)));
     var=H.ntx;  chkerr(ncmpi_put_var1_int(fh,ntx_id,  &offset,&var));
     var=H.nty;  chkerr(ncmpi_put_var1_int(fh,nty_id,  &offset,&var));
     var=H.nvar; chkerr(ncmpi_put_var1_int(fh,nvar_id, &offset,&var));
     var=H.nstep;chkerr(ncmpi_put_var1_int(fh,nstep_id,&offset,&var));
  }

  chkerr(ncmpi_end_indep_data(fh));

  //Create subarray to represent data to write inside each process without the ghost cells
  uold_shape[0]=H.nvar;
  uold_shape[1]=H.nyt;
  uold_shape[2]=H.nxt;
  uold_noghost_shape[0]=H.nvar;
  uold_noghost_shape[1]=H.ny;
  uold_noghost_shape[2]=H.nx;
  pos_noghost[0]=0;
  pos_noghost[1]=2;
  pos_noghost[2]=2;
  MPI_Type_create_subarray(ndim,uold_shape,uold_noghost_shape,pos_noghost,
                           MPI_ORDER_C,MPI_DOUBLE,&type_in);
  MPI_Type_commit(&type_in);

  //Write data
  starts[0]  = 0      ; starts[1]  = H.starty ; starts[2]  = H.startx ;
  counts[0]  = H.nvar ; counts[1]  = H.ny     ; counts[2]  = H.nx ;
  strides[0] = 1      ; strides[1] = 1        ; strides[2] = 1;
  chkerr(ncmpi_put_vars_all(fh,u_id,starts,counts,strides,Hv->uold,1,type_in));

  //Free datatype
  MPI_Type_free(&type_in);

  //Close file
  chkerr(ncmpi_close(fh));

  timer_stop("Parallel-netCDF",WRITING,H.nvar*H.ntx*H.nty*sizeof(double));
}


void
input_pnetcdf(hydroparam_t *H, hydrovar_t * Hv)
{
  const int ndim = 3;
  MPI_Datatype type_in;
  int fh, mode;
  int var_id;
  int nstep, ntx, nty, nvar;
  double gam;
  int pos_noghost[3], uold_shape[3], uold_noghost_shape[3];
  MPI_Offset offset = 0;
  MPI_Offset counts[3], starts[3], strides[3];

  if (mpi_vars.rank==0) printf("Parallel-netCDF : Reading file %s\n",H->restart_file);

  timer_start();

  //Open file
  mode = NC_NOWRITE;
  chkerr(ncmpi_open(mpi_vars.comm2d,H->restart_file,mode,MPI_INFO_NULL,&fh));

  //Read run parameters
  chkerr(ncmpi_inq_varid(fh,"t",    &var_id));chkerr(ncmpi_get_var_double_all(fh,var_id,&(H->t)));
  chkerr(ncmpi_inq_varid(fh,"gamma",&var_id));chkerr(ncmpi_get_var_double_all(fh,var_id,&gam));
  chkerr(ncmpi_inq_varid(fh,"ntx",  &var_id));chkerr(ncmpi_get_var_int_all(fh,var_id,&ntx));
  chkerr(ncmpi_inq_varid(fh,"nty",  &var_id));chkerr(ncmpi_get_var_int_all(fh,var_id,&nty));
  chkerr(ncmpi_inq_varid(fh,"nvar", &var_id));chkerr(ncmpi_get_var_int_all(fh,var_id,&nvar));
  chkerr(ncmpi_inq_varid(fh,"nstep",&var_id));chkerr(ncmpi_get_var_int_all(fh,var_id,&nstep));H->nstep=nstep;
  if( gam!=H->gamma || ntx!=H->ntx || nty!=H->nty || nvar!=H->nvar ) {
    printf("ERROR: Invalid parameters in restart_file:\n");
    printf("%d In nml:          gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,H->gamma,H->ntx,H->nty,H->nvar);
    printf("%d In restart file: gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,gam,ntx,nty,nvar);
    MPI_Abort(mpi_vars.comm2d,1);
  }

  //Create subarray to represent data to read inside each process without the ghost cells
  uold_shape[0]=H->nvar;
  uold_shape[1]=H->nyt;
  uold_shape[2]=H->nxt;
  uold_noghost_shape[0]=H->nvar;
  uold_noghost_shape[1]=H->ny;
  uold_noghost_shape[2]=H->nx;
  pos_noghost[0]=0;
  pos_noghost[1]=2;
  pos_noghost[2]=2;
  MPI_Type_create_subarray(ndim,uold_shape,uold_noghost_shape,pos_noghost,
                           MPI_ORDER_C,MPI_DOUBLE,&type_in);
  MPI_Type_commit(&type_in);

  //Read data
  starts[0]  = 0       ; starts[1]  = H->starty ; starts[2]  = H->startx ;
  counts[0]  = H->nvar ; counts[1]  = H->ny     ; counts[2]  = H->nx ;
  strides[0] = 1       ; strides[1] = 1         ; strides[2] = 1;
  chkerr(ncmpi_inq_varid(fh,"u",&var_id));
  chkerr(ncmpi_get_vars_all(fh,var_id,starts,counts,strides,Hv->uold,1,type_in));

  //Free datatype
  MPI_Type_free(&type_in);

  //Close file
  chkerr(ncmpi_close(fh));

  timer_stop("Parallel-netCDF",READING,H->nvar*H->ntx*H->nty*sizeof(double));
}

static void
chkerr(int status)
{
  if (status!= NC_NOERR) {
    printf("%s\n",ncmpi_strerror(status));
    MPI_Abort(mpi_vars.comm2d,1);
  }
}

#endif
