#ifdef NETCDF4

#include "main.h"
#include <netcdf_par.h>

static void chkerr(int);

void
output_netcdf(int noutput, const hydroparam_t H, hydrovar_t * Hv)
{
  const int ndim = 3;
  MPI_Datatype type_in;
  char filename[256];
  int i,j,k;
  int dims[ndim], fh, var;
  int old_mode;
  int ntx_dim_id, nty_dim_id, nvar_dim_id;
  int gamma_id, nstep_id, ntx_id, nty_id, nvar_id, t_id, u_id;
  int pos_noghost[ndim], uold_shape[ndim], uold_noghost_shape[ndim];
  size_t offset = 0;
  size_t counts[ndim], starts[ndim];
  ptrdiff_t imap[ndim], strides[ndim];

  if (mpi_vars.rank==0) printf("NetCDF : Outputting array of size=%ld %ld %ld\n",H.ntx,H.nty,H.nvar);

  timer_start();

  //Determine filename
  snprintf(filename,sizeof(filename),"output_%05d.nc4",noutput);

  //Create file
  chkerr(nc_create_par(filename,NC_MPIIO|NC_NETCDF4,mpi_vars.comm2d,MPI_INFO_NULL,&fh));

  //Define dimensions
  nc_def_dim(fh,"ntx", H.ntx, &ntx_dim_id);
  nc_def_dim(fh,"nty", H.nty, &nty_dim_id);
  nc_def_dim(fh,"nvar",H.nvar,&nvar_dim_id);

  //Define variables
  dims[0]=0;
  nc_def_var(fh,"t",    NC_DOUBLE,0,dims,&t_id);
  nc_def_var(fh,"gamma",NC_DOUBLE,0,dims,&gamma_id);
  nc_def_var(fh,"ntx",  NC_INT,   0,dims,&ntx_id);
  nc_def_var(fh,"nty",  NC_INT,   0,dims,&nty_id);
  nc_def_var(fh,"nvar", NC_INT,   0,dims,&nvar_id);
  nc_def_var(fh,"nstep",NC_INT,   0,dims,&nstep_id);

  dims[0]=nvar_dim_id;
  dims[1]=nty_dim_id;
  dims[2]=ntx_dim_id;
  nc_def_var(fh,"u",NC_DOUBLE,3,dims,&u_id);

  //Deactivate prefilling of variables (performance impact)
  chkerr(nc_set_fill(fh,NC_NOFILL,&old_mode));

  //Definition of dimensions and variables finished
  chkerr(nc_enddef(fh));

  //Write run parameters
  //By default all accesses are independent
  if (mpi_vars.rank==0) {
     chkerr(nc_put_var1_double(fh,t_id,    &offset,&(H.t)));
     chkerr(nc_put_var1_double(fh,gamma_id,&offset,&(H.gamma)));
     var=H.ntx;  chkerr(nc_put_var1_int(fh,ntx_id,  &offset,&var));
     var=H.nty;  chkerr(nc_put_var1_int(fh,nty_id,  &offset,&var));
     var=H.nvar; chkerr(nc_put_var1_int(fh,nvar_id, &offset,&var));
     var=H.nstep;chkerr(nc_put_var1_int(fh,nstep_id,&offset,&var));
  }

  //Describe data structure in memory
  imap[0]=(H.nx+4)*(H.ny+4);
  imap[1]=H.nx+4;
  imap[2]=1;

  //Set access to collective for this variable
  nc_var_par_access(fh,u_id, NC_COLLECTIVE);

  //Write data
  starts[0]  = 0      ; starts[1]  = H.starty ; starts[2]  = H.startx ;
  counts[0]  = H.nvar ; counts[1]  = H.ny     ; counts[2]  = H.nx ;
  strides[0] = 1      ; strides[1] = 1        ; strides[2] = 1;
  nc_put_varm_double(fh,u_id,starts,counts,strides,imap,&Hv->uold[2*(H.nx+4)+2]);

  //Close file
  chkerr(nc_close(fh));

  timer_stop("NetCDF",WRITING,H.nvar*H.ntx*H.nty*sizeof(double));
}


void
input_netcdf(hydroparam_t *H, hydrovar_t * Hv)
{
  const int ndim = 3;
  int fh;
  int var_id;
  int nstep, ntx, nty, nvar;
  double gam;
  size_t counts[ndim], starts[ndim];
  ptrdiff_t imap[ndim], strides[ndim];

  if (mpi_vars.rank==0) printf("NetCDF : Reading file %s\n",H->restart_file);

  timer_start();

  //Open file
  chkerr(nc_open_par(H->restart_file,NC_MPIIO|NC_NETCDF4,mpi_vars.comm2d,MPI_INFO_NULL,&fh));

  //Read run parameters
  chkerr(nc_inq_varid(fh,"t",    &var_id));chkerr(nc_get_var_double(fh,var_id,&(H->t)));
  chkerr(nc_inq_varid(fh,"gamma",&var_id));chkerr(nc_get_var_double(fh,var_id,&gam));
  chkerr(nc_inq_varid(fh,"ntx",  &var_id));chkerr(nc_get_var_int(fh,var_id,&ntx));
  chkerr(nc_inq_varid(fh,"nty",  &var_id));chkerr(nc_get_var_int(fh,var_id,&nty));
  chkerr(nc_inq_varid(fh,"nvar", &var_id));chkerr(nc_get_var_int(fh,var_id,&nvar));
  chkerr(nc_inq_varid(fh,"nstep",&var_id));chkerr(nc_get_var_int(fh,var_id,&nstep));H->nstep=nstep;
  if( gam!=H->gamma || ntx!=H->ntx || nty!=H->nty || nvar!=H->nvar ) {
    printf("ERROR: Invalid parameters in restart_file:\n");
    printf("%d In nml:          gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,H->gamma,H->ntx,H->nty,H->nvar);
    printf("%d In restart file: gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,gam,ntx,nty,nvar);
    MPI_Abort(mpi_vars.comm2d,1);
  }

  //Describe data structure in memory
  imap[0]=(H->nx+4)*(H->ny+4);
  imap[1]=H->nx+4;
  imap[2]=1;

  //Get variable information
  chkerr(nc_inq_varid(fh,"u",&var_id));

  //Set access to collective for this variable
  nc_var_par_access(fh,var_id, NC_COLLECTIVE);

  //Read data
  starts[0]  = 0       ; starts[1]  = H->starty ; starts[2]  = H->startx ;
  counts[0]  = H->nvar ; counts[1]  = H->ny     ; counts[2]  = H->nx ;
  strides[0] = 1       ; strides[1] = 1         ; strides[2] = 1;
  nc_get_varm_double(fh,var_id,starts,counts,strides,imap,&Hv->uold[2*(H->nx+4)+2]);

  //Close file
  chkerr(nc_close(fh));

  timer_stop("NetCDF",READING,H->nvar*H->ntx*H->nty*sizeof(double));
}

static void
chkerr(int status)
{
  if (status!= NC_NOERR) {
    printf("%s\n",nc_strerror(status));
    MPI_Abort(mpi_vars.comm2d,1);
  }
}

#endif
