#ifdef MPIIO

#include "main.h"

void
output_mpiio(int noutput, hydroparam_t H, hydrovar_t * Hv)
{
  const int ndim = 3;
  char filename[256];
  MPI_File fh;
  MPI_Datatype type_in, type_out;
  MPI_Offset disp;
  int glob_pos[3], pos_noghost[3];
  int glob_uold_shape[3], uold_shape[3], uold_noghost_shape[3];

  if (mpi_vars.rank==0) printf("MPI-IO : Outputting array of size=%ld %ld %ld\n",H.ntx,H.nty,H.nvar);

  timer_start();

  //Determine filename
  snprintf(filename,sizeof(filename),"output_%05d.mp",noutput);

  //Open file
  if( MPI_File_open(mpi_vars.comm2d,filename,MPI_MODE_WRONLY+MPI_MODE_CREATE,MPI_INFO_NULL,&fh)
                   !=MPI_SUCCESS) {
    fprintf(stderr,"ERROR in creating file %s\n",filename);
    MPI_Abort(mpi_vars.comm2d,1);
  }

  //Write run parameters
  if (mpi_vars.rank==0) {
    MPI_File_write(fh,&(H.t)    ,1,MPI_DOUBLE, MPI_STATUS_IGNORE);
    MPI_File_write(fh,&(H.gamma),1,MPI_DOUBLE, MPI_STATUS_IGNORE);
    MPI_File_write(fh,&(H.ntx)  ,1,MPI_INTEGER,MPI_STATUS_IGNORE);
    MPI_File_write(fh,&(H.nty)  ,1,MPI_INTEGER,MPI_STATUS_IGNORE);
    MPI_File_write(fh,&(H.nvar) ,1,MPI_INTEGER,MPI_STATUS_IGNORE);
    MPI_File_write(fh,&(H.nstep),1,MPI_INTEGER,MPI_STATUS_IGNORE);
    MPI_File_get_position(fh,&disp);
  }

  //Broadcast offset
  MPI_Bcast(&disp,sizeof(disp),MPI_BYTE,0,mpi_vars.comm2d);

  //Create subarray to represent data to write inside each process without the ghost cells
  uold_shape[0]=H.nvar;
  uold_shape[1]=H.nyt;
  uold_shape[2]=H.nxt;
  uold_noghost_shape[0]=H.nvar;
  uold_noghost_shape[1]=H.ny;
  uold_noghost_shape[2]=H.nx;
  pos_noghost[0]=0;
  pos_noghost[1]=2;
  pos_noghost[2]=2;
  MPI_Type_create_subarray(ndim,uold_shape,uold_noghost_shape,pos_noghost,
                           MPI_ORDER_C,MPI_DOUBLE,&type_in);
  MPI_Type_commit(&type_in);

  //Create subarray to represent data to write inside the global array
  glob_uold_shape[0]=H.nvar;
  glob_uold_shape[1]=H.nty;
  glob_uold_shape[2]=H.ntx;
  glob_pos[0]=0;
  glob_pos[1]=H.starty;
  glob_pos[2]=H.startx;
  MPI_Type_create_subarray(ndim,glob_uold_shape,uold_noghost_shape,glob_pos,
                           MPI_ORDER_C,MPI_DOUBLE,&type_out);
  MPI_Type_commit(&type_out);

  //Set file view (each process sees only its own part of uold)
  MPI_File_set_view(fh,disp,MPI_DOUBLE,type_out,"native",MPI_INFO_NULL);

  //Write data
  MPI_File_write_all(fh,Hv->uold,1,type_in,MPI_STATUS_IGNORE);

  //Free datatypes
  MPI_Type_free(&type_out);
  MPI_Type_free(&type_in);

  //Close file
  MPI_File_close(&fh);

  timer_stop("MPI-IO",WRITING,H.nvar*H.ntx*H.nty*sizeof(double));
}


void
input_mpiio(hydroparam_t *H, hydrovar_t * Hv)
{
  const int ndim = 3;
  char filename[256];
  int ntx, nty, nvar;
  double gam;
  MPI_File fh;
  MPI_Datatype type_in, type_out;
  MPI_Offset disp;
  int glob_pos[3], pos_noghost[3];
  int glob_uold_shape[3], uold_shape[3], uold_noghost_shape[3];

  if (mpi_vars.rank==0) printf("MPI-IO : Reading file %s\n",H->restart_file);

  timer_start();

  //Open file
  if( MPI_File_open(mpi_vars.comm2d,H->restart_file,MPI_MODE_RDONLY,MPI_INFO_NULL,&fh)
                   !=MPI_SUCCESS) {
    fprintf(stderr,"ERROR in opening file %s\n",H->restart_file);
    MPI_Abort(mpi_vars.comm2d,1);
  }

  //Read run parameters
  MPI_File_read_all(fh,&(H->t)    ,1,MPI_DOUBLE, MPI_STATUS_IGNORE);
  MPI_File_read_all(fh,&gam       ,1,MPI_DOUBLE, MPI_STATUS_IGNORE);
  MPI_File_read_all(fh,&ntx       ,1,MPI_INTEGER,MPI_STATUS_IGNORE);
  MPI_File_read_all(fh,&nty       ,1,MPI_INTEGER,MPI_STATUS_IGNORE);
  MPI_File_read_all(fh,&nvar      ,1,MPI_INTEGER,MPI_STATUS_IGNORE);
  MPI_File_read_all(fh,&(H->nstep),1,MPI_INTEGER,MPI_STATUS_IGNORE);

  MPI_File_get_position(fh,&disp);

  if( gam!=H->gamma || ntx!=H->ntx || nty!=H->nty || nvar!=H->nvar ) {
    printf("ERROR: Invalid parameters in restart_file:\n");
    printf("%d In nml:          gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,H->gamma,H->ntx,H->nty,H->nvar);
    printf("%d In restart file: gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,gam,ntx,nty,nvar);
    MPI_Abort(mpi_vars.comm2d,1);
  }

  //Create subarray to represent data to read inside each process without the ghost cells
  uold_shape[0]=H->nvar;
  uold_shape[1]=H->nyt;
  uold_shape[2]=H->nxt;
  uold_noghost_shape[0]=H->nvar;
  uold_noghost_shape[1]=H->ny;
  uold_noghost_shape[2]=H->nx;
  pos_noghost[0]=0;
  pos_noghost[1]=2;
  pos_noghost[2]=2;
  MPI_Type_create_subarray(ndim,uold_shape,uold_noghost_shape,pos_noghost,
                           MPI_ORDER_C,MPI_DOUBLE,&type_in);
  MPI_Type_commit(&type_in);

  //Create subarray to represent data to read inside the global array
  glob_uold_shape[0]=H->nvar;
  glob_uold_shape[1]=H->nty;
  glob_uold_shape[2]=H->ntx;
  glob_pos[0]=0;
  glob_pos[1]=H->starty;
  glob_pos[2]=H->startx;
  MPI_Type_create_subarray(ndim,glob_uold_shape,uold_noghost_shape,glob_pos,
                           MPI_ORDER_C,MPI_DOUBLE,&type_out);
  MPI_Type_commit(&type_out);

  //Set file view (each process sees only its own part of uold)
  MPI_File_set_view(fh,disp,MPI_DOUBLE,type_out,"native",MPI_INFO_NULL);

  //Write data
  MPI_File_read_all(fh,Hv->uold,1,type_in,MPI_STATUS_IGNORE);

  //Free datatypes
  MPI_Type_free(&type_out);
  MPI_Type_free(&type_in);

  //Close file
  MPI_File_close(&fh);

  timer_stop("MPI-IO",READING,H->nvar*H->ntx*H->nty*sizeof(double));
}

#endif
