#ifdef HDF5

#include "main.h"
#include <hdf5.h>
#include <hdf5_hl.h>

static int is_initialized = 0;

void
output_hdf5(int noutput, const hydroparam_t H, hydrovar_t * Hv)
{
  char filename[256];
  hid_t fh, mode_id, prp_id;
  hid_t da_id, ds_id, fs_id;
  hsize_t counts[3], dims[3], starts[3];
  int var;

  check_hdf5_initialization();

  if (mpi_vars.rank==0) printf("HDF5 : Outputting array of size=%ld %ld %ld\n",H.ntx,H.nty,H.nvar);

  timer_start();

  //Determine filename
  snprintf(filename,sizeof(filename),"output_%05d.h5",noutput);

  //Create property list for parallel access
  prp_id = H5Pcreate(H5P_FILE_ACCESS);
  H5Pset_fapl_mpio(prp_id,mpi_vars.comm2d,MPI_INFO_NULL);

  //Open file
  fh = H5Fcreate(filename,H5F_ACC_TRUNC,H5P_DEFAULT,prp_id);
  H5Pclose(prp_id);

  //Create property list for access mode (used when switching between
  //independent and collective MPI-I/O modes)
  mode_id = H5Pcreate(H5P_DATASET_XFER);
  //mode_indep_id = H5Pcreate(H5P_DATASET_XFER);

  //Put mode_id in collective mode
  H5Pset_dxpl_mpio(mode_id,H5FD_MPIO_COLLECTIVE);
  //Put mode_indep_id in independent mode
  //H5Pset_dxpl_mpio(mode_indep_id,H5FD_MPIO_INDEPENDENT);

  //Write run parameters
  H5LTset_attribute_double(fh,"/","t",    &(H.t),1);
  H5LTset_attribute_double(fh,"/","gamma",&(H.gamma),1);
  var=H.ntx;  H5LTset_attribute_int(fh,"/","ntx",  &var,1);
  var=H.nty;  H5LTset_attribute_int(fh,"/","nty",  &var,1);
  var=H.nvar; H5LTset_attribute_int(fh,"/","nvar", &var,1);
  var=H.nstep;H5LTset_attribute_int(fh,"/","nstep",&var,1);

  //Description of data in memory
  counts[0] = H.nvar ; counts[1] = H.ny   ; counts[2] = H.nx ;
  dims[0]   = H.nvar ; dims[1]   = H.ny+4 ; dims[2]   = H.nx+4 ;
  starts[0] = 0      ; starts[1] = 2      ; starts[2]  = 2;
  ds_id = H5Screate_simple(3,dims,NULL);
  H5Sselect_hyperslab(ds_id,H5S_SELECT_SET,starts,NULL,counts,NULL);

  //Description of data in file
  counts[0] = H.nvar ; counts[1] = H.ny     ; counts[2] = H.nx ;
  dims[0]   = H.nvar ; dims[1]   = H.nty    ; dims[2]   = H.ntx ;
  starts[0] = 0      ; starts[1] = H.starty ; starts[2]  = H.startx ;
  fs_id = H5Screate_simple(3,dims,NULL);
  da_id = H5Dcreate(fh,"u",H5T_NATIVE_DOUBLE,fs_id,H5P_DEFAULT,H5P_DEFAULT,H5P_DEFAULT);
  H5Sselect_hyperslab(fs_id,H5S_SELECT_SET,starts,NULL,counts,NULL);

  //Write data
  H5Dwrite(da_id,H5T_NATIVE_DOUBLE,ds_id,fs_id,mode_id,Hv->uold);

  //Free resources
  H5Dclose(da_id);
  H5Sclose(ds_id);
  H5Sclose(fs_id);
  H5Pclose(mode_id);
  //H5Pclose(mode_indep_id);

  //Close file
  H5Fclose(fh);

  timer_stop("HDF5",WRITING,H.nvar*H.ntx*H.nty*sizeof(double));
}


void
input_hdf5(hydroparam_t *H, hydrovar_t * Hv)
{
  hid_t fh, mode_id, prp_id;
  hid_t da_id, ds_id, fs_id;
  hsize_t counts[3], dims[3], starts[3];
  int nstep, ntx, nty, nvar;
  double gam;

  if (mpi_vars.rank==0) printf("HDF5 : Reading file %s\n",H->restart_file);

  check_hdf5_initialization();

  timer_start();

  //Create property list for parallel access
  prp_id = H5Pcreate(H5P_FILE_ACCESS);
  H5Pset_fapl_mpio(prp_id,mpi_vars.comm2d,MPI_INFO_NULL);

  //Open file
  fh = H5Fopen(H->restart_file,H5F_ACC_RDONLY,prp_id);
  H5Pclose(prp_id);

  //Create property list for access mode (used when switching between
  //independent and collective MPI-I/O modes)
  mode_id = H5Pcreate(H5P_DATASET_XFER);
  //mode_indep_id = H5Pcreate(H5P_DATASET_XFER);

  //Put mode_id in collective mode
  H5Pset_dxpl_mpio(mode_id,H5FD_MPIO_COLLECTIVE);
  //Put mode_indep_id in independent mode
  //H5Pset_dxpl_mpio(mode_indep_id,H5FD_MPIO_INDEPENDENT);

  //Read run parameters
  H5LTget_attribute_double(fh,"/","t",    &(H->t));
  H5LTget_attribute_double(fh,"/","gamma",&gam);
  H5LTget_attribute_int(fh,"/","ntx",  &ntx);
  H5LTget_attribute_int(fh,"/","nty",  &nty);
  H5LTget_attribute_int(fh,"/","nvar", &nvar);
  H5LTget_attribute_int(fh,"/","nstep",&nstep);H->nstep=nstep;
  if( gam!=H->gamma || ntx!=H->ntx || nty!=H->nty || nvar!=H->nvar ) {
    printf("ERROR: Invalid parameters in restart_file:\n");
    printf("%d In nml:          gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,H->gamma,H->ntx,H->nty,H->nvar);
    printf("%d In restart file: gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,gam,ntx,nty,nvar);
    MPI_Abort(mpi_vars.comm2d,1);
  }

  //Description of data in memory
  counts[0] = H->nvar ; counts[1] = H->ny   ; counts[2] = H->nx ;
  dims[0]   = H->nvar ; dims[1]   = H->ny+4 ; dims[2]   = H->nx+4 ;
  starts[0] = 0       ; starts[1] = 2       ; starts[2]  = 2;
  ds_id = H5Screate_simple(3,dims,NULL);
  H5Sselect_hyperslab(ds_id,H5S_SELECT_SET,starts,NULL,counts,NULL);

  //Description of data in file
  counts[0] = H->nvar ; counts[1] = H->ny     ; counts[2] = H->nx ;
  starts[0] = 0       ; starts[1] = H->starty ; starts[2]  = H->startx ;
  da_id = H5Dopen(fh,"u",H5P_DEFAULT);
  fs_id = H5Dget_space(da_id);
  H5Sselect_hyperslab(fs_id,H5S_SELECT_SET,starts,NULL,counts,NULL);

  //Read data
  H5Dread(da_id,H5T_NATIVE_DOUBLE,ds_id,fs_id,mode_id,Hv->uold);

  //Free resources
  H5Dclose(da_id);
  H5Sclose(ds_id);
  H5Sclose(fs_id);
  H5Pclose(mode_id);
  //H5Pclose(mode_indep_id);

  //Close file
  H5Fclose(fh);

  timer_stop("HDF5",READING,H->nvar*H->ntx*H->nty*sizeof(double));
}


void
check_hdf5_initialization()
{
  if(is_initialized!=0) return;

  H5open();
  is_initialized = 1;
}


void
clean_hdf5()
{
   if(is_initialized==0) return;

   H5close();
   is_initialized = 0;
}

#endif
