#ifdef ADIOS

#include "main.h"
#include <adios.h>
#include <adios_read.h>

static int is_write_initialized = 0;

void
output_adios(int noutput, hydroparam_t H, hydrovar_t * Hv)
{
  char filename[256];
  int i, j, k;
  int64_t fh;
  uint64_t group_size, total_size;
  double *work;

  check_adios_write_initialization();

  if (mpi_vars.rank==0) printf("ADIOS : Outputting array of size=%ld %ld %ld\n",H.ntx,H.nty,H.nvar);

  timer_start();

  //Determine filename
  snprintf(filename,sizeof(filename),"output_%05d.ad",noutput);

  //Open file
  adios_open(&fh,"output",filename,"w",mpi_vars.comm2d);

  //Provide group_size to ADIOS
  group_size = (2+H.nx*H.ny*H.nvar)*sizeof(double)+8*sizeof(int);
  adios_group_size(fh,group_size,&total_size);

  //Write run parameters
  adios_write(fh,"t",    &(H.t));
  adios_write(fh,"gamma",&(H.gamma));
  adios_write(fh,"ntx",  &(H.ntx));
  adios_write(fh,"nty",  &(H.nty));
  adios_write(fh,"nvar", &(H.nvar));
  adios_write(fh,"nstep",&(H.nstep));

  //Copy data into a contiguous region
  work = malloc(sizeof(double)*H.nx*H.ny*H.nvar);
  for (k=0;k<H.nvar;k++)
    for (j=0;j<H.ny;j++)
      for (i=0;i<H.nx;i++)
        work[(k*H.ny+j)*H.nx+i]=Hv->uold[(k*(H.ny+4)+j+2)*(H.nx+4)+i+2];

  //Write data structure (necessary to provide to ADIOS all necessary information to write u)
  adios_write(fh,"nx",&(H.nx));
  adios_write(fh,"ny",&(H.ny));
  adios_write(fh,"startx",&(H.startx));
  adios_write(fh,"starty",&(H.starty));

  //Write data
  adios_write(fh,"u",work);

  //Free work array
  free(work);

  //Close file
  adios_close(fh);

  timer_stop("ADIOS",WRITING,H.nvar*H.ntx*H.nty*sizeof(double));
}


void
input_adios(hydroparam_t *H, hydrovar_t * Hv)
{
  //This implementation works only if restarting with the same number of processes
  int i, j, k;
  int64_t fh;
  uint64_t sz_int, sz_double;
  double *work;
  int nstep, ntx, nty, nvar;
  double gam;

  check_adios_write_initialization();

  if (mpi_vars.rank==0) printf("ADIOS : Reading file %s\n",H->restart_file);

  timer_start();

  sz_int = sizeof(int);
  sz_double = sizeof(double);

  //Open file
  adios_open(&fh,"output",H->restart_file,"r",mpi_vars.comm2d);

  //Read run parameters
  adios_read(fh,"t",    &(H->t),sz_double);
  adios_read(fh,"gamma",&gam,sz_double);
  adios_read(fh,"ntx",  &ntx,sz_int);
  adios_read(fh,"nty",  &nty,sz_int);
  adios_read(fh,"nvar", &nvar,sz_int);
  adios_read(fh,"nstep",&(H->nstep),sz_int);

  //Read data (get data into a contiguous region)
  work = malloc(sizeof(double)*H->nx*H->ny*H->nvar);
  adios_read(fh,"u",work,sizeof(work));

  //Close file
  //Warning: data is only available after this
  adios_close(fh);

  if( gam!=H->gamma || ntx!=H->ntx || nty!=H->nty || nvar!=H->nvar ) {
    printf("ERROR: Invalid parameters in restart_file:\n");
    printf("%d In nml:          gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,H->gamma,H->ntx,H->nty,H->nvar);
    printf("%d In restart file: gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,gam,ntx,nty,nvar);
    MPI_Abort(mpi_vars.comm2d,1);
  }

  //Copy data to u
  for (k=0;k<H->nvar;k++)
    for (j=0;j<H->ny;j++)
      for (i=0;i<H->nx;i++)
        Hv->uold[(k*(H->ny+4)+j+2)*(H->nx+4)+i+2]=work[(k*H->ny+j)*H->nx+i];

  //Free work array
  free(work);

  timer_stop("ADIOS",READING,H->nvar*H->ntx*H->nty*sizeof(double));

  #if 0
  //NOT YET IMPLEMENTED
  //To restart with any number of processes
  //Warning: works only with BP fileformat
  const int method = ADIOS_READ_METHOD_BP;
  ADIOS_FILE *fh;

  if (mpi_vars.rank==0) printf("ADIOS: Reading file %s\n",H->restart_file);

  adios_read_init_method(method,mpi_vars.comm2d,"verbose=3");

  fh = adios_read_open_file(H->restart_file,method,mpi_vars.comm2d);

  adios_schedule_read(fh,NULL,"t",0,1,&(H->t));

  printf("t=%f\n",H->t);

  adios_read_close(fh);

  adios_read_finalize_method (method);

  MPI_Barrier(mpi_vars.comm2d);
  MPI_Abort(mpi_vars.comm2d,1);
#endif
}


void
check_adios_write_initialization()
{
  if(is_write_initialized!=0) return;

  adios_init("config_adios.xml",mpi_vars.comm2d);
  is_write_initialized = 1;
}


void
clean_adios_write()
{
   if(is_write_initialized==0) return;

   adios_finalize(mpi_vars.rank);
   is_write_initialized = 0;
}

#endif
