#include "main.h"

static double tb, te;

void
output(int noutput, const hydroparam_t H, hydrovar_t * Hv)
{
  switch (H.fileformat_out) {
  case FILE_STD:
    output_std(noutput, H, Hv); break;
#ifdef ADIOS
  case FILE_ADIOS:
    output_adios(noutput, H, Hv); break;
#endif
#ifdef HDF5
    case FILE_HDF5:
    output_hdf5(noutput, H, Hv); break;
#endif
#ifdef MPIIO
    case FILE_MPIIO:
    output_mpiio(noutput, H, Hv); break;
#endif
#ifdef NETCDF4
    case FILE_NETCDF:
    output_netcdf(noutput, H, Hv); break;
#endif
#ifdef PNETCDF
  case FILE_PNETCDF:
    output_pnetcdf(noutput, H, Hv); break;
#endif
#ifdef SIONLIB
  case FILE_SIONLIB:
    output_sionlib(noutput, H, Hv); break;
#endif
  case FILE_VTK:
    output_vtk(noutput, H, Hv); break;
  default:
    fprintf(stderr,"ERROR: Unknown fileformat_out\n");
    MPI_Abort(MPI_COMM_WORLD,1);
  }
}


void
input(int *noutput,hydroparam_t *H, hydrovar_t * Hv)
{
  switch (H->fileformat_in) {
  case FILE_STD:
    input_std(H, Hv); break;
#ifdef ADIOS
  case FILE_ADIOS:
    input_adios(H, Hv); break;
#endif
#ifdef HDF5
  case FILE_HDF5:
    input_hdf5(H, Hv); break;
#endif
#ifdef MPIIO
    case FILE_MPIIO:
    input_mpiio(H, Hv); break;
#endif
#ifdef NETCDF
    case FILE_NETCDF:
    input_netcdf(H, Hv); break;
#endif
#ifdef PNETCDF
    case FILE_PNETCDF:
    input_pnetcdf(H, Hv); break;
#endif
#ifdef SIONLIB
    case FILE_SIONLIB:
    input_sionlib(H, Hv); break;
#endif
  case FILE_VTK:
    input_vtk(H, Hv); break;
    default:
      fprintf(stderr,"ERROR: Unknown fileformat_in\n");
      MPI_Abort(MPI_COMM_WORLD,1);
  }
  H->nstepstart = H->nstep;
  *noutput = H->nstepstart/H->noutput;
}


void
output_std(int noutput, const hydroparam_t H, hydrovar_t * Hv)
{
  int i,j,k;
  char filename[256];
  FILE *fd;
  double *work;

  if (mpi_vars.rank==0) printf("Std : Outputting array of size=%ld %ld %ld\n",H.ntx,H.nty,H.nvar);

  timer_start();

  //Determine filename
  snprintf(filename,sizeof(filename),"output_%05d.%05d",noutput,mpi_vars.rank);

  //Open file
  fd = fopen(filename,"w");

  //Write run parameters
  fwrite(&(H.t),    sizeof(double),1,fd);
  fwrite(&(H.gamma),sizeof(double),1,fd);
  fwrite(&(H.nx),  sizeof(int),1,fd);
  fwrite(&(H.ny),  sizeof(int),1,fd);
  fwrite(&(H.nvar), sizeof(int),1,fd);
  fwrite(&(H.nstep),sizeof(int),1,fd);

  //Copy data into a contiguous region
  work = malloc(sizeof(double)*H.nx*H.ny*H.nvar);
  for (k=0;k<H.nvar;k++)
    for (j=0;j<H.ny;j++)
      for (i=0;i<H.nx;i++)
        work[(k*H.ny+j)*H.nx+i]=Hv->uold[(k*(H.ny+4)+j+2)*(H.nx+4)+i+2];

  //Write data
  fwrite(work,sizeof(double)*H.nvar*H.nx*H.ny,1,fd);

  //Free work array
  free(work);

  //Close file
  fclose(fd);

  timer_stop("Std",WRITING,H.nvar*H.ntx*H.nty*sizeof(double));
}


void
input_std(hydroparam_t *H, hydrovar_t * Hv)
{
  int i,j,k;
  char filename[256];
  int nx, ny, nvar;
  double gam;
  FILE *fd;
  double *work;

  if (mpi_vars.rank==0) printf("Std : Reading file %s\n",H->restart_file);

  timer_start();

  //Determine filename
  snprintf(filename,sizeof(filename),"%s.%05d",H->restart_file,mpi_vars.rank);

  //Open file
  fd = fopen(filename,"r");

  //Read run parameters
  fread(&(H->t),    sizeof(double),1,fd);
  fread(&gam,      sizeof(double),1,fd);
  fread(&nx,      sizeof(int),1,fd);
  fread(&ny,      sizeof(int),1,fd);
  fread(&nvar,     sizeof(int),1,fd);
  fread(&(H->nstep),sizeof(int),1,fd);
  if( gam!=H->gamma || nx!=H->nx || ny!=H->ny || nvar!=H->nvar ) {
    printf("ERROR: Invalid parameters in restart_file:\n");
    printf("%d In nml:          gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,H->gamma,H->nx,H->ny,H->nvar);
    printf("%d In restart file: gamma=%lf nx=%d ny=%d nvar=%d\n",mpi_vars.rank,gam,nx,ny,nvar);
    MPI_Abort(mpi_vars.comm2d,1);
  }

  //Read data (get data into a contiguous region)
  work = malloc(sizeof(double)*H->nx*H->ny*H->nvar);
  fread(work,sizeof(double)*H->nvar*H->nx*H->ny,1,fd);

  //Copy data to u
  for (k=0;k<H->nvar;k++)
    for (j=0;j<H->ny;j++)
      for (i=0;i<H->nx;i++)
        Hv->uold[(k*(H->ny+4)+j+2)*(H->nx+4)+i+2]=work[(k*H->ny+j)*H->nx+i];

  //Free work array
  free(work);

  //Close file
  fclose(fd);

  timer_stop("Std",READING,H->nvar*H->ntx*H->nty*sizeof(double));
}

void print_io_versions()
{
  unsigned majnum,minnum,relnum;
  int mpiversion,mpisubversion;
  int main_version,sub_version,patch_level,fileformat_version;

  if (mpi_vars.rank==0) {
#ifdef ADIOS
     printf("  Compiled     with ADIOS support\n");
#else
     printf("  Not compiled with ADIOS support\n");
#endif
#ifdef HDF5
     H5get_libversion(&majnum,&minnum,&relnum);
     printf("  Compiled     with parallel HDF5 support (version %d.%d.%d)\n",majnum,minnum,relnum);
#else
     printf("  Not compiled with parallel HDF5 support\n");
#endif
     MPI_Get_version(&mpiversion,&mpisubversion);
#ifdef MPIIO
     printf("  Compiled     with MPI-I/O support (MPI-%d.%d)\n",mpiversion,mpisubversion);
#else
     printf("  Not compiled with MPI-I/O support (MPI-%d.%d)\n",mpiversion,mpisubversion);
#endif
#ifdef NETCDF4
     //Crash on Poincare printf("  Compiled     with NetCDF-4 support (%s)\n",nc_inq_libvers());
     printf("  Compiled     with NetCDF-4 support\n");
#else
     printf("  Not compiled with NetCDF-4 support\n");
#endif
#ifdef PNETCDF
     printf("  Compiled     with Parallel-NetCDF support (%s)\n",ncmpi_inq_libvers());
#else
     printf("  Not compiled with Parallel-NetCDF support\n");
#endif
#ifdef SIONLIB
     sion_get_version(&main_version,&sub_version,&patch_level,&fileformat_version);
     printf("  Compiled     with SIONlib support (version %d.%dp%d, fileformat=%d)\n",
            main_version,sub_version,patch_level,fileformat_version);
#else
     printf("  Not compiled with SIONlib support\n");
#endif
     printf("\n");
  }
}


void timer_start()
{
  int ierr;

  //Write buffers to disk before writing files
  sync();

  MPI_Barrier(MPI_COMM_WORLD);
  tb = MPI_Wtime();
}


void timer_stop(char *filetype, int mode,unsigned long size)
{
  int ierr;
  double MiB=1024.*1024, GiB=1024.*1024*1024, TiB=1024.*1024*1024*1024;
  double timeloc, tmin, tmax, throughput;
  double timewithsync, tsync, tminsync, tmaxsync, throughputwithsync;
  char sizetext[64],throughputtext[64],throughputwithsynctext[64];

  te = MPI_Wtime();
  timeloc = te-tb;

  //Write buffers to disk (warning: all files are concerned)
  sync();
  tsync = MPI_Wtime();
  timewithsync = tsync-tb;

  MPI_Reduce(&timeloc,&tmin,1,MPI_DOUBLE,MPI_MIN,0,MPI_COMM_WORLD);
  MPI_Reduce(&timeloc,&tmax,1,MPI_DOUBLE,MPI_MAX,0,MPI_COMM_WORLD);

  MPI_Reduce(&timewithsync,&tminsync,1,MPI_DOUBLE,MPI_MIN,0,MPI_COMM_WORLD);
  MPI_Reduce(&timewithsync,&tmaxsync,1,MPI_DOUBLE,MPI_MAX,0,MPI_COMM_WORLD);

  if (size<GiB) {
    snprintf(sizetext,sizeof(sizetext),"%8.3fMiB",size/MiB);
  } else if (size<TiB) {
    snprintf(sizetext,sizeof(sizetext),"%8.3fGiB",size/GiB);
  } else {
    snprintf(sizetext,sizeof(sizetext),"%8.3fTiB",size/TiB);
  }

  throughput = size/tmax;
  if (throughput<GiB) {
    snprintf(throughputtext,sizeof(throughputtext),"%8.3fMiB/s",throughput/MiB);
  } else if (throughput<TiB) {
    snprintf(throughputtext,sizeof(throughputtext),"%8.3fGiB/s",throughput/GiB);
  } else {
    snprintf(throughputtext,sizeof(throughputtext),"%8.3fTiB/s",throughput/TiB);
  }

  throughputwithsync = size/tmaxsync;
  if (throughputwithsync<GiB) {
    snprintf(throughputwithsynctext,sizeof(throughputwithsynctext),"%8.3fMiB/s",throughputwithsync/MiB);
  } else if (throughputwithsync<TiB) {
    snprintf(throughputwithsynctext,sizeof(throughputwithsynctext),"%8.3fGiB/s",throughputwithsync/GiB);
  } else {
    snprintf(throughputwithsynctext,sizeof(throughputwithsynctext),"%8.3fTiB/s",throughputwithsync/TiB);
  }

  if(mpi_vars.rank==0) {
    if(mode==WRITING) {
      printf("%s : Time to write             max=%fs min=%fs\n",filetype,tmax,tmin);
      printf("%s : Time to write (with sync) max=%fs min=%fs\n",filetype,tmaxsync,tminsync);
      printf("%s : Write throughput:             %s (data written: %s)\n",filetype,throughputtext,sizetext);
      printf("%s : Write throughput (with sync): %s (data written: %s)\n",filetype,throughputwithsynctext,sizetext);
    } else if (mode==READING){
      printf("%s : Time to read max=%fs min=%fs\n",filetype,tmax,tmin);
      printf("%s : Read throughput: %s (data read: %s)\n",filetype,throughputtext,sizetext);
    } else {
     printf(" ERROR: Invalid mode in time_stop\n");
    }
  }
}
