# Python code solving Jacobi solver for the Laplacian equation in 2D
# adapted by P. Elyakime for the "Atelier Visualisation in Situ", IMFT 11-13 december 2017
# (IMFT)

import time

import sys, math
import numpy as sp

# this code is largely inspired from Brad Whitlock's updateplots.py

# this code is similar in nature and algorithm to the F90 code contributed earlier
# see tools/DataManualExamples/Simulations/contrib
# this is a very first attempt to use some numpy arrays.
# A grid of size (M+2)x(M+2) is allocated.
# the working buffer is of size MxM
#

# no need for that if you set your PYTHONPATH to
# /path-to/current/linux-x86_64/lib from your install directory

#sys.path.append("/local/apps/VisIt/2.3.0/current/linux-x86_64/lib")    # for _simV2.so
#sys.path.append("/local/apps/VisItSrc/src/sim/V2/swig/python")         # for simV2.py

#sys.path.append('/local/apps/VisIt/2.3.0/current/linux-x86_64/lib')
#sys.path.append('/Applications/VisIt.app/Contents/Resources/2.12.0/darwin-x86_64/lib')

#sys.path.append('/Applications/VisIt.app/Contents/Resources/2.12.0/darwin-x86_64/include/visit/visitpy/') #mpicom

from simV2 import *
#import mpicom  # only for parallel
from mpi4py import MPI  # other module for parallel
def __init__(self):
        Simulation.__init__(self)
        self.VISIT_COMMAND_PROCESS = 0
        self.VISIT_COMMAND_SUCCESS = 1
        self.VISIT_COMMAND_FAILURE = 2
MAX_CYCLE = 200

class Simulation:
    def __init__(self):
        self.done = 0
        self.cycle = 0
        self.time = 0.
#        self.runMode = VISIT_SIMMODE_STOPPED
        self.par_size = 1
        self.par_rank = 0
        self.savingFiles = 0
        self.saveCounter = 0
        self.M = 20
        self.MP = self.M # to be redefined if parallel
        self.rmesh_dims = [self.M+2, self.M+2, 1]
        self.dx = 1.0/(self.rmesh_dims[0]-1)
        self.rmesh_ndims = 2
        self.commands = ("halt", "step", "run", "addplot")
        self.v = sp.zeros([self.rmesh_dims[0],self.rmesh_dims[1]]) # includes 2 ghosts
        self.vnew = sp.zeros([self.M, self.M])

    def Initialize(self, simname, visitdir):
        VisItOpenTraceFile("trace.%d.txt" % self.par_rank)
        VisItSetDirectory(visitdir)
        VisItSetupEnvironment()
 #       if self.par_rank == 0:
 #           VisItInitializeSocketAndDumpSimFile(simname, 
 #			"Heat Equation solver in Python", 
 #			"/path/to/where/visit/was/started",
 #			None, None, None)
        self.rmesh_dims = [self.MP+2, self.M+2, 1]
        self.v = sp.zeros([self.rmesh_dims[0],self.rmesh_dims[1]]) # includes 2 ghosts
        self.vnew = sp.zeros([self.MP, self.M])
        self.set_initial_bc()

    def set_initial_bc(self):
        if self.par_size > 1:
          if self.par_rank == 0:
            for j in range(self.rmesh_dims[1]):
              self.v[0,j] = math.sin(math.pi*j*self.dx)
          if self.par_rank == (self.par_size-1):
            for j in range(self.rmesh_dims[1]):
              self.v[-1,j] = math.sin(math.pi*j*self.dx)* math.exp(-math.pi)
        else:
          #first (bottom) row
          for j in range(self.rmesh_dims[1]):
            self.v[0,j] = math.sin(math.pi*j*self.dx)
          #last (top) row
          self.v[-1,:] = self.v[0,:]* math.exp(-math.pi)

    def Finalize(self):
        VisItCloseTraceFile()
        
    def SimulateOneTimestep(self):
        self.cycle = self.cycle + 1
        self.time = self.time + 1.
        if self.par_rank == 0:
            print "Simulating time step: cycle=%d, time=%g" % (self.cycle,self.time)

        self.vnew = 0.25 * ( self.v[2:, 1:-1]  + # north neighbor
                             self.v[0:-2, 1:-1] + # south neighbor
                             self.v[1:-1, 2:] + # east neighbor
                             self.v[1:-1, :-2]) # west neighbor
        # copy now vnew to the interior region of v, leaving the boundary walls untouched.
        self.v[1:-1,1:-1] = self.vnew.copy()

        if self.par_size >1:
          # if in parallel, exchange ghost cells now
          # define who is my neighbor above and below

          below = self.par_rank-1
          above = self.par_rank+1
          if self.par_rank == 0:
            below = MPI.PROC_NULL

          if self.par_rank == (self.par_size-1):
            above = MPI.PROC_NULL
          ret1 = self.comm.sendrecv(self.v[-2,], dest=above, source=below)
          if ret1 is not None:
            self.v[0,] = ret1
          ret1 = self.comm.sendrecv(self.v[1,], dest=below, source=above)
          if ret1 is not None:
            self.v[-1,] = ret1

        # Tell visit a new time step was computed
        VisItTimeStepChanged()

        # Tell visit to update the plots
        VisItUpdatePlots()
        if self.savingFiles:
            filename = "Heat%04d.jpg" % self.saveCounter
            if VisItSaveWindow(filename, 800, 800, VISIT_IMAGEFORMAT_JPEG) == VISIT_OKAY:
                self.saveCounter = self.saveCounter + 1
                if self.par_rank == 0:
                    print "Saved",filename
            else:
                print "The image could not be saved to",filename

        if self.cycle >= MAX_CYCLE:
            self.done = True
            

	def ProcessVisItCommand(self):
		return (VisItProcessEngineCommand() == VISIT_OKAY)

	def DoPrompt(self):
		if self.par_rank == 0:
			sys.stdout.write("command>")
			sys.stdout.flush()

	# A commenter
    #def ProcessConsoleCommand(self):
        #cmd = self.GetInputFromConsole()

        #if cmd == "quit":
            #self.done = 1
        #elif cmd == "halt":
            #self.runMode = VISIT_SIMMODE_STOPPED
        #elif cmd == "step":
            #self.SimulateOneTimestep()
        #elif cmd == "run":
            #self.runMode = VISIT_SIMMODE_RUNNING
        #elif cmd == "saveon":
            #self.savingFiles = 1
        #elif cmd == "saveoff":
            #self.savingFiles = 0
        #elif cmd == "addplot":
            #VisItExecuteCommand('AddPlot("Pseudocolor", "temperature")\n')
            #VisItExecuteCommand('DrawPlots()\n')

    def GetInputFromConsole(self):
        return VisItReadConsole()

    def GetInputFromVisIt(self, blocking):
        console = sys.stdin.fileno()
        return VisItDetectInput(blocking, console)

    def MainLoop(self):
		if VisItInitializeRuntime() != VISIT_OKAY :
			print "FATAL ERROR - VisIt could not be properly initialized"
			exit(1)
		
		VisItSetSlaveProcessCallback(self.slave_process_callback)
		VisItSetGetMetaData(self.GetMetaData, None)
		VisItSetGetMesh(self.GetMesh, 0)
		VisItSetGetVariable(self.GetVariable, 0)
		VisItSetGetDomainList(self.GetDomainList, 0)
        
		VisItTimeStepChanged()
		VisItAddPlot("Mesh", "mesh");
		VisItAddPlot("Pseudocolor", "temperature");
		VisItDrawPlots();
		
		while self.done == 0 :			
			# jpg image each 10 time step until time step 100
			if self.cycle <= 100 and self.cycle%10 == 0 :
				VisItTimeStepChanged()
				VisItUpdatePlots()
				filename = "pjacobi_img_%05d.jpg" % self.cycle
				ierr = VisItSaveWindow(filename, 800, 800, VISIT_IMAGEFORMAT_JPEG) 
				if ierr == VISIT_OKAY :
					print "Saved",filename 
				
				
			# VTK file each 10 time step until time step 100
			if self.cycle <= 100 and self.cycle%10 == 0 :
				VisItTimeStepChanged()
				var = VisIt_NameList_alloc()
				VisIt_NameList_addName(var, "default")
				VisIt_NameList_addName(var, "mesh2d/nodeid")
				
				options = VisIt_OptionList_alloc()
				VisIt_OptionList_setValueB(options,"Strip mesh name prefix",1)
				filename = "pjacobi_%05d.jpg" % self.cycle
				VisItExportDatabaseWithOptions(filename,"VTK_1.0",var,options)
				if ierr == VISIT_OKAY :
					print "Exported %s.vtk" % filename
				else :
					print "Pb visitexportdatabasewithoptions"
				VisIt_NameList_free(var);
				VisIt_OptionList_free(options)
			
			self.SimulateOneTimestep()

    def ConnectCallbacks(self):
        VisItSetCommandCallback(self.ControlCommandCallback, 0)
        VisItSetGetMetaData(self.GetMetaData, None)
        VisItSetGetMesh(self.GetMesh, 0)
        VisItSetGetVariable(self.GetVariable, 0)

    #
    # Data access and control functions
    #

    def ControlCommandCallback(self, cmd, args, cbdata):
        if cmd == "halt":
            self.runMode = VISIT_SIMMODE_STOPPED
        elif cmd == "step":
            self.SimulateOneTimestep()
        elif cmd == "run":
            self.runMode = VISIT_SIMMODE_RUNNING
        elif cmd == "addplot":
            VisItExecuteCommand('AddPlot("Pseudocolor", "temperature")\n')
            VisItExecuteCommand('DrawPlots()\n')

    # Callback called by visit to get metadata
    def GetMetaData(self, cbdata):
        md = VisIt_SimulationMetaData_alloc()
        if md != VISIT_INVALID_HANDLE:
            VisIt_SimulationMetaData_setCycleTime(md, self.cycle, self.time)
            # Add mesh metadata.
            mmd = VisIt_MeshMetaData_alloc()
            if mmd != VISIT_INVALID_HANDLE:
                # Set the mesh's properties.
                VisIt_MeshMetaData_setName(mmd, "mesh")
                VisIt_MeshMetaData_setMeshType(mmd, VISIT_MESHTYPE_RECTILINEAR)
                VisIt_MeshMetaData_setTopologicalDimension(mmd, 2)
                VisIt_MeshMetaData_setSpatialDimension(mmd, 2)
                VisIt_MeshMetaData_setNumDomains(mmd, self.par_size)
                VisIt_MeshMetaData_setDomainTitle(mmd, "Domains")
                VisIt_MeshMetaData_setDomainPieceName(mmd, "domain")
                VisIt_MeshMetaData_setNumGroups(mmd, 0)
                VisIt_MeshMetaData_setXUnits(mmd, "cm")
                VisIt_MeshMetaData_setYUnits(mmd, "cm")
                VisIt_MeshMetaData_setXLabel(mmd, "Width")
                VisIt_MeshMetaData_setYLabel(mmd, "Height")
                VisIt_SimulationMetaData_addMesh(md, mmd)

            # Add a variable.
            vmd = VisIt_VariableMetaData_alloc()
            if vmd != VISIT_INVALID_HANDLE:
                VisIt_VariableMetaData_setName(vmd, "temperature")
                VisIt_VariableMetaData_setMeshName(vmd, "mesh")
                VisIt_VariableMetaData_setType(vmd, VISIT_VARTYPE_SCALAR)
                VisIt_VariableMetaData_setCentering(vmd, VISIT_VARCENTERING_NODE)
                VisIt_SimulationMetaData_addVariable(md, vmd)

            # Add some commands
            for c in self.commands:
                cmd = VisIt_CommandMetaData_alloc()
                if cmd != VISIT_INVALID_HANDLE:
                    VisIt_CommandMetaData_setName(cmd, c)
                    VisIt_SimulationMetaData_addGenericCommand(md, cmd)
        return md

    def GetMesh(self, domain, name, cbdata):
        h = VISIT_INVALID_HANDLE
        if name == "mesh":
            h = VisIt_RectilinearMesh_alloc()
            if h != VISIT_INVALID_HANDLE:
                minRealIndex = [0,0,0]
                maxRealIndex = [0,0,0]
                maxRealIndex[0] = self.rmesh_dims[1]-1
                if self.par_rank == (self.par_size-1):
                  maxRealIndex[1] = self.rmesh_dims[0]-1
                else:
                  maxRealIndex[1] = self.rmesh_dims[0]-2

                maxRealIndex[2] = self.rmesh_dims[2]-1

                rmesh_x = sp.linspace(0.0, 1.0, self.rmesh_dims[1])
                rmesh_y = sp.linspace(self.par_rank*self.MP*self.dx, ((self.par_rank+1)*self.MP+1)*self.dx, self.rmesh_dims[0])
                hx = VisIt_VariableData_alloc()
                hy = VisIt_VariableData_alloc()
                VisIt_VariableData_setDataD(hx, VISIT_OWNER_SIM, 1,
                                            rmesh_x.size, rmesh_x.tolist())
                VisIt_VariableData_setDataD(hy, VISIT_OWNER_SIM, 1,
                                            rmesh_y.size, rmesh_y.tolist())
                VisIt_RectilinearMesh_setCoordsXY(h, hx, hy)
                VisIt_RectilinearMesh_setRealIndices(h, minRealIndex, maxRealIndex)
        return h

    def GetVariable(self, domain, name, cbdata):
        h = VISIT_INVALID_HANDLE
        if name == "temperature":
            nTuples = self.v.size
            h = VisIt_VariableData_alloc()
            VisIt_VariableData_setDataD(h, VISIT_OWNER_SIM, 1,
                nTuples, self.v.flatten().tolist())
        return h


#*****************************************************************************
# Class: ParallelSimulation
#
# Purpose:
#   This subclass of Simulation overrides some input handling methods and
#   provides a few extra callbacks to provide an MPI-parallel simulation.
#
# Programmer: Brad Whitlock
# Date:       Fri Mar 18 14:24:17 PDT 2011
#
# Modifications:
#
#*****************************************************************************

class ParallelSimulation(Simulation):
    def __init__(self):
        Simulation.__init__(self)
        self.VISIT_COMMAND_PROCESS = 0
        self.VISIT_COMMAND_SUCCESS = 1
        self.VISIT_COMMAND_FAILURE = 2

    # Override Initialize for parallel
    def Initialize(self, simname, visitdir):
        # mpi4py
        self.comm = MPI.COMM_WORLD

		# Installing broadcast functions as required by libsim
        VisItSetBroadcastIntFunction(self.broadcast_int)
        VisItSetBroadcastStringFunction(self.broadcast_string)

        self.par_size = self.comm.Get_size() #mpicom.size()
        self.par_rank = self.comm.Get_rank() #mpicom.rank()

		# Informing VisIt about parallel operation and about rank
        VisItSetParallel(self.par_size  >1)
        VisItSetParallelRank(self.par_rank)


        self.MP = self.M / self.par_size
        Simulation.Initialize(self, simname, visitdir)

        print "Hello - I am rank ",self.par_rank,"/",self.par_size

    # Wrapper to VisItReadConsole
    # If rank 0, read characters from console, else read broadcasted characters
    # Return the command
    def GetInputFromConsole(self):
        cmd = ""
        if self.par_rank == 0:
            cmd = VisItReadConsole()
            self.comm.bcast(cmd, root=0)
        else:
            cmd = self.comm.bcast(obj=None, root=0)
        return cmd

    # Wrapper to GetInputFromVisIt
    # If rank 0, call VisItDetectInput, else read status from broadcast
    # Return the status
    def GetInputFromVisIt(self, blocking):
        s = 0
        if self.par_rank == 0:
             
            print "coucou - blocking = " + str(blocking)
            s = VisItDetectInput(blocking, -1)
            print "coucou - self.GetInputFromVisIt returned " + str(s)
            self.comm.bcast(s, root=0)
        else:
            s = self.comm.bcast(obj=None, root=0)
        return s
		

	def Finalize(self):
		Simulation.Finalize(self)
		if self.par_rank==0:
			print "Bye bye"

    def ConnectCallbacks(self):
        Simulation.ConnectCallbacks(self)
        VisItSetSlaveProcessCallback(self.slave_process_callback)
        VisItSetGetDomainList(self.GetDomainList, 0)

    def GetDomainList(self, name, cbdata):
        h = VisIt_DomainList_alloc()
        if h != VISIT_INVALID_HANDLE:
            hdl = VisIt_VariableData_alloc()
            VisIt_VariableData_setDataI(hdl, VISIT_OWNER_VISIT, 1, 1, [self.par_rank])
            VisIt_DomainList_setDomains(h, self.par_size, hdl)
        return h


    # Callback functions required by libsim to broadcast int or string
    def broadcast_int(self, ival, sender):
        if self.par_rank == 0:
            ret = self.comm.bcast(ival, root=0)
        else:
            ret = self.comm.bcast(obj=None, root=0)
        return ret

    def broadcast_string(self, sval, slen, sender):
        if self.par_rank == 0:
            ret = self.comm.bcast(sval, root=0)
        else:
            ret = self.comm.bcast(obj=None, root=0)
        return ret

    def slave_process_callback(self):
        s = self.VISIT_COMMAND_PROCESS
        if self.par_rank == 0:
            self.comm.bcast(s, root=0)
        else:
            self.comm.bcast(obj=None, root=0)

    def ProcessVisItCommand(self):
        if self.par_rank == 0:
            success = VisItProcessEngineCommand()
            if success == VISIT_OKAY:
                self.comm.bcast(self.VISIT_COMMAND_SUCCESS, root=0)
                return 1
            else:
                self.comm.bcast(self.VISIT_COMMAND_FAILURE)
                return 0
        else:
            while 1:
                command = self.comm.bcast(obj=None, root=0)
                if command == self.VISIT_COMMAND_PROCESS:
                    VisItProcessEngineCommand()
                elif command == self.VISIT_COMMAND_SUCCESS:
                    return 1
                elif command == self.VISIT_COMMAND_FAILURE:
                    return 0

#
# Main program
#
def main():
    #sim = Simulation()
    #sim.Initialize("heat_equation", "../../..")
    sim = ParallelSimulation()
    sim.Initialize("heat_equation_par", "../../..")
    sim.MainLoop()
    sim.Finalize()

main()


