
import time

import sys, math
import numpy as sp

# this code is largely inspired from Brad Whitlock's updateplots.py

# this code is similar in nature and algorithm to the F90 code contributed earlier
# see tools/DataManualExamples/Simulations/contrib
# this is a very first attempt to use some numpy arrays.
# A grid of size (M+2)x(M+2) is allocated.
# the working buffer is of size MxM
#

# no need for that if you set your PYTHONPATH to
# /path-to/current/linux-x86_64/lib from your install directory

#sys.path.append("/local/apps/VisIt/2.3.0/current/linux-x86_64/lib")    # for _simV2.so
#sys.path.append("/local/apps/VisItSrc/src/sim/V2/swig/python")         # for simV2.py

#sys.path.append('/local/apps/VisIt/2.3.0/current/linux-x86_64/lib')
#sys.path.append('/Applications/VisIt.app/Contents/Resources/2.12.0/darwin-x86_64/lib')

#sys.path.append('/Applications/VisIt.app/Contents/Resources/2.12.0/darwin-x86_64/include/visit/visitpy/') #mpicom

from simV2 import *
#import mpicom  # only for parallel
from mpi4py import MPI  # other module for parallel

MAX_CYCLE = 200

class Simulation:
    def __init__(self):
        self.done = 0
        self.cycle = 0
        self.time = 0.
        self.runMode = VISIT_SIMMODE_STOPPED
        self.par_size = 1
        self.par_rank = 0
        self.savingFiles = 0
        self.saveCounter = 0
        self.M = 20
        self.MP = self.M # to be redefined if parallel
        self.rmesh_dims = [self.M+2, self.M+2, 1]
        self.dx = 1.0/(self.rmesh_dims[0]-1)
        self.rmesh_ndims = 2
        self.commands = ("halt", "step", "run", "addplot") # Add new command here
        self.v = sp.zeros([self.rmesh_dims[0],self.rmesh_dims[1]]) # includes 2 ghosts
        self.vnew = sp.zeros([self.M, self.M])

    def Initialize(self, simname, visitdir):
        VisItOpenTraceFile("trace.%d.txt" % self.par_rank)
        VisItSetDirectory(visitdir)
        VisItSetupEnvironment()
        if self.par_rank == 0:
            VisItInitializeSocketAndDumpSimFile(simname,
                "Heat Equation solver in Python", "/path/to/where/visit/was/started",
                 None, None, None)
        self.rmesh_dims = [self.MP+2, self.M+2, 1]
        self.v = sp.zeros([self.rmesh_dims[0],self.rmesh_dims[1]]) # includes 2 ghosts
        self.vnew = sp.zeros([self.MP, self.M])
        self.set_initial_bc()

    def set_initial_bc(self):
        if self.par_size > 1:
          if self.par_rank == 0:
            for j in range(self.rmesh_dims[1]):
              self.v[0,j] = math.sin(math.pi*j*self.dx)
          if self.par_rank == (self.par_size-1):
            for j in range(self.rmesh_dims[1]):
              self.v[-1,j] = math.sin(math.pi*j*self.dx)* math.exp(-math.pi)
        else:
          #first (bottom) row
          for j in range(self.rmesh_dims[1]):
            self.v[0,j] = math.sin(math.pi*j*self.dx)
          #last (top) row
          self.v[-1,:] = self.v[0,:]* math.exp(-math.pi)
          
          ### Add here for the new variable

    def Finalize(self):
        VisItCloseTraceFile()

    def SimulateOneTimestep(self):
        self.cycle = self.cycle + 1
        self.time = self.time + 1.
        if self.par_rank == 0:
            print "Simulating time step: cycle=%d, time=%g" % (self.cycle,self.time)

        self.vnew = 0.25 * ( self.v[2:, 1:-1]  + # north neighbor
                             self.v[0:-2, 1:-1] + # south neighbor
                             self.v[1:-1, 2:] + # east neighbor
                             self.v[1:-1, :-2]) # west neighbor
        # copy now vnew to the interior region of v, leaving the boundary walls untouched.
        self.v[1:-1,1:-1] = self.vnew.copy()
        
        ### Add here for the new variable

        if self.par_size >1:
          # if in parallel, exchange ghost cells now
          # define who is my neighbor above and below

          below = self.par_rank-1
          above = self.par_rank+1
          if self.par_rank == 0:
            #below = mpicom.MPI_PROC_NULL   # tells MPI not to perform send/recv
            below = MPI.PROC_NULL

          if self.par_rank == (self.par_size-1):
            #above = mpicom.MPI_PROC_NULL   # should only receive/send from/to below
            above = MPI.PROC_NULL
          #ret1 = mpicom.sendrecv(self.v[-2,], above, below)
          ret1 = self.comm.sendrecv(self.v[-2,], dest=above, source=below)
          #print "ret1 a = ",ret1
          if ret1 is not None:
            self.v[0,] = ret1
          #ret1 = mpicom.sendrecv(self.v[1,], below, above)
          ret1 = self.comm.sendrecv(self.v[1,], dest=below, source=above)
          #print "ret1 b = ",ret1
          if ret1 is not None:
            self.v[-1,] = ret1

        # Tell visit a new time step was computed
        VisItTimeStepChanged()

	# Tell visit to update the plots
        VisItUpdatePlots()
        if self.savingFiles:
            filename = "Heat%04d.jpg" % self.saveCounter
            if VisItSaveWindow(filename, 800, 800, VISIT_IMAGEFORMAT_JPEG) == VISIT_OKAY:
                self.saveCounter = self.saveCounter + 1
                if self.par_rank == 0:
                    print "Saved",filename
            else:
                print "The image could not be saved to",filename

	if self.cycle >= MAX_CYCLE:
		self.done = True

    def ProcessVisItCommand(self):
        return (VisItProcessEngineCommand() == VISIT_OKAY)

    def DoPrompt(self):
        if self.par_rank == 0:
            sys.stdout.write("command>")
            sys.stdout.flush()

    def ProcessConsoleCommand(self):
        cmd = self.GetInputFromConsole()

        if cmd == "quit":
            self.done = 1
        elif cmd == "halt":
            self.runMode = VISIT_SIMMODE_STOPPED
        elif cmd == "step":
            self.SimulateOneTimestep()
        elif cmd == "run":
            self.runMode = VISIT_SIMMODE_RUNNING
        elif cmd == "saveon":
            self.savingFiles = 1
        elif cmd == "saveoff":
            self.savingFiles = 0
        elif cmd == "addplot":
            VisItExecuteCommand('AddPlot("Pseudocolor", "temperature")\n')
            VisItExecuteCommand('DrawPlots()\n')

    def GetInputFromConsole(self):
        return VisItReadConsole()

    def GetInputFromVisIt(self, blocking):
        console = sys.stdin.fileno()
        #return VisItDetectInputWithTimeout(blocking, 100000, console)
        #return VisItDetectInput(blocking, -1)
        return VisItDetectInput(blocking, console)

    def MainLoop(self):
        # self.DoPrompt()
        while self.done == 0:
            blocking = 1
            if self.runMode == VISIT_SIMMODE_RUNNING:
                blocking = 0

            visitstate = self.GetInputFromVisIt(blocking)

            # 0 --> timeout, visit is not there, just go on with simulation
            if visitstate == 0:
                #time.sleep(1)				# Simulating a heaviy computation
                self.SimulateOneTimestep()

            # 1 --> VisIt is trying to connect - Try to complete connection
            elif visitstate == 1:
                if VisItAttemptToCompleteConnection() == VISIT_OKAY:
		    if self.par_rank==0:
                       print "VisIt connected"
                    self.runMode = VISIT_SIMMODE_STOPPED
                    self.ConnectCallbacks()
                else:
		   if self.par_rank==0:
                       print "VisIt did not connect"

            # 2 --> VisIt telling something to the program, try to process 
            elif visitstate == 2:
                if not self.ProcessVisItCommand():
                    VisItDisconnect()
                    self.runMode = VISIT_SIMMODE_RUNNING

            # 3 --> ????
            elif visitstate == 3:
                self.ProcessConsoleCommand()
                self.DoPrompt()

            # Anything else --> Error
            else:
                print "Error: ", visitstate

    def ConnectCallbacks(self):
        VisItSetCommandCallback(self.ControlCommandCallback, 0)
        VisItSetGetMetaData(self.GetMetaData, None)
        VisItSetGetMesh(self.GetMesh, 0)
        VisItSetGetVariable(self.GetVariable, 0)

    #
    # Data access and control functions
    #

    def ControlCommandCallback(self, cmd, args, cbdata):
        if cmd == "halt":
            self.runMode = VISIT_SIMMODE_STOPPED
        elif cmd == "step":
            self.SimulateOneTimestep()
        elif cmd == "run":
            self.runMode = VISIT_SIMMODE_RUNNING
        elif cmd == "addplot":
            VisItExecuteCommand('AddPlot("Pseudocolor", "temperature")\n')
            VisItExecuteCommand('DrawPlots()\n')
	### Add here for the new variable

    # Callback called by visit to get metadata
    def GetMetaData(self, cbdata):
        md = VisIt_SimulationMetaData_alloc()
        if md != VISIT_INVALID_HANDLE:
            VisIt_SimulationMetaData_setCycleTime(md, self.cycle, self.time)
            # Add mesh metadata.
            mmd = VisIt_MeshMetaData_alloc()
            if mmd != VISIT_INVALID_HANDLE:
                # Set the mesh's properties.
                VisIt_MeshMetaData_setName(mmd, "mesh")
                VisIt_MeshMetaData_setMeshType(mmd, VISIT_MESHTYPE_RECTILINEAR)
                VisIt_MeshMetaData_setTopologicalDimension(mmd, 2)
                VisIt_MeshMetaData_setSpatialDimension(mmd, 2)
                VisIt_MeshMetaData_setNumDomains(mmd, self.par_size)
                VisIt_MeshMetaData_setDomainTitle(mmd, "Domains")
                VisIt_MeshMetaData_setDomainPieceName(mmd, "domain")
                VisIt_MeshMetaData_setNumGroups(mmd, 0)
                VisIt_MeshMetaData_setXUnits(mmd, "cm")
                VisIt_MeshMetaData_setYUnits(mmd, "cm")
                VisIt_MeshMetaData_setXLabel(mmd, "Width")
                VisIt_MeshMetaData_setYLabel(mmd, "Height")
                VisIt_SimulationMetaData_addMesh(md, mmd)

            # Add a variable.
            vmd = VisIt_VariableMetaData_alloc()
            if vmd != VISIT_INVALID_HANDLE:
                VisIt_VariableMetaData_setName(vmd, "temperature")
                VisIt_VariableMetaData_setMeshName(vmd, "mesh")
                VisIt_VariableMetaData_setType(vmd, VISIT_VARTYPE_SCALAR)
                VisIt_VariableMetaData_setCentering(vmd, VISIT_VARCENTERING_NODE)
                VisIt_SimulationMetaData_addVariable(md, vmd)

            ### Add a vector variable on mesh2d here

            # Add some commands
            for c in self.commands:
                cmd = VisIt_CommandMetaData_alloc()
                if cmd != VISIT_INVALID_HANDLE:
                    VisIt_CommandMetaData_setName(cmd, c)
                    VisIt_SimulationMetaData_addGenericCommand(md, cmd)
        return md

    def GetMesh(self, domain, name, cbdata):
        h = VISIT_INVALID_HANDLE
        if name == "mesh":
            h = VisIt_RectilinearMesh_alloc()
            if h != VISIT_INVALID_HANDLE:
                minRealIndex = [0,0,0]
                maxRealIndex = [0,0,0]
                maxRealIndex[0] = self.rmesh_dims[1]-1
                if self.par_rank == (self.par_size-1):
                  maxRealIndex[1] = self.rmesh_dims[0]-1
                else:
                  maxRealIndex[1] = self.rmesh_dims[0]-2

                maxRealIndex[2] = self.rmesh_dims[2]-1

                rmesh_x = sp.linspace(0.0, 1.0, self.rmesh_dims[1])
                rmesh_y = sp.linspace(self.par_rank*self.MP*self.dx, ((self.par_rank+1)*self.MP+1)*self.dx, self.rmesh_dims[0])
                hx = VisIt_VariableData_alloc()
                hy = VisIt_VariableData_alloc()
                VisIt_VariableData_setDataD(hx, VISIT_OWNER_SIM, 1,
                                            rmesh_x.size, rmesh_x.tolist())
                VisIt_VariableData_setDataD(hy, VISIT_OWNER_SIM, 1,
                                            rmesh_y.size, rmesh_y.tolist())
                VisIt_RectilinearMesh_setCoordsXY(h, hx, hy)
                VisIt_RectilinearMesh_setRealIndices(h, minRealIndex, maxRealIndex)
        return h

    def GetVariable(self, domain, name, cbdata):
        h = VISIT_INVALID_HANDLE
        if name == "temperature":
            nTuples = self.v.size
            h = VisIt_VariableData_alloc()
            VisIt_VariableData_setDataD(h, VISIT_OWNER_SIM, 1,
                nTuples, self.v.flatten().tolist())
	### Add here for the new variable gradtemp
        return h


#*****************************************************************************
# Class: ParallelSimulation
#
# Purpose:
#   This subclass of Simulation overrides some input handling methods and
#   provides a few extra callbacks to provide an MPI-parallel simulation.
#
# Programmer: Brad Whitlock
# Date:       Fri Mar 18 14:24:17 PDT 2011
#
# Modifications:
#
#*****************************************************************************

class ParallelSimulation(Simulation):
    def __init__(self):
        Simulation.__init__(self)
        self.VISIT_COMMAND_PROCESS = 0
        self.VISIT_COMMAND_SUCCESS = 1
        self.VISIT_COMMAND_FAILURE = 2

    # Override Initialize for parallel
    def Initialize(self, simname, visitdir):
        # mpi4py
        self.comm = MPI.COMM_WORLD
        #mpicom.init(sys.argv)

	# Installing broadcast functions as required by libsim
        VisItSetBroadcastIntFunction(self.broadcast_int)
        VisItSetBroadcastStringFunction(self.broadcast_string)

        self.par_size = self.comm.Get_size() #mpicom.size()
        self.par_rank = self.comm.Get_rank() #mpicom.rank()

	# Informing VisIt about parallel operation and about rank
        VisItSetParallel(self.par_size  >1)
        VisItSetParallelRank(self.par_rank)


        self.MP = self.M / self.par_size
        Simulation.Initialize(self, simname, visitdir)

        print "Hello - I am rank ",self.par_rank,"/",self.par_size
	#print self.v

    # Wrapper to VisItReadConsole
    # If rank 0, read characters from console, else read broadcasted characters
    # Return the command
    def GetInputFromConsole(self):
        cmd = ""
        if self.par_rank == 0:
            cmd = VisItReadConsole()
            #mpicom.broadcast(cmd)
            self.comm.bcast(cmd, root=0)
        else:
            #cmd = mpicom.broadcast()
            cmd = self.comm.bcast(obj=None, root=0)
        return cmd

    # Wrapper to GetInputFromVisIt
    # If rank 0, call VisItDetectInput, else read status from broadcast
    # Return the status
    def GetInputFromVisIt(self, blocking):
        s = 0
        if self.par_rank == 0:
            #console = sys.stdin.fileno()
            #s = VisItDetectInputWithTimeout(blocking, 100000, console)
             
            print "coucou - blocking = " + str(blocking)
            s = VisItDetectInput(blocking, -1)
            print "coucou - self.GetInputFromVisIt returned " + str(s)
            #mpicom.broadcast(s)
            self.comm.bcast(s, root=0)
        else:
            #s = mpicom.broadcast()
            s = self.comm.bcast(obj=None, root=0)
        return s

    def Finalize(self):
        Simulation.Finalize(self)
	if self.par_rank==0:
		print "Bye bye"
        #mpicom.finalize()

    def ConnectCallbacks(self):
        Simulation.ConnectCallbacks(self)
        VisItSetSlaveProcessCallback(self.slave_process_callback)
        VisItSetGetDomainList(self.GetDomainList, 0)

    def GetDomainList(self, name, cbdata):
        h = VisIt_DomainList_alloc()
        if h != VISIT_INVALID_HANDLE:
            hdl = VisIt_VariableData_alloc()
            VisIt_VariableData_setDataI(hdl, VISIT_OWNER_VISIT, 1, 1, [self.par_rank])
            VisIt_DomainList_setDomains(h, self.par_size, hdl)
        return h


    # Callback functions required by libsim to broadcast int or string
    def broadcast_int(self, ival, sender):
        if self.par_rank == 0:
            #ret = mpicom.broadcast(ival)
            ret = self.comm.bcast(ival, root=0)
        else:
            #ret = mpicom.broadcast()
            ret = self.comm.bcast(obj=None, root=0)
        return ret

    def broadcast_string(self, sval, slen, sender):
        if self.par_rank == 0:
            #ret = mpicom.broadcast(sval)
            ret = self.comm.bcast(sval, root=0)
        else:
            #ret = mpicom.broadcast()
            ret = self.comm.bcast(obj=None, root=0)
        return ret

    def slave_process_callback(self):
        s = self.VISIT_COMMAND_PROCESS
        if self.par_rank == 0:
            #mpicom.broadcast(s)
            self.comm.bcast(s, root=0)
        else:
            #mpicom.broadcast()
            self.comm.bcast(obj=None, root=0)

    def ProcessVisItCommand(self):
        if self.par_rank == 0:
            success = VisItProcessEngineCommand()
            if success == VISIT_OKAY:
                #mpicom.broadcast(self.VISIT_COMMAND_SUCCESS)
                self.comm.bcast(self.VISIT_COMMAND_SUCCESS, root=0)
                return 1
            else:
                #mpicom.broadcast(self.VISIT_COMMAND_FAILURE)
                self.comm.bcast(self.VISIT_COMMAND_FAILURE)
                return 0
        else:
            while 1:
                #command = mpicom.broadcast()
                command = self.comm.bcast(obj=None, root=0)
                if command == self.VISIT_COMMAND_PROCESS:
                    VisItProcessEngineCommand()
                elif command == self.VISIT_COMMAND_SUCCESS:
                    return 1
                elif command == self.VISIT_COMMAND_FAILURE:
                    return 0
	    
	def MPIIOWriteData(self, filename):
		
		amode = MPI.MODE_WRONLY|MPI.MODE_CREATE
		fh = MPI.File.Open(MPI.COMM_WORLD, filename, amode)
		
		dimuids = [self.M+2, self.M+2]
		disp = 0
		
#		buffer1 = np.empty(2, dtype=np.int)
		
		if self.par_rank == 0 :
			fh.Write( dimuids )
				
		disp = 2*MPI.INT.Get_size()
		ustart = [self.MP*self.par_rank, 0]
		ucount = [self.MP, self.M+2]
		
		if self.par_rank == self.par_size-1 :
			ucount[0] = self.MP+2
		
		# Create the subarray representing the local block	
		filetype = MPI.DOUBLE.Create_subarray(2, dimuids,ucount, ustart, ORDER_C) 
		filetype.Commit()
		
		fh.Set_view(disp, MPI.DOUBLE, filetype=filetype)
		
		fh.Write_all(self.v)  
		filetype.Free()
		fh.Close()

#
# Main program
#
def main():
    #sim = Simulation()
    #sim.Initialize("heat_equation", "../../..")
    sim = ParallelSimulation()
    sim.Initialize("heat_equation_par", "../../..")
    sim.MainLoop()
    sim.Finalize()

main()


