! A parallel Jacobi solver for the Laplacian equation in 2D
! Written by Jean M. Favre, Swiss National Supercomputing Center
! Sun Oct 31 2010
! Code inspired from an older example by Kadin Tseng, Boston University, November 1999
! The compile flag -D_VISIT_ enables compilation with VisIt. Otherwise, the program runs
! in stand-alone mode

program Jacobi
use jacobi_module
implicit none
#if ( defined _VISIT_ )
  include "visitfortransimV2interface.inc"
  external processvisitcommand
  integer processvisitcommand
#endif
  character (len=80) :: filename
  integer :: vars,options  

  call InitMPI()

  m = 20 ! mesh size = (m+2)x(m+2) including the bc grid lines
! We make no attempt to check that the number of grid points divides evenly
! with the number of MPI tasks.
! rank 0 will display the bottom (southern) boundary wall
! rank (size-1) will display the top (northern) boundary wall
! if run with m=20 and 4 MPI tasks, we will have 5 grid lines per rank
! and VisIt will display a 22x22 grid
#if ( defined _VISIT_ )
  if(par_size.gt.1) then
    ierr = VISITSETPARALLEL(1)
  end if
  ierr = VISITSETPARALLELRANK(par_rank)

  call SIMULATIONARGUMENTS()

  ierr = VISITSETUPENV()
!     Have the master process write the sim file.
  if(par_rank.eq.0) then
  ierr = VISITINITIALIZESIM("pjacobi", 7, &
            "Interface to pjacobi", 20, &
            "/no/useful/path", 15, &
            VISIT_F77NULLSTRING, VISIT_F77NULLSTRINGLEN, &
            VISIT_F77NULLSTRING, VISIT_F77NULLSTRINGLEN, &
            VISIT_F77NULLSTRING, VISIT_F77NULLSTRINGLEN)
  end if

#endif

  call MPI_Bcast(m, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)

! no check is done to make sure the two numbers below divide evenly
  mp = m/par_size                            ! columns for each proc

  print *, 'proc ', par_rank, ' out of ', par_size, ' m=', m, ' mp=', mp

  allocate ( Temp(0:m+1,0:mp+1), oldTemp(0:m+1,0:mp+1) )  ! mem for Temp, v
  allocate ( GradTemp(2,0:m+1,0:mp+1) )  ! mem for GradTemp

  call set_initial_bc()             ! set up boundary values

#if ( defined _VISIT_ )
      ierr = visitinitializeruntime()
      ierr = visittimestepchanged()
      ierr = visitaddplot("Mesh", 4, "mesh", 4)
      ierr = visitaddplot("Pseudocolor", 11, "temperature", 11)
      ierr = visitdrawplots()
#endif

  do while (gdel > TOL)               ! iterate until error below threshold

  if(iter > MAXSTEPS) then
    write(*,*)'Iteration terminated (exceeds ', MAXSTEPS, ')'
    exit                            ! nonconvergent solution
  end if

#if ( defined _VISIT_ )
    ! jpg image each 10 time step until time step 100
    if ((iter <= 100) .and. mod(iter,10)==0) then
      ierr = visittimestepchanged()
      ierr = visitupdateplots()
      write (filename, "(A12,I4.4,A4)"),"pjacobi_img_",iter,".jpg"
      ierr=visitsavewindow(filename,20,800,800,VISIT_IMAGEFORMAT_JPEG)
     if(ierr.eq.VISIT_OKAY) then
        write (6,*) 'Saved ', filename
      endif
    end if

    ! VTK file each 10 time step until time step 100
    if ((iter <= 100) .and. mod(iter,10)==0) then
      ierr = visittimestepchanged()
      ierr = visitnamelistalloc(vars)
      ierr = visitnamelistaddname(vars, "default", 7)
      ierr = visitnamelistaddname(vars, "mesh2d/nodeid", 13)

      ierr = visitoptionlistalloc(options)
      ierr = visitoptionlistsetvalueb(options,"Strip mesh name prefix", 22, 1);

      write (filename, "(A8,I4.4)"), "pjacobi_",iter
      ierr=visitexportdatabasewithoptions(filename,12,"VTK_1.0",7,vars,options)
      if(ierr.eq.VISIT_OKAY) then
        write (6,*) 'Exported ',trim(filename),'.vtk'
      else
        write (6,*) 'Pb visitexportdatabasewithoptions'
      endif
      ierr = visitnamelistfree(vars)
      ierr = visitoptionlistfree(options)
    end if


#endif
    call simulate_one_timestep()
  end do
  call MPIIOWriteData('Jacobi.bin')
  deallocate (Temp, oldTemp)

  call MPI_Finalize(ierr)

end program Jacobi
 
