/*
 A parallel Jacobi solver for the Laplacian equation in 2D
 Written by Jean M. Favre, Swiss National Supercomputing Center
 September 24, 2011
 Code inspired from an older example by Kadin Tseng, Boston University, November 1999
 The compile flag -DUSE_CATALYST enables compilation with VisIt. Otherwise, the program runs
 in stand-alone mode
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <mpi.h>
#include "solvers.h"
#include "result_vtr.h"


#ifdef USE_CATALYST
// Le programme de simulation communique avec catalyst
// par l'intemediaire d'un "adaptor".
// Cet adaptateur est ecrit en C++ mais linke a la mamniere du C
// Il implemente trois fonctions: init, processing, finalize
#include "catalyst_adaptor.h"
#endif

int iter, par_rank, par_size, below, above;
int m, mp;
double gdel, *oldTemp, *Temp, *GradTemp;
float *cx, *cy;

int main(int argc, char *argv[])
{
  int i;
  int blocking, visitstate, err = 0;
  par_rank=0; par_size = 1;
  char *env = NULL;

  MPI_Init(&argc, &argv);                       /* starts MPI */
  MPI_Comm_rank(MPI_COMM_WORLD, &par_rank); /* get current process id */
  MPI_Comm_size(MPI_COMM_WORLD, &par_size); /* get # procs from env or */
  printf ("Hello from pjacobi.c - This is rank %i/%i\n",par_rank,par_size);
  neighbors();

#ifdef USE_CATALYST 
    const char* coproc_def_file = "coproc.py";
    char* coproc_file = NULL;
    if (argc == 2) {
      coproc_file = (char*) malloc(sizeof(char)*(strlen(argv[1]+1)));
      strcpy(coproc_file,argv[1]);
    } else {
      coproc_file = (char*) malloc(sizeof(char)*(strlen(coproc_def_file)+1));
      strcpy(coproc_file,coproc_def_file);
    }
    printf("Using python file %s\n",coproc_file);
    catalyst_init(coproc_file);
#endif                              

  m = 20; // mesh size = (m+2)x(m+2) including the bc grid lines

// We make no attempt to check that the number of grid points divides evenly
// with the number of MPI tasks.
// rank 0 will display the bottom (southern) boundary wall
// rank (size-1) will display the top (northern) boundary wall
// if run with m=20 and 4 MPI tasks, we will have 5 grid lines per rank
// and VisIt will display a 22x22 grid
  MPI_Bcast(&m, 1, MPI_INT, 0, MPI_COMM_WORLD);

// We use (m + 2) grid points in the X direction, i.e. interior points plus 2 b.c. points
  mp = m/par_size;
// We use (mp + 2) grid points in the Y direction, i.e. interior points plus 2 b.c. points

  oldTemp = (double *)malloc((mp + 2) * (m + 2) * sizeof(double));
  Temp    = (double *)malloc((mp + 2) * (m + 2) * sizeof(double));
// ATTENTION - Dans l'exemple Visit on avait GradTemp en 2D - Ici, catalyst impose le 3D !
  GradTemp= (double *)malloc(3 * (mp + 2) * (m + 2) * sizeof(double));

  cx = (float *)malloc(sizeof(float) * (m + 2));
  cy = (float *)malloc(sizeof(float) * (mp + 2));

  float hsize = 1.0/(m+1.0);

  for(i = 0; i < (m + 2); i++)
    cx[i] = i * hsize;
  for(i = 0; i < (mp + 2); i++)
    cy[i] = (i + par_rank*mp) * hsize;

  gdel = 1.0;
  iter = 0;
  set_initial_bc();

  while (gdel > TOL)
    {  // iterate until error below threshold or too many steps
    
    if(iter > MAXSTEPS)
      {
      if (par_rank==0) {
          fprintf(stdout,"Iteration terminated (exceeds %6d", MAXSTEPS);
          fprintf(stdout," )\n");
      }
      break;       /* nonconvergent solution */
      }
      simulate_one_timestep();

#ifdef USE_CATALYST 
    result_catalyst(Temp,GradTemp,par_rank,m,mp,iter);
    //call result_catalyst(Temp,m+1,mp+1,iter)

#endif

    }
  if (!par_rank)
    {
    fprintf(stdout,"Stopped at iteration %d\n",iter);
    fprintf(stdout,"The maximum error = %f\n",gdel);
    }

#ifdef USE_CATALYST
  catalyst_finalize();
#endif

  result_vtr (Temp,GradTemp,par_rank,m,mp,iter,par_size);

  MPIIOWriteData("Jacobi.bin");
  MPI_Barrier(MPI_COMM_WORLD);
  MPI_Finalize();

  free(oldTemp); free(Temp);
  free(cx); free(cy);
  return (0);
}

