/*
!> Coprocess result using CATALYST pipeline
!> Calls CXX functions from the Catalyst API
   This file implements the Catalyst Adaptor, as descripbed in the Catalyst User's Guide
   The Catalyst Adaptor is the bridge between the simulation code AND the catalyst libraries
   It is also the bridge between the simulation code written in C and the C++-based libraries (see the extern "C" statements)
   Code modified by Emmanuel Courcelle, CALMIP (emmanuel.courcelle@inp-toulouse.fr)

*/

#include <string>
using namespace std;

#include <stdio.h>

// Catalyst API headers
#include <vtkCPAdaptorAPI.h>
#include <CAdaptorAPI.h>
#include <CPythonAdaptorAPI.h>

#include <vtkCPPythonAdaptorAPI.h>
#include "vtkCPDataDescription.h"
#include "vtkCPInputDataDescription.h"
#include "vtkCPProcessor.h"
#include "vtkCPPythonScriptPipeline.h"

// VTK API headers
#include "vtkSmartPointer.h"
#include "vtkDoubleArray.h"
#include "vtkPointData.h"
#include "vtkStructuredGrid.h"
#include "vtkStructuredGrid.h"

// Ces fonctions sont utilisees uniquement a partir de ce module
// Pas la peine de prevoir une edition de lien de type C
void creategriddata(int nx, int ny, int nz, double* x, double* y);
void addscalarfield(double* scalars, const char* name);
void addvectorfield(double* uvw, const char* name);

// Ces fonctions sont appellees par le programme principal, ecrit en C
// Il est donc necessaire d'envisager une edition de liens de type "C"
//
extern "C" {

#include "catalyst_adaptor.h"

void catalyst_init(const char* coproc_file) {
	int fileNameLength = strlen(coproc_file);
	coprocessorinitializewithpython((char*)coproc_file, &fileNameLength);
}
void catalyst_finalize() {
	coprocessorfinalize();
}
	
void result_catalyst(double* Temp, int par_rank, int m, int mp, int iter) {

  //use jacobi_module
  //use iso_c_binding
  
  //real*8,  dimension(0:m+1) :: x
  //real*8,  dimension(0:mp+1) :: y  

  double* x = (double*) malloc(sizeof(double)*(m+2));
  double* y = (double*) malloc(sizeof(double)*(mp+2));

  //integer :: i,j
  //integer :: flag
  //integer :: nz=1
  //real*8 :: t
  //real*8 :: hsize

  int i,j;
  int flag;
  int nz=1;
  double t,hsize;
  t = 0.1*iter ;

  if (par_rank == 0) printf("CATALYST coprocessing at t=%5.3f\n", t);
  hsize = 1.0/(m+1);

  // do i=0,m+1
  //   x(i) = float(i) * hsize
  // end do
  // do i=0,mp+1
  //   y(i) = float(i+par_rank*mp) * hsize
  // end do

  for(i=0;i<m+2;++i) {
    x[i] = i * hsize;
  };
  for(i=0;i<mp+2;++i) {
    y[i] = (i+par_rank*mp) * hsize;
  };
  
//     if( par_rank == 0) print *, 'id ', par_rank, ' m =', m, ' mp=', mp*
/*
! all ranks send their block including their south boundary wall
! the northern-most rank sends also the northern boundary wall
 
  !----------------------------------------------------------------------------
  ! Request for coprocessing flag (flag=1 = coprocessing to be made)
  !----------------------------------------------------------------------------
*/
  requestdatadescription(&iter,&t,&flag);
  
  if (flag != 0) {
 
    /* 
    !--------------------------------------------------------------------------
    ! Check if the grid has already been built (flag=1 : grid not built yet)
    !--------------------------------------------------------------------------
    */
    needtocreategrid(&flag);
    if (flag != 0) {
      /*
      !------------------------------------------------------------------------
      ! Call grid creation callback (creategriddata_ in catalyst_callbacks.cxx)
      !------------------------------------------------------------------------
      */
      // il vaudrait mieux reecrire la fonctio griddata pour utiliser le passage de variables
      // par valeur et pas par adresse, mais en l'occurrence on a prefere utiliser la meme version que la 
      // version fortran (voir le repertoire F90)
      // Il faut donc passer pa une variable intermediaire
      int tmp_m = m+2;
      int tmp_mp = mp+2;
      // creategriddata(m+2,mp+2,nz,x[0],y[0]); // ! see catalyst_callbacks.cxx::creategriddata_ 
      //creategriddata(&tmp_m,&tmp_mp,&nz,x,y); // ! see catalyst_callbacks.cxx::creategriddata_ 
      creategriddata(m+2,mp+2,nz,x,y);
    }
   
    /* 
    !--------------------------------------------------------------------------
    ! Add temperature field
    !--------------------------------------------------------------------------
    */
    addscalarfield(Temp,"Temp");
    
    /*
    !--------------------------------------------------------------------------
    ! Do coprocessing
    !--------------------------------------------------------------------------
    */
    coprocess();

  }
}

} // Fin de bloc extern "C"


/** Create the grid
 */
void creategriddata(int nx, int ny, int nz, double* x, double* y)
{
  double dx1,dy1,dz1;
  double * xp;
  if (!vtkCPPythonAdaptorAPI::GetCoProcessorData())
    {
    vtkGenericWarningMacro("Unable to access CoProcessorData.");
    return;
    }

  vtkSmartPointer<vtkStructuredGrid> grid = vtkSmartPointer<vtkStructuredGrid>::New();
  vtkSmartPointer<vtkPoints> points = vtkSmartPointer<vtkPoints>::New();
  for (int k=0; k<(nz==0 ? 1 : nz); k++)
  {
    for (int j=0; j<ny; j++,y++)
    {
      xp = x;
      for (int i=0; i<nx; i++,xp++)
      {
        points->InsertNextPoint(*xp,*y,0.0);
      }
    }
  }
  grid->SetDimensions(nx,ny,nz);
  grid->SetPoints(points);
  vtkCPPythonAdaptorAPI::GetCoProcessorData()->GetInputDescriptionByName("input")->SetGrid(grid);
  vtkCPPythonAdaptorAPI::GetCoProcessorData()->GetInputDescriptionByName("input")->SetWholeExtent(0, nx-1, 0, ny-1, 0, nz-1);
}

// Add a scalar field to the data container.
void addscalarfield(double* scalars, const char* name)
{
  vtkCPInputDataDescription* idd =
    vtkCPPythonAdaptorAPI::GetCoProcessorData()->GetInputDescriptionByName("input");

  vtkStructuredGrid* grid = vtkStructuredGrid::SafeDownCast(idd->GetGrid());

  if (!grid)
    {
    vtkGenericWarningMacro("No adaptor grid to attach field data to.");
    return;
    }

  if (idd->IsFieldNeeded(name))
    {
    vtkSmartPointer<vtkDoubleArray> field = vtkSmartPointer<vtkDoubleArray>::New();
    field->SetNumberOfComponents(1);
    field->SetName(name);
    field->SetArray(scalars, grid->GetNumberOfPoints(), 1);
    grid->GetPointData()->AddArray(field);
    }
}


/** Add a vector field to the data
 */
void addvectorfield(double* uvw, const char* name)
{
  vtkCPInputDataDescription* idd =
    vtkCPPythonAdaptorAPI::GetCoProcessorData()->GetInputDescriptionByName("input");
  vtkStructuredGrid* grid = vtkStructuredGrid::SafeDownCast(idd->GetGrid());

  if (!grid)
    {
    vtkGenericWarningMacro("No adaptor grid to attach field data to.");
    return;
    }

  if (idd->IsFieldNeeded(name))
    {
    vtkSmartPointer<vtkDoubleArray> field = vtkSmartPointer<vtkDoubleArray>::New();
    field->SetNumberOfComponents(3);
    field->SetName(name);
    field->SetArray(uvw, 3 * grid->GetNumberOfPoints(), 1);
    grid->GetPointData()->AddArray(field);
    }
}


