/*
 * rr.c: Writes data randomly to a file.
 *
 * Loic Tortay, 2012.
 *
 */

#include <sys/stat.h>
#include <sys/time.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "errwarn.h"

const char	progname[] = "rw";
const off_t	default_alignment = 512;
const size_t	default_minbsize = 512;
const size_t	default_maxbsize = 32768;
const int	default_pattern = 'T';

void usage(FILE *);

void usage(FILE *fp)
{
	fprintf(fp,
"\nWrites data randomly to a file.\n"
"\nUsage:\n"
"%s [-b minbsize] [-B maxbsize] [-L length] [-O startoffset] [-P] [-S size] [-Z alignment] filename\n"
"\nWhere:\n"
" -b minbsize: set the minimum default write block size to minbsize.\n"
"    Default is: %lu bytes.\n"
" -B maxbsize: set the maximum default write block size to maxbsize.\n"
"    Default is: %lu bytes.\n"
" -L length: max offset to write from, relative to startoffset, default is\n"
"    (file size - offset).\n"
" -O startoffset: do I/Os in the file from that offset on, default is 0\n"
"    start of file.\n"
" -P Use pwrite instead of lseek/write\n"
" -S size: amount of data to write, default is 1/8 of the file size.\n"
" -W pattern: Write 'pattern' to file, pattern has to be a (decimal) integer \n"
"    between 1 and 255. Default is 84 ('T').\n"
" -Z alignment: align write block boundaries on alignment (bytes).\n"
"    For pure random writes use 1, 512 for sector alignment, 4096 for generic\n"
"    FS block alignment, etc.  Default is sector alignment.\n",
	    progname, (unsigned long) default_minbsize,
	    (unsigned long) default_maxbsize);

	exit(1);
}


int main(int argc, char *argv[])
{
	struct timeval	 tv;
	struct stat	 st;
	unsigned char	*buffer = NULL;
	char		*filename = NULL;
	off_t		 offset = 0, start_offset;
	off_t		 alignment = default_alignment;
	size_t		 towrite = 0, length = 0, bsize = 0;
	size_t		 minbsize = default_minbsize;
	size_t		 maxbsize = default_maxbsize;;
	ssize_t		 nw = 0;
	unsigned int	 seed = 0;
	int		 fd = 1, pattern;
	int		 ch = -1;
	long long int	 opt_size = 0, opt_maxbsize = 0, opt_minbsize = 0;
	long long int	 opt_offset = 0, opt_length = 0, opt_alignment = 0;
	int		 opt_pwrite = 0, opt_pattern = 0;

	while ((ch = getopt(argc, argv, ":b:B:L:O:PS:Z:")) != -1) {
		switch (ch) {
		case 'b':
			opt_minbsize = atoll(optarg);
			break;
		case 'B':
			opt_maxbsize = atoll(optarg);
			break;
		case 'L':
			opt_length = atoll(optarg);
			break;
		case 'O':
			opt_offset = atoll(optarg);
			break;
		case 'P':
			opt_pwrite = 1;
			break;
		case 'S':
			opt_size = atoll(optarg);
			break;
		case 'W':
			opt_pattern = atoi(optarg);
			break;
		case 'Z':
			opt_alignment = atoll(optarg);
			break;
		default:
			usage(stderr);
			break;
		}
	}
	if (optind != (argc - 1)) {
		warning(-1, "Filename required");
		usage(stderr);
	}
	filename = argv[optind];

	memset(&st, 0, sizeof(st));
	if (lstat(filename, &st) == -1)
		error(1, errno, "Unable to stat '%s'", argv[1]);

	if (!S_ISREG(st.st_mode))
		error(1, -1, "'%s' is not a regular file", argv[1]);

	if (opt_size) {
		if (opt_size > 0)
			towrite = (size_t) opt_size;
		else {
			close(fd);
			error(1, -1, "Invalid block size: %d", opt_size);
		}
	} else
		towrite = (size_t) st.st_size / 8;

	if (opt_maxbsize) {
		if (opt_maxbsize > 0)
			maxbsize = (size_t) opt_maxbsize;
		else {
			close(fd);
			error(1, -1, "Invalid max block size: %d", opt_maxbsize);
		}
	}

	if (opt_minbsize) {
		if (opt_minbsize > 0)
			minbsize = (size_t) opt_minbsize;
		else {
			close(fd);
			error(1, -1, "Invalid min block size: %d", opt_minbsize);
		}
	}

	if (opt_length) {
		if (opt_length > 0)
			length = (off_t) opt_length;
		else {
			close(fd);
			error(1, -1, "Invalid initial length: %d", opt_length);
		}
	} else
		length = (size_t) st.st_size;

	if (opt_offset) {
		if (opt_offset > 0)
			start_offset = (off_t) opt_offset;
		else {
			close(fd);
			error(1, -1, "Invalid initial offset: %d", opt_offset);
		}
	} else
		start_offset = (off_t) 0U;

	if (opt_alignment) {
		if (opt_alignment > 0)
			alignment = (off_t) opt_alignment;
		else {
			close(fd);
			error(1, -1, "Invalid alignmen: %d", opt_alignment);
		}
	} else
		alignment = default_alignment;

	if (opt_pattern) {
		if (opt_pattern > 0 && opt_pattern < 255)
			pattern = opt_pattern;
		else {
			close(fd);
			error(1, -1, "Invalid alignmen: %d", opt_pattern);
		}
	} else
		pattern = default_pattern;

	memset(&tv, 0, sizeof(tv));
	if (gettimeofday(&tv, NULL) == -1)
		error(1, errno, "Unable to get time of day");

	seed = (unsigned) tv.tv_usec;
	srand(seed);

	buffer = malloc(maxbsize);
	if (buffer == NULL)
		error(1, errno, "Unable to allocate memory for buffer (%lu "
		    "bytes)", (unsigned long) maxbsize);
	memset(buffer, pattern, maxbsize);

	fd = open(filename, O_RDWR);
	if (fd == -1)
		error(1, errno, "Unable to open '%s'", filename);

	offset = start_offset;
	printf("Will write %lu bytes from '%s', window: [%lu:%lu], minb: "
	    "%lu, maxb: %lu, alignment: %lu\n",
	    (unsigned long) towrite, filename, (unsigned long) offset,
	    (unsigned long) (offset + length), (unsigned long) minbsize,
	    (unsigned long) maxbsize, (unsigned long) alignment);
	if (maxbsize < minbsize) {
		printf("minbsize (%lu) > maxbsize (%lu), min <=> max\n",
		    (unsigned long) minbsize, (unsigned long) maxbsize);
		size_t m = minbsize; minbsize = maxbsize; maxbsize = m;
		if (realloc(buffer, maxbsize) == NULL)
			error(1, errno, "Failed to expand buffer to %lu bytes",
			    maxbsize);
	}
	while (towrite > 0) {
		if (towrite < maxbsize)
			maxbsize = towrite;

		if (maxbsize < minbsize)
			minbsize = maxbsize;

		bsize = minbsize + (size_t) ((double) (maxbsize - minbsize)
			* (double) rand() / (double) RAND_MAX);
		offset = start_offset + (off_t) ((double) length
			* (double) rand() / (double) RAND_MAX);

		if (offset > st.st_size)
			offset = st.st_size;

		offset -= offset % alignment;

		if (offset >= (off_t) bsize)
			offset -= (off_t) bsize;

		if (opt_pwrite)
			nw = pwrite(fd, buffer, bsize, offset);
		else {
			if (lseek(fd, offset, SEEK_SET) != offset)
				warning(errno, "Unable to seek to %lu in '%s'",
				    (unsigned long) offset, filename);

			nw = write(fd, buffer, bsize);
		}
		if (nw == -1)
			error(1, errno, "Error while writing '%s', offset: %lu",
			    filename, (unsigned long) offset);

		towrite -= (size_t) nw;
	}
	if (close(fd) == -1)
		warning(errno, "Problem closing '%s'", filename);

	free(buffer);

	return (0);
}

