/*
 * trivial-server: TCP server to be used with trivial-client (or netcat).
 * On client connection, send back the client hostname.
 *
 * Loic Tortay, 2012.
 *
 */

#include <sys/types.h>
#include <sys/socket.h>

#define __USE_MISC
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "errwarn.h"

const char progname[] = "trivial-server";


int main(int argc, char *argv[])
{
	char			 hostname[NI_MAXHOST];
	struct addrinfo		 aih;
	struct sockaddr_storage	 sas;
	struct addrinfo		 *aip = NULL;
	socklen_t		 saslen = 0;
	ssize_t			 nr = 0;
	size_t			 mlen = 0;
	int			 sfd = -1, cfd = -1;
	int			 reuse = 1, res = -1;

	if (argc != 3) {
		fprintf(stderr, "Usage:\n\t%s addr port\n", progname);
		exit (1);
	}

	memset(&aih, 0, sizeof(aih));
	/*
	 * AI_PASSIVE: return info usable w/ bind(),
	 * AI_NUMERICSERV: port number must be a number not a service name.
	 */
	aih.ai_flags = AI_PASSIVE | AI_NUMERICSERV;
	/* AF_UNSPEC: accept IPv4 & IPv6 */
	aih.ai_family = AF_UNSPEC;
	/* Connected byte streams (TCP) */
	aih.ai_socktype = SOCK_STREAM;

	res = getaddrinfo(argv[1], argv[2], &aih, &aip);
	if (res != 0)
		error(1, -1, "Unable to find addr(%s): %s (%d)", argv[1],
		    gai_strerror(res), res);

	printf("Opening socket on %s:%s\n", argv[1], argv[2]);
	sfd = socket(aip->ai_family, aip->ai_socktype, aip->ai_protocol);
	if (sfd == -1)
		error(1, errno, "Unable to open TCP socket");

	/* This is a server, allow port reuse */
	reuse = 1;
	res = setsockopt(sfd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse));
	if (res == -1)
		error(1, errno, "Unable to set socket option (REUSEADDR)");

	/* Avoid coalescing of small segments, send immediatly. */
	reuse = 1;
	res = setsockopt(sfd, IPPROTO_TCP, TCP_NODELAY, &reuse, sizeof(reuse));
	if (res == -1)
		error(1, errno, "Unable to set TCP option (NODELAY)");

	res = bind(sfd, aip->ai_addr, aip->ai_addrlen);
	if (res != 0)
		error(1, errno, "Unable to bind socket");
	printf("Bound socket\n");

	/* Start listening for clients */
	res = listen(sfd, SOMAXCONN);
	if (res != 0)
		error(1, errno, "Unable to liston on socket");

	while (1) {
		memset(&sas, 0, sizeof(sas));
		saslen = (socklen_t) sizeof(sas);
		/* Wait for clients */
		printf("Waiting for connections\n");
		cfd = accept(sfd, (struct sockaddr *) &sas, &saslen);
		if (cfd == -1)
			warning(errno, "Failed to accept connection");

		printf("Connection received\n");

		/* Check client info */
		if (sas.ss_family != AF_INET && sas.ss_family != AF_INET6) {
			warning(-1, "Unknown socket addr family: %d",
			    sas.ss_family);
			if (close(cfd) != 0)
				warning(errno, "Problem detected while closing "
				    "client connection");
		}

		/* Get client name */
		res = getnameinfo((struct sockaddr *) &sas, saslen, hostname,
			sizeof(hostname), NULL, 0, NI_NUMERICSERV);
		if (res != 0)
			warning(-1, "Unable to get client name: %s (%d)",
			    gai_strerror(res), res);

		printf("Client is: %s\n", hostname);

		/* Send the client its own name */
		mlen = sizeof(hostname);
		nr = send(cfd, hostname, mlen, 0);
		if (nr != (ssize_t) mlen)
			warning(errno, "Failed to send \"%s\" to '%s'", hostname,
			    hostname);

		printf("Sent '%s' to \"%s\"\n", hostname, hostname);

		/* Bye. */
		if (close(cfd) != 0)
			warning(errno, "Problem detected while closing client "
			    "connection");
	}

	freeaddrinfo(aip);

	return (0);
}

