/*
 * file-server: Simple single file transfer server.  Send the content of a
 * file to clients on connection.  To be used with file-client or netcat.
 * The transfer is done using send()/recv() POSIX functions.
 *
 * Loic Tortay, 2012.
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>

#define __USE_MISC
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "errwarn.h"

const char	progname[] = "file-server";
const size_t	default_bsize = 1048576U;
const int	round_mark = 200;

#ifndef HAS_TIMERSUB
#define timersub(tvp, uvp, vvp)						\
	do {								\
		(vvp)->tv_sec = (tvp)->tv_sec - (uvp)->tv_sec;		\
		(vvp)->tv_usec = (tvp)->tv_usec - (uvp)->tv_usec;	\
		if ((vvp)->tv_usec < 0) {				\
			(vvp)->tv_sec--;				\
			(vvp)->tv_usec += 1000000;			\
		}							\
	} while (0)
#endif


int main(int argc, char *argv[])
{
	char			 hostname[NI_MAXHOST];
	struct addrinfo		 aih;
	struct sockaddr_storage	 sas;
	struct stat		 st;
	struct timeval		 starttime, endtime, runtime;
	struct addrinfo		*aip = NULL;
	unsigned char		*buffer = NULL;
	char			*filename;
	double			 druntime = 0.0;
	socklen_t		 saslen = 0;
	off_t			 bytesread = 0;
	size_t			 toread = 0, tosend = 0, bytessent = 0;
	size_t			 bsize = default_bsize;
	ssize_t			 nr = 0;
	int			 sfd = -1, cfd = -1, fd = -1;
	int			 reuse = 1, res = -1, round = 0;

	if (argc != 4) {
		fprintf(stderr, "Usage:\n\t%s addr port filename\n",
		    progname);
		exit (1);
	}

	filename = argv[3];
	fd = open(filename, O_RDONLY, 0);
	if (fd == -1)
		error(1, errno, "Unable to open file '%s'", filename);

	setvbuf(stdout, NULL, _IONBF, 0);

	memset(&st, 0, sizeof(st));
	if (fstat(fd, &st) == -1)
		error(1, errno, "Unable to stat '%s'", filename);

	/* Make sure target is a file */
	if (!S_ISREG(st.st_mode))
		error(1, -1, "'%s' is not a regular file", filename);

	if (fd == -1)
		error(1, errno, "Unable to open file '%s'", filename);

	buffer = malloc(bsize);
	if (buffer == NULL)
		error(1, errno, "Unable to allocate memory for buffer (%lu "
		    "bytes)", (unsigned long) bsize);

	memset(&aih, 0, sizeof(aih));
	/*
	 * AI_PASSIVE: return info usable w/ bind(),
	 * AI_NUMERICSERV: port number must be a number not a service name.
	 */
	aih.ai_flags = AI_PASSIVE | AI_NUMERICSERV;
	/* AF_UNSPEC: accept IPv4 & IPv6 */
	aih.ai_family = AF_UNSPEC;
	/* Connected byte streams (TCP) */
	aih.ai_socktype = SOCK_STREAM;

	res = getaddrinfo(argv[1], argv[2], &aih, &aip);
	if (res != 0)
		error(1, -1, "Unable to find addr(%s): %s (%d)", argv[1],
		    gai_strerror(res), res);

	printf("Opening socket on %s:%s\n", argv[1], argv[2]);
	sfd = socket(aip->ai_family, aip->ai_socktype, aip->ai_protocol);
	if (sfd == -1)
		error(1, errno, "Unable to open TCP socket");

	/* This is a server, allow port reuse */
	reuse = 1;
	res = setsockopt(sfd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse));
	if (res == -1)
		error(1, errno, "Unable to set socket option (REUSEADDR)");

	/* Avoid coalescing of small segments, send immediatly. */
	reuse = 1;
	res = setsockopt(sfd, IPPROTO_TCP, TCP_NODELAY, &reuse, sizeof(reuse));
	if (res == -1)
		error(1, errno, "Unable to set TCP option (NODELAY)");

	res = bind(sfd, aip->ai_addr, aip->ai_addrlen);
	if (res != 0)
		error(1, errno, "Unable to bind socket");

	/* Start listening for clients */
	res = listen(sfd, SOMAXCONN);
	if (res != 0)
		error(1, errno, "Unable to liston on socket");

	signal(SIGPIPE, SIG_IGN);

	while (1) {
		memset(&sas, 0, sizeof(sas));
		saslen = (socklen_t) sizeof(sas);
		/* Wait for clients */
		printf("Waiting for connections\n");
		cfd = accept(sfd, (struct sockaddr *) &sas, &saslen);
		if (cfd == -1)
			warning(errno, "Failed to accept connection");

		printf("Client connection received\n");

		/* Check client info */
		if (sas.ss_family != AF_INET && sas.ss_family != AF_INET6) {
			warning(-1, "Unknown socket addr family: %d",
			    sas.ss_family);
			if (close(cfd) != 0)
				warning(errno, "Problem detected while closing "
				    "client connection");
		}

		/* Get client name */
		res = getnameinfo((struct sockaddr *) &sas, saslen, hostname,
			sizeof(hostname), NULL, 0, NI_NOFQDN);
		if (res != 0)
			warning(-1, "Unable to get client name: %s (%d)",
			    gai_strerror(res), res);

		printf("Sending '%s' to \"%s\": ", filename, hostname);

		/* Prepare to send the file to the client */
		toread = (size_t) st.st_size;
		bytesread = 0;
		bytessent = 0;
		bsize = default_bsize;
		round = 0;
		/* Clear timers */
		memset(&starttime, 0, sizeof(starttime));
		memset(&endtime, 0, sizeof(endtime));
		memset(&runtime, 0, sizeof(runtime));
		/* Get start time */
		if (gettimeofday(&starttime, NULL))
			error(1, errno, "Unable to get start time");

		while (toread > 0) {
			if (toread < bsize)
				bsize = toread;

			nr = pread(fd, buffer, bsize, bytesread);
			if (nr == -1) {
				warning(errno, "Error while reading '%s', "
				    "offset: %lu", filename,
				    (unsigned long) bytesread);
				break;
			}
			toread -= (size_t) nr;
			bytesread += (off_t) nr;

			tosend = (size_t) nr;
			while (tosend > 0) {
				nr = send(cfd, buffer, tosend, 0);
				if (nr == -1) {
					warning(errno, "Error while sending "
					    "'%s' (offset: %lu) to \"%s\"",
					    filename, (unsigned long) bytessent,
					    hostname);
					break;
				}
				tosend -= (size_t) nr;
				bytessent += (size_t) nr;
			}
			if (nr == -1)
				break;
			if (round % round_mark == 0)
				putc('.', stdout);
			round++;
		}
		if (gettimeofday(&endtime, NULL))
			error(1, errno, "Unable to get end time");
		if (nr != -1)
			putc('\n', stdout);
		/* Get end time */
		timersub(&endtime, &starttime, &runtime);
		druntime = (runtime.tv_sec * 1000000.0) + runtime.tv_usec;
		druntime /= 1000000.0;
		printf("Sent %lu bytes from '%s' to \"%s\" in %.3lf seconds "
		    "(%.4lf MiB/s)\n", (unsigned long) bytessent, filename,
		    hostname, druntime,
		    (double) bytessent / (druntime * 1048576.0));

		/* Say bye to this client */
		if (shutdown(cfd, SHUT_RDWR) != 0)
			warning(errno, "Problem while shutting down client "
			    "connection");

		if (close(cfd) != 0)
			warning(errno, "Problem while closing client "
			    "connection");
	}

	if (close(fd) != 0)
		warning(errno, "Problem while closing '%s'");

	if (shutdown(sfd, SHUT_RDWR) != 0)
		warning(errno, "Problem while shutting down socket");

	if (close(sfd) != 0)
		warning(errno, "Problem while closing server socket");

	signal(SIGPIPE, SIG_DFL);

	free(buffer);
	freeaddrinfo(aip);

	return (0);
}

