/*
 * file-client: Simple TCP file transfer client.  Saves to a file the byte
 * stream sent by the server (file-server, sendfile-server or netcat).
 *
 * Loic Tortay, 2012.
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>

#define __USE_MISC
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "errwarn.h"

const char	progname[] = "file-client";
const size_t	default_bsize = 1048576U;
const int	round_mark = 200;

#ifndef HAS_TIMERSUB
#define timersub(tvp, uvp, vvp)						\
	do {								\
		(vvp)->tv_sec = (tvp)->tv_sec - (uvp)->tv_sec;		\
		(vvp)->tv_usec = (tvp)->tv_usec - (uvp)->tv_usec;	\
		if ((vvp)->tv_usec < 0) {				\
			(vvp)->tv_sec--;				\
			(vvp)->tv_usec += 1000000;			\
		}							\
	} while (0)
#endif


int main(int argc, char *argv[])
{
	struct addrinfo		 aih;
	struct timeval		 starttime, endtime, runtime;
	struct addrinfo		*aip = NULL;
	unsigned char		*buffer = NULL;
	char			*filename, *server, *port;
	double			 druntime = 0.0;
	size_t			 bsize = default_bsize, breceived = 0;
	size_t			 towrite = 0;
	off_t			 bwritten = 0, start_offset = 0;
	ssize_t			 nr = 0, nw = 0;
	int			 cfd = -1, fd = -1;
	int			 reuse = 1, res = -1, round = 0;

	if (argc != 4 && argc != 5) {
		fprintf(stderr, "Usage:\n\t%s addr port outfilename [offset]\n",
		    progname);
		exit (1);
	}

	server = argv[1];
	port = argv[2];
	filename = argv[3];
	if (argc == 5)
		start_offset = (off_t) atoll(argv[4]);

	fd = open(filename, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);

	if (fd == -1)
		error(1, errno, "Unable to create '%s'", filename);

	setvbuf(stdout, NULL, _IONBF, 0);

	buffer = malloc(bsize);
	if (buffer == NULL)
		error(1, errno, "Unable to allocate memory for buffer (%lu "
		    "bytes)", (unsigned long) bsize);

	memset(&aih, 0, sizeof(aih));
	/*
	 * AI_NUMERICSERV: port number must be a number not a service name.
	 */
	aih.ai_flags = AI_NUMERICSERV;
	/* AF_UNSPEC: accept IPv4 & IPv6 */
	aih.ai_family = AF_UNSPEC;
	/* Connected byte streams (TCP) */
	aih.ai_socktype = SOCK_STREAM;

	res = getaddrinfo(server, port, &aih, &aip);
	if (res != 0)
		error(1, -1, "Unable to find addr(%s): %s (%d)", server,
		    gai_strerror(res), res);

	printf("Opening socket to %s:%s\n", server, port);
	cfd = socket(aip->ai_family, aip->ai_socktype, aip->ai_protocol);
	if (cfd == -1)
		error(1, errno, "Unable to open TCP socket");

	/* Avoid coalescing of small segments, send immediatly. */
	reuse = 1;
	res = setsockopt(cfd, IPPROTO_TCP, TCP_NODELAY, &reuse, sizeof(reuse));
	if (res == -1)
		error(1, errno, "Unable to set TCP option (NODELAY)");

	res = connect(cfd, aip->ai_addr, aip->ai_addrlen);
	if (res == -1)
		error(1, errno, "Unable to connect to '%s' on port %s", server,
		    port);

	printf("Writing from \"%s\" to '%s': ", server, filename);

	bsize = default_bsize;
	breceived = 0;
	round = 0;
	nw = 0;
	/* Clear timers & get start time */
	memset(&starttime, 0, sizeof(starttime));
	memset(&endtime, 0, sizeof(endtime));
	memset(&runtime, 0, sizeof(runtime));
	if (gettimeofday(&starttime, NULL))
		error(1, errno, "Unable to get start time");

	if (start_offset != 0)
		bwritten = start_offset;

	do {
		nr = recv(cfd, buffer, bsize, 0);
		if (nr == -1) {
			warning(errno, "Error receiving data");
			break;
		}
		breceived += (size_t) nr;

		towrite = (size_t) nr;
		while (towrite > 0) {
			nw = pwrite(fd, buffer, towrite, bwritten);
			if (nw < 1) {
				warning(errno, "Error writing output '%s', "
				    "offset: %lu", filename,
				    (unsigned long) bwritten);
				break;
			}
			bwritten += (off_t) nw;
			towrite -= (size_t) nw;
		}
		if (nw == -1)
			break;
		if (round % round_mark == 0)
			putc('.', stdout);
		round++;
	} while (nr > 0);

	/* Get end time */
	if (gettimeofday(&endtime, NULL))
		error(1, errno, "Unable to get end time");

	if (nw != -1)
		putc('\n', stdout);

	if (start_offset != 0)
		bwritten -= start_offset;

	if (breceived != (size_t) bwritten)
		warning(-1, "Inconsistent numbers of bytes received & written "
		    "(r: %lu/w: %lu)", breceived, bwritten);

	timersub(&endtime, &starttime, &runtime);
	druntime = (runtime.tv_sec * 1000000.0) + runtime.tv_usec;
	druntime /= 1000000.0;
	printf("Wrote %lu bytes from \"%s\" to '%s' in %.3lf seconds (%.4lf "
	    "MiB/s)\n", (unsigned long) bwritten, server, filename, druntime,
	    (double) bwritten / (druntime * 1048576.0));

	res = shutdown(cfd, SHUT_RDWR);
	if (res == -1)
		warning(errno, "Error during socket shutdown");

	res = close(cfd);
	if (res == -1)
		warning(errno, "Error closing client socket");

	res = close(fd);
	if (res == -1)
		warning(errno, "Error closing output file");

	free(buffer);
	freeaddrinfo(aip);

	return (0);
}

