/*
 * Simple "lockf()" & "fcntl()" locking example.
 */

#define _XOPEN_SOURCE 500 /* Needed for pwrite() on Linux */

#include <sys/stat.h>
#include <sys/types.h>

#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <limits.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

void error(const int rc, const char *msg, ...);


#define LOCK_HOLD_TIME		20U
#define LOCK_WAIT_TIME		1U
#define LOCK_WAIT_TIME_MAX	15U


int lockfile(int, const char *, off_t, size_t, unsigned int);
int unlockfile(int, const char*, off_t, size_t);
int64_t strtonum(const char *, int64_t, int64_t);


int
main(int argc, char *argv[])
{
	char	*buffer;
	size_t   size;
	ssize_t  we;
	off_t    offset;
	int	 fd = -1, le;

	if (argc != 5)
		error(1, "Usage: locker file pattern offset size\n");

	if ((fd = open(argv[1], O_RDWR | O_CREAT, S_IRUSR | S_IWUSR)) == -1)
		error(2, "Unable to open '%s': %s (%d)", argv[1],
		    strerror(errno), errno);

	offset = (off_t) strtonum(argv[3], 0, INT_MAX);

	size = (size_t) strtonum(argv[4], 0, INT_MAX);

	if ((buffer = malloc(size)) == NULL)
		error(2, "Unable to allocate memory for buffer: %s (%d)",
		    strerror(errno), errno);

	memset(buffer, (int) argv[2][0], size);

	if ((le = lockfile(fd, argv[1], offset, size, LOCK_WAIT_TIME_MAX)) != 0)
		error(2, "Unable to lock '%s': %s (%d)", argv[1], strerror(le),
		    le);
	else
		printf("locked '%s', will hold the lock for %u seconds\n",
		    argv[1], LOCK_HOLD_TIME);

	if ((we = pwrite(fd, buffer, size, offset)) != (ssize_t) size)
		error(2, "Problem writing to '%s': %s (%d, %d)", strerror(errno),
		    errno, (int) we);

	if (sleep(LOCK_HOLD_TIME))
		error(2, "Unable to sleep for %u seconds: %s (%d)",
		    LOCK_HOLD_TIME, strerror(errno), errno);

	if ((le = unlockfile(fd, argv[1], offset, size)) != 0)
		error(2, "Unable to unlock '%s': %s (%d)", argv[1], strerror(le),
		    le);
	else
		printf("unlocked '%s'\n", argv[1]);

	if (close(fd) == -1)
		error(2, "Error closing '%s': %s (%d)", argv[1], strerror(errno),
		    errno);

	return (0);
}


int
lockfile(int fd, const char *name, off_t offset, size_t length,
	unsigned int maxwait)
{
#ifdef USE_FCNTL
	struct flock	fl;
#endif /* USE_FCNTL */
	unsigned int	waittime = 0U;
	int		rc = 0;

#ifdef USE_FCNTL
	memset(&fl, 0, sizeof(fl));
	fl.l_type = F_WRLCK;
	fl.l_whence = SEEK_SET;
	fl.l_start = offset;
	fl.l_len = (off_t) length;
#else /* USE_FCNTL */
	if (offset != 0 && lseek(fd, offset, SEEK_SET) == -1) {
		fprintf(stderr, "Unable to seek to %" PRId64 " into '%s': "
		    "%s (%d)\n", (int64_t) offset, name, strerror(errno),
		    errno);
		rc = errno;
	}
#endif /* USE_FCNTL */

#ifdef USE_FCNTL
	while (rc == 0 && fcntl(fd, F_SETLK, &fl) == -1) {
#else /* USE_FCNTL */
	while (rc == 0 && lockf(fd, F_TLOCK, (off_t) length) == -1) {
#endif /* USE_FCNTL */
		if (errno == EAGAIN && maxwait != 0U) {
			fprintf(stderr, "'%s' is already locked, waiting for "
			    "lock availability (%u/%u)\n", name, waittime,
			    maxwait);

			if (sleep(LOCK_WAIT_TIME))
				error(2, "Unable to sleep for %u seconds: %s"
				    " (%d)", LOCK_WAIT_TIME, strerror(errno),
				    errno);

			waittime += LOCK_WAIT_TIME;
			if (waittime >= maxwait) {
				fprintf(stderr, "Waited too long for a lock on "
				    "'%s': %u seconds\n", name, waittime);
				rc = ENOLCK;
			}
		} else
			rc = errno;
	}

	return (rc);
}


int
unlockfile(int fd, const char *name, off_t offset, size_t length)
{
#ifdef USE_FCNTL
	struct flock	fl;
#endif /* USE_FCNTL */
	int		rc = 0;

#ifndef USE_FCNTL
	if (offset != 0 && lseek(fd, offset, SEEK_SET) == -1) {
		fprintf(stderr, "Unable to seek to %" PRId64 " into '%s': "
		    "%s (%d)\n", (int64_t) offset, name, strerror(errno),
		    errno);
		rc = errno;
	}
#else /* !USE_FCNTL */
	memset(&fl, 0, sizeof(fl));
	fl.l_type = F_UNLCK;
	fl.l_whence = SEEK_SET;
	fl.l_start = offset;
	fl.l_len = (off_t) length;
#endif /* !USE_FCNTL */

#ifdef USE_FCNTL
	if (rc == 0 && fcntl(fd, F_SETLK, &fl) == -1)
#else /* !USE_FCNTL */
	if (rc == 0 && lockf(fd, F_ULOCK, (off_t) length) == -1)
#endif /* !USE_FCNTL */
		rc = errno;

	return (rc);
}


/***************************************************************************/

/*
 * strtonum:
 *	Convert human-provided number in 's' into a usable value.
 *
 * Input:
 *	s: NUL terminated string of character represnting the number,
 *	min: lower bound for returned value,
 *	max: upper bound for returned value,
 *
 * Result:
 *	value of 's' converted to a 64-bit signed int.
 *
 */
int64_t
strtonum(const char* s, int64_t min, int64_t max)
{
	double		 dresult = 0.0;
	int64_t		 result = 0LL;
	char		*e = NULL;

	errno = 0;
	dresult = strtod(s, &e);
	if ((dresult == 0.0 && errno != 0) || (e == s) || (e == NULL))
		error(4, "Error while converting '%.100s' to number: %s (%d)",
		    s, strerror(errno), errno);

	result = (int64_t) llrint(dresult);

	if (result < min)
		error(4, "The value '%.100s' is out of range (<)", s);

	if (result > max)
		error(4, "The value '%.100s' is out of range (>)", s);

	return (result);
}

/***************************************************************************/

void error(const int rc, const char *msg, ...)
{
	va_list ap;

	va_start(ap, msg);
	vfprintf(stderr, msg, ap);
	va_end(ap);
	fprintf(stderr, "\n");

	exit(rc);
}

