/*
 * hfile.c: reads a file and computes the hash of the file's data using OpenSSL.
 *
 * Loic Tortay, 2012.
 */

#include <sys/stat.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <openssl/evp.h>

#include "errwarn.h"

const char	progname[] = "hfile";
const char	hash[] = "sha256";
const size_t	default_bsize = 1048576U;


int main(int argc, char *argv[])
{
	unsigned char	 digest[EVP_MAX_MD_SIZE];
	char	 	 hrdigest[EVP_MAX_MD_SIZE * 2 + 1];
	struct stat	 st;
	EVP_MD_CTX	 dgst_ctx;
	EVP_MD		*digester = NULL;
	unsigned char	*buffer = NULL;
	char		*filename = NULL;
	size_t		 toread = 0, bytesread = 0;
	size_t		 bsize = default_bsize, slen = 0;
	ssize_t		 nr = 0;
	unsigned int	 dgst_len = 0, h;
	int		 fd = 1, i = 0;

	if (argc < 2)
		error(1, -1, "Filename required");

	OpenSSL_add_all_digests();
	digester = (EVP_MD*) EVP_get_digestbyname(hash);
	if (digester == NULL)
		error(1, -1, "OpenSSL does not seem to support '%s'", hash);

	for (i = 1; i < argc; i++) {
		filename = argv[i];
		fd = open(filename, O_RDONLY);
		if (fd == -1)
			error(1, errno, "Unable to open '%s'", filename);

		memset(&st, 0, sizeof(st));
		if (fstat(fd, &st) == -1)
			error(1, errno, "Unable to stat '%s'", filename);

		if (!S_ISREG(st.st_mode))
			error(1, -1, "'%s' is not a regular file", filename);

		toread = st.st_size;

		buffer = malloc(bsize);
		if (buffer == NULL)
			error(1, errno, "Unable to allocate memory for buffer");

		EVP_MD_CTX_init(&dgst_ctx);
		EVP_DigestInit_ex(&dgst_ctx, (const EVP_MD*) digester, NULL);
		while (toread > 0) {
			if (toread < bsize)
				bsize = toread;

			nr = read(fd, buffer, bsize);
			if (nr == -1)
				error(1, errno, "Error while reading '%s' at "
				    "offset %lu", filename, bytesread);

			bytesread += nr;
			toread -= nr;

			EVP_DigestUpdate(&dgst_ctx, buffer, nr);
		}
		EVP_DigestFinal_ex(&dgst_ctx, digest, &dgst_len);

		slen = dgst_len * 2;
		for (h = 0; h < dgst_len; h++)
			sprintf(&hrdigest[h * 2], "%02x", digest[h]);
		hrdigest[slen] = '\0';
		EVP_MD_CTX_cleanup(&dgst_ctx);

		if (close(fd))
			warning(errno, "Problem closing '%s'", filename);

		free(buffer);

		printf("%s(%s): %s\n", hash, filename, hrdigest);
	}

	return (0);
}

