      program laplace
!
!$    USE OMP_LIB

      IMPLICIT NONE
!
      Integer,  Parameter  ::  rp = Kind (1.0d0)
!
      Real(rp),  Allocatable, Dimension(:)  ::  xm
      Real(rp),  Allocatable, Dimension(:)  ::  ym
!
      Real(rp),  Allocatable, Dimension(:)  ::  x
      Real(rp),  Allocatable, Dimension(:)  ::  y
!
      Real(rp),  Allocatable, Dimension(:,:)  ::  V, V_old
      Real(rp),  Allocatable, Dimension(:,:)  ::  eps
      Real(rp),  Allocatable, Dimension(:,:)  ::  vw, vs, vn, ve
      Real(rp),  Allocatable, Dimension(:,:)  ::  vc, sou
!
!$    Integer,   Dimension(0:64,-1:11)  ::  itab_synchro
!
      Real(rp)  ::  distx, disty, dxx, dyy, vv
!$    Real(rp)  ::  t1, t2
      Real(rp)  ::  eps0, epsR, uc, ua, erreur_tol
      Real(rp)  ::  omega, omega1,  erreur, erreur_max
!
      Integer   ::  nx, ny
      Integer   ::  i, j, k, iter_max, ierr
      Integer   ::  idielg, idield, nxdg, nxdd
!
      Integer   ::  myrank, nbthd
      Integer   ::  imin, imin1
      Integer   ::  imax, imax1
!
      READ (5,*) nx, ny, iter_max, erreur_tol
!     nx=1920
!     ny=1920
!     iter_max=5000
!     erreur_tol=1.0d-5

!
!Allocation des tableaux
      ALLOCATE (xm(0:nx), ym(0:ny), x(nx), y(ny), V(0:nx+1,0:ny+1), V_old(0:nx+1,0:ny+1),
     &  eps(0:nx+1,0:ny+1), vn(nx,ny), ve(nx,ny), vs(nx,ny), vw(nx,ny), vc(nx,ny),
     &  sou(nx,ny), STAT=ierr)
      IF (ierr /= 0) THEN
         Write (6,'(A)') 'ERREUR ALLOCATION DYNAMIQUE'
         STOP
      END IF
!
!$OMP PARALLEL DEFAULT(NONE) 
!$OMP&SHARED (xm, ym, x, y, V, V_old, eps, vn, ve, vs, vw, vc, sou)   
!$OMP&SHARED (erreur_max, erreur_tol, nx, ny, itab_synchro, iter_max) 
!$OMP&PRIVATE (i, j, k, distx, disty, dxx, dyy, eps0, epsR)           
!$OMP&PRIVATE (uc, ua, erreur, myrank, nbthd, omega1, vv, t1, t2)     
!$OMP&PRIVATE (omega, idielg, idield, nxdg, nxdd, imin, imax, imin1, imax1) 

      myrank = 0
      nbthd  = 1
!$    myrank  = OMP_GET_THREAD_NUM()
!$    nbthd = OMP_GET_NUM_THREADS()

! Initialisation OpenMP
!$OMP SINGLE
!$    IF (.TRUE.) THEN
!$       WRITE (6,'(A)')    'Execution en parallele OpenMP'
!$       WRITE (6,'(A,I2)') 'Nombre de threads OpenMP : ', nbthd
!$    ELSE
         WRITE (6,'(A)') 'Execution en sequentiel'
!$    END IF
!$OMP END SINGLE NOWAIT

!------------------------------------                    
! discretisaton du domaine de calcul
!------------------------------------    
      imin = 1  +(myrank*nx) / nbthd
      imax = ((myrank+1)*nx) / nbthd

      if (imin == 1) then
         imin1 = 0
      else
         imin1 = imin
      end if
      if (imax == nx) then
         imax1 = nx+1
      else
         imax1 = imax
      end if

      distx = 1.0d0
      disty = 1.0d0
!    
      dxx = distx / Real(nx,rp)
      dyy = disty / Real(ny,rp)
!$OMP SINGLE
      erreur_max = 0.0d0
      xm(0) = 0.0d0
      do i = 1, nx
         xm(i) = xm(i-1) + dxx
         x (i) = (xm(i) + xm(i-1) ) * 0.5d0
      end do
!$OMP END SINGLE NOWAIT
!
! initialisation du drapeau sur les colonnes
!$OMP SINGLE
!$      itab_synchro(1,-1) = ny+1
!$      DO j = 0, nbthd-1
!$         itab_synchro(1,j) = 0
!$      END DO
!$OMP END SINGLE NOWAIT
!
!$OMP SINGLE
      ym(0) = 0.0d0
      do j = 1, ny
         ym(j) = ym(j-1) + dyy
         y(j) = (ym(j) + ym(j-1) ) * 0.5d0
      end do
!$OMP END SINGLE

!------------------------------------------------        
! initialisation des parametres et des variables
!------------------------------------------------
      eps0 = 8.85d-14  !permittivite (F/cm)   
!
      do j = 0, ny+1
         do i = imin1, imax1
            V  (i,j) = 0.0d0
            eps(i,j) = eps0
         end do
      end do

! choix des valeurs des potentiels fixes a gauche (uc) et a droite (ua)
      uc = 0.0d0
      ua = 1.0d0

! prise en compte des dielectriques 1 = oui ou 0 = non
      idielg = 1
      idield = 1

      nxdg = 11   ! epaisseur a gauche nxdg-1 mailles
      nxdd = 90   ! epaisseur a droite nx-nxdd mailles
      epsR = 5.0d0
!
      if (idielg == 1) then
!$OMP SINGLE
         write (6,*) 'dielectrique a gauche jusqu a ', xm(nxdg-1)
!$OMP END SINGLE
!$OMP DO
         do j = 1, ny
            do i = 1, nxdg-1
                eps(i,j) = eps0 * epsR
            end do
         end do
!$OMP END DO NOWAIT
      end if
!
      if (idield == 1) then
!$OMP SINGLE
         write(6,*)'dielectrique a droite depuis ', xm(nxdd)
!$OMP END SINGLE
!$OMP DO
         do j = 1, ny
            do i = nxdd+1, nx
                eps(i,j) = eps0 * epsR
            end do
         end do
!$OMP END DO NOWAIT
      end if

!-------------------------------------
! Resolution de l'equation de Laplace
!-------------------------------------
 
! choix du critere de convergence
!      erreur_tol = 1.0d-5

! choix de la methode de resolution
! omega = 1 : Gauss-Seidel
! 1<omega<2 : Sur-relaxation
      omega  = 1.5d0
      omega1 = 1.0d0 - omega

! Definition des coefficients de l'equation de Laplace
! cas general
      do j = 1, ny
         do i = imin, imax
            ve(i,j) = 2.0d0 * eps(i,j) * eps(i+1,j  ) / ( (eps(i,j) + eps(i+1,j) ) * dxx**2)
            vn(i,j) = 2.0d0 * eps(i,j) * eps(i  ,j+1) / ( (eps(i,j) + eps(i,j+1) ) * dyy**2)
            vs(i,j) = 2.0d0 * eps(i,j) * eps(i  ,j-1) / ( (eps(i,j) + eps(i,j-1) ) * dyy**2)
            vw(i,j) = 2.0d0 * eps(i,j) * eps(i-1,j  ) / ( (eps(i,j) + eps(i-1,j) ) * dxx**2)
            vc(i,j) = omega / ( vw(i,j) + vn(i,j) + vs(i,j) + ve(i,j) )
            sou(i,j) = 0.0d0
         end do
      end do

! Conditions aux limites de Neumann en j = 1 et j = ny
      j = 1
      do i = imin, imax
         vs(i,j) = 0.0d0
         vc(i,j) = omega / ( vw(i,j) + vn(i,j)           + ve(i,j) )
      end do
      j = ny
      do i = imin, imax
         vn(i,j) = 0.0d0
         vc(i,j) = omega / ( vw(i,j)           + vs(i,j) + ve(i,j) )
      end do

! Conditions aux limites de Dirichlet en i = 1 et i = nx
!$    if (myrank == 0) then
         i = 1
         do j = 1, ny
            vv       = 2.0d0*eps(i,j) / (dxx)**2
            vc (i,j) = omega / ( vv      + vn(i,j) + vs(i,j) + ve(i,j) )
            sou(i,j) = vv      * uc
            vw (i,j) = 0.0d0
         end do
!$    end if
!$    if (myrank == nbthd-1) then
         i = nx
         do j = 1, ny
            vv       = 2.0d0 * eps(i,j) / (dxx)**2
            vc (i,j) = omega / ( vw(i,j) + vn(i,j) + vs(i,j) + vv      )
            sou(i,j) = vv      * ua
            ve (i,j) = 0.0d0
         end do
!$    end if

!$OMP BARRIER

!$    t1 = OMP_Get_WTime ()
!
! resolution sur k iterations
      do k = 1, iter_max

!$OMP SINGLE
         if ( MOD(k, 100) == 0) then
            write (6,*) 'k = ', k, ' erreur = ', erreur_max
         end if
         erreur_max = 0.0d0
!$OMP FLUSH (erreur_max)
!$OMP END SINGLE

! Resolution de l'equation de Laplace
         do j = 1, ny
! attente sur le drapeau sur les colonnes
!$          synchro: do
!$OMP FLUSH(itab_synchro)
!$                      if ( itab_synchro (1, myrank) < itab_synchro (1, myrank-1)  ) exit
!$          end do synchro

            do i = imin, imax
               V_old(i,j) = V(i,j)
               V(i,j) = omega1 * V_old(i,j) 
     &           + ( sou(i,j)
     &           + ve(i,j) * V(i+1,j  ) + vw(i,j) * V(i-1,j  ) 
     &           + vn(i,j) * V(i  ,j+1) + vs(i,j) * V(i  ,j-1) ) * vc(i,j)
               if ( abs(V(i,j) ) < 1d-20) V(i,j) = 0.0d0
            end do
!$OMP FLUSH(itab_synchro,V)
!$           itab_synchro (1, myrank) = j
!$OMP FLUSH(itab_synchro)
         end do

! calcul du critere de convergence
         erreur = 0.0d0
         do j = 1, ny
            do i = imin, imax
               erreur = max (erreur, abs (V(i,j) - V_old(i,j) ) )
            end do
         end do

!$OMP ATOMIC
         erreur_max = max (erreur_max, erreur)
!$OMP FLUSH (erreur_max)
!$OMP BARRIER
         erreur = erreur_max

! remise a zero du drapeau sur les colonnes
!$       itab_synchro (1, myrank) = 0
!$OMP FLUSH(itab_synchro)

! critere d'arret 
         if (erreur < erreur_tol) then
!$OMP SINGLE
            write (6,*) 'erreur_max ', erreur, ' it ', k
!$OMP END SINGLE NOWAIT
            go to 10
         end if

!$OMP BARRIER

      end do

!$OMP SINGLE
      write (6,*) 'erreur finale ', erreur
!$OMP END SINGLE NOWAIT

10    continue 
!$    t2 = OMP_Get_WTime () - t1
!$    if (myrank == 0)  write (6,'(A, F8.2)') 'Temps ecoule : ', t2
!
!$OMP END PARALLEL

!---------------------
! sortie des resultats
!---------------------
!     open (unit=41,file='Potentiel2D.dat',status='unknown',form='formatted')
!     write (41,'(3ES12.5)') ( ( x(i), y(j), V(i,j), j=1,ny), i=1,nx)
!     close (41)

!     open (unit=44,file='Potentiel1D.dat',status='unknown',form='formatted')
!     j = nx /2
!     write (44,*) (x(i), V(i,j), i = 1, nx)
!     close (44)

!Liberation des ressources
      DEALLOCATE (xm, ym, x, y, V, V_old, eps, vn, ve, vs, vw, vc, sou)

      stop
      end program laplace
