      program laplace
!
      USE MPI
!
      IMPLICIT NONE
!
      Integer,  Parameter  ::  rp = Kind (1.0d0)
!
      Real(rp),  Allocatable, Dimension(:)  ::  xm
      Real(rp),  Allocatable, Dimension(:)  ::  ym
!
      Real(rp),  Allocatable, Dimension(:)  ::  x
      Real(rp),  Allocatable, Dimension(:)  ::  y
!
      Real(rp),  Allocatable, Dimension(:,:)  ::  V, V_old
      Real(rp),  Allocatable, Dimension(:,:)  ::  eps
      Real(rp),  Allocatable, Dimension(:,:)  ::  vw, vs, vn, ve
      Real(rp),  Allocatable, Dimension(:,:)  ::  vc, sou
!
      Real(rp)  ::  distx, disty, dxx, dyy, omega1, t1, t2
      Real(rp)  ::  eps0, epsR, uc, ua, erreur_tol, vv
      Real(rp)  ::  omega, erreur_max, erreur_loc
!
      Integer   ::  nx, ny
      Integer   ::  i, j, k, iter_max
      Integer   ::  idielg, idield, nxdg, nxdd
!
      Integer   ::  imin, imax
      Integer   ::  nbproc, myrank, ierr
      Integer   ::  ip_y, comm1d
      Integer,  Dimension(3)  ::  itab
      Integer,  Dimension(1)  ::  idims
      Integer,  Dimension(1)  ::  icoords
      Logical,  Dimension(1)  ::  periods
      Integer,  Dimension(2)  ::  voisins
      Integer,  Parameter     ::  ivw = 1, ive = 2
      Logical  ::  reorg = .TRUE.
      Integer  ::  isendtag, irecvtag, ireq1, ireq2
      Integer,  Allocatable,  dimension(:)  ::  ireq3
      INTEGER,  DIMENSION(MPI_STATUS_SIZE)  ::  status
!
      Character (3)  ::  str
      Character(19)  ::  filename
      Character(80)  ::  nom
!
!-------------------
! Debut du programme
!-------------------
!
!Initialisation  de  MPI
      CALL  MPI_Init (ierr)
!
!Savoir  quel  processeur  je  suis
      CALL  MPI_Comm_Rank (MPI_COMM_WORLD, myrank, ierr)
!
!Connaitre le nombre total de processeurs
      CALL  MPI_Comm_Size (MPI_COMM_WORLD, nbproc, ierr)
!
      if ( myrank == 0) then
         read (5,*) itab(1), itab(2), itab(3), erreur_tol
      end if
      CALL MPI_BCast (itab, 3, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
      nx = itab(1)
      ny = itab(2)
      iter_max = itab(3)
      CALL MPI_BCast (erreur_tol, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

! "Gros" parametres d'execution
!     nx=1920
!     ny=1920
!     iter_max=5000
!     erreur_tol=1.0d-5
!
! "Petits" parametres d'execution
!     nx=1920
!     ny=1920
!     iter_max=5000
!     erreur_tol=1.0d-8


!--------------
! topologie MPI
!--------------
      idims(1:1) = (/ 0 /)
      CALL MPI_Dims_Create (nbproc, 1, idims, ierr)
!
!Creation de la grille de processeurs 1D sans periodicite
      periods(1:1) = (/ .FALSE. /)
!
      CALL MPI_Cart_Create (MPI_COMM_WORLD, 1, idims, periods, reorg, comm1d, ierr)
      CALL MPI_Comm_Rank   (comm1d, myrank, ierr)
!
!Recherche de ses 2 voisins pour chaque processeur
      voisins(1:2) = (/ MPI_PROC_NULL, MPI_PROC_NULL /)
!
!Recherche des voisins Ouest et Est
      CALL  MPI_Cart_Shift (comm1d, 0, 1, voisins(ivw), voisins(ive), ierr)
!
! Recherche des coordonnees dans la grille
      CALL MPI_CART_GET( comm1d, 1, idims, periods, icoords, ierr)
!
      CALL MPI_Get_Processor_Name (nom, i, ierr)
!
      WRITE (6,'(I4,1x,A,1x,4I4)') myrank, nom(1:i), voisins(1:2), idims(1), icoords(1)
!
!Determiner les indices de chaque sous-domaine
      imin = 1  +(icoords(1)*nx) / idims(1)
      imax = ((icoords(1)+1)*nx) / idims(1)
!
!Envoi de donnees regulierement espacees au voisin W et reception du voisin  E
!extraction des "x" dans les "y"
      CALL MPI_TYPE_VECTOR (
     &       ny,            ! nombre de blocs
     &       1,                      ! longueur d'un bloc
     &       imax-imin+3,            ! pas entre le debut de
     &       MPI_DOUBLE_PRECISION,   ! deux blocs consecutifs
     &       ip_y, ierr)
      CALL MPI_TYPE_COMMIT (ip_y, ierr)

!Allocation des tableaux
      ALLOCATE (xm(imin-1:imax), ym(0:ny), x(imin:imax), y(ny), 
     &          V  (imin-1:imax+1,0:ny+1), V_old(imin-1:imax+1,0:ny+1), 
     &          eps(imin-1:imax+1,0:ny+1), vn(imin:imax,1:ny), ve(imin:imax,1:ny),
     &          vs (imin:imax,1:ny),       vw(imin:imax,1:ny), vc(imin:imax,1:ny),
     &          sou(imin:imax,1:ny), ireq3(1:ny), STAT=ierr)
      IF (ierr /= 0) THEN
         Write (6,'(A)') 'ERREUR ALLOCATION DYNAMIQUE'
         CALL MPI_Abort (MPI_COMM_WORLD, -1, ierr)
         STOP
      END IF

!------------------------------------
! discretisaton du domaine de calcul
!------------------------------------
      erreur_max = 0.0d0

      distx = 1.0d0
      disty = 1.0d0
!    
      dxx = distx / Real(nx,rp)
      xm(imin-1) = real(imin-1,rp) * dxx
      do i = imin, imax
         xm(i) = xm(i-1) + dxx
         x (i) = (xm(i) + xm(i-1) ) * 0.5d0
      end do
! 
      dyy = disty / Real(ny,rp)
      ym(0) = 0.0d0
      do j = 1, ny
         ym(j) = ym(j-1) + dyy
         y (j) = (ym(j) + ym(j-1) ) * 0.5d0
      end do

!------------------------------------------------        
! initialisation des parametres et des variables
!------------------------------------------------
      eps0 = 8.85d-14  !permittivite (F/cm)   
!
      do j = 0, ny+1
         do i = imin-1, imax+1
            V  (i,j) = 0.0d0
            eps(i,j) = eps0
         end do
      end do

! choix des valeurs des potentiels fixes a gauche (uc) et a droite (ua)
      uc = 0.0d0
      ua = 1.0d0

! prise en compte des dielectriques 1 = oui ou 0 = non
      idielg = 1
      idield = 1

      nxdg = 11   ! epaisseur a gauche nxdg-1 mailles
      nxdd = 90   ! epaisseur a droite nx-nxdd mailles
      epsR = 5.0d0
!
      if (imin < nxdg) then
         if (idielg == 1) then
            if (imax > nxdg) write (6,*) 'dielectrique a gauche jusqu a ', xm(nxdg-1)
            do j = 1, ny
               do i = imin, min(imax,nxdg-1)
                  eps(i,j) = eps0 * epsR
               end do
            end do
         end if
      end if
!
      if (imax > nxdd) then
         if (idield == 1) then
            if (imin <nxdd) write(6,*)'dielectrique a droite depuis ', xm(nxdd)
            do j = 1, ny
               do i = max (imin,nxdd+1), imax
                  eps(i,j) = eps0 * epsR
               end do
            end do
         end if
      end if

!-------------------------------------
! Resolution de l'equation de Laplace
!-------------------------------------

! choix du critere de convergence
!      erreur_tol = 1.0d-5

! choix de la methode de resolution
! omega = 1 : Gauss-Seidel
! 1<omega<2 : Sur-relaxation
      omega  = 1.5d0
      omega1 = 1.0d0 - omega


! Definition des coefficients de l'equation de Laplace
! cas general
      do j = 1, ny
         do i = imin, imax
            ve(i,j) = 2.0d0 * eps(i,j) * eps(i+1,j  ) / ( (eps(i,j) + eps(i+1,j) ) * dxx**2)
            vn(i,j) = 2.0d0 * eps(i,j) * eps(i  ,j+1) / ( (eps(i,j) + eps(i,j+1) ) * dyy**2)
            vs(i,j) = 2.0d0 * eps(i,j) * eps(i  ,j-1) / ( (eps(i,j) + eps(i,j-1) ) * dyy**2)
            vw(i,j) = 2.0d0 * eps(i,j) * eps(i-1,j  ) / ( (eps(i,j) + eps(i-1,j) ) * dxx**2)
            vc(i,j) = omega / ( vw(i,j) + vn(i,j) + vs(i,j) + ve(i,j) )
            sou(i,j) = 0.0d0
         end do
      end do

! Conditions aux limites de Neumann en j = 1 et j = ny
         do i = imin, imax
            j = 1
            vs(i,j) = 0.0d0
            vc(i,j) = omega / ( vw(i,j) + vn(i,j)           + ve(i,j) )
            j = ny
            vn(i,j) = 0.0d0
            vc(i,j) = omega / ( vw(i,j)           + vs(i,j) + ve(i,j) )
         end do

! Conditions aux limites de Dirichlet en i = 1 et i = nx
      if (icoords(1) == 0) then
         i = 1
         do j = 1, ny
            vv       = 2.0d0 * eps(i,j) / (dxx)**2
            vc (i,j) = omega / ( vv      + vn(i,j) + vs(i,j) + ve(i,j) )
            sou(i,j) = vv * uc
            vw (i,j) = 0.0d0
         end do
      end if

      if (icoords(1) == idims(1)-1) then
         i = nx
         do j = 1, ny
            vv       = 2.0d0 * eps(i,j) / (dxx)**2
            vc (i,j) = omega / ( vw(i,j) + vn(i,j) + vs(i,j) + vv      )
            sou(i,j) = vv * ua
            ve (i,j) = 0.0d0
         end do
      end if

      t1 = MPI_WTime ()

! resolution sur k iterations
      do k = 1, iter_max
!
! Mise en memoire des potentiels a chaque iteration
         do j = 1, ny
            do i = imin, imax
               V_old(i,j) = V(i,j)
            end do
         end do
!
! Attente reception bord est, envoi bord ouest
         IF (k /= 1) THEN
            CALL MPI_Wait (ireq1, status, ierr)
            CALL MPI_Wait (ireq2, status, ierr)
         END IF
!
! Resolution de l'equation de Laplace
         do j = 1, ny
!
! Reception point fantome bord ouest
            CALL MPI_Recv (V(imin-1,j), 1, MPI_DOUBLE_PRECISION, voisins(ivw), j, comm1d, status, ierr)
!
! Validation envoi point bord est
            IF (k /= 1) Call MPI_Wait (ireq3(j), status, ierr)
!
            do i = imin, imax
               V(i,j) = omega1 * V_old(i,j)
     &                + ( sou(i,j)
     &                  + ve(i,j) * V(i+1,j  ) + vw(i,j) * V(i-1,j  )
     &                  + vn(i,j) * V(i  ,j+1) + vs(i,j) * V(i  ,j-1) ) * vc(i,j)
            end do
!
! Envoi point bord est
!            if ( voisins(ive) /= MPI_PROC_NULL)
            CALL MPI_ISend (V(imax,j), 1, MPI_DOUBLE_PRECISION, voisins(ive), j, comm1d, ireq3 (j), ierr)
!
         end do
!
! Rafraichissement interface bord est
         irecvtag = 3000+myrank
         CALL MPI_IRecv (V(imax+1,1), 1, ip_y, voisins(ive), irecvtag, comm1d, ireq1, ierr)
!
! Envoi bord ouest
         isendtag = 3000+voisins(ivw)
         CALL MPI_ISend (V(imin,1), 1, ip_y, voisins(ivw), isendtag, comm1d, ireq2, ierr)
!
! calcul du critere de convergence
         erreur_loc = 0.0d0
         do j = 1, ny
            do i = imin, imax
               erreur_loc = max (erreur_loc, abs (V(i,j) - V_old(i,j) ) )
            end do
         end do
         CALL MPI_AllReduce (erreur_loc, erreur_max, 1, MPI_DOUBLE_PRECISION, MPI_MAX, comm1d, ierr)

! critere d'arret 
         if (erreur_max < erreur_tol) then
            if (myrank == 0) write (6,*) 'erreur_max ', erreur_max, ' it ', k
            go to 10
         end if
  
         if ( MOD(k, 100) == 0) then
            if (myrank == 0) write (6,*) 'k = ', k, ' erreur = ', erreur_max
         end if

      end do

      if (myrank == 0) write (6,*) 'erreur_max finale : ', erreur_max, ' it ', iter_max

10    continue

      CALL MPI_Wait (ireq1, status, ierr)
      CALL MPI_Wait (ireq2, status, ierr)
!
      do j = 1, ny
         call MPI_Wait (ireq3(j), status, ierr)
      end do
!
      call mpi_barrier (mpi_comm_world, ierr)

      t2 = MPI_WTime () - t1
      if (myrank == 0) write (6,'(A, F8.2)') 'Temps ecoule : ', t2

!---------------------
! sortie des resultats
!---------------------
!     write (str, '(i3.3)') myrank
!     filename = 'Potentiel2D_' // str // '.dat'
!     open (unit=41, file=filename, status='unknown', form='formatted')
!     write (41,'(3ES12.5)') ( ( x(i), y(j), V(i,j), j=1, ny), i=imin, imax)
!     close (41)

!     if ( (nx/2 >= 1) .AND. (nx/2 <= ny) ) then
!        write (str, '(i3.3)') myrank
!        filename = 'Potentiel1D_' // str // '.dat'
!        open (unit=44,file=filename, status='unknown', form='formatted')
!        write (44,*) (x(i), V(i,j), i=imin, imax)
!        close (44)
!     end if
!
!Liberation des ressources
      DEALLOCATE (xm, ym, x, y, V, V_old, eps, vn, ve, vs, vw, vc, sou, ireq3)

!Desactivation  de  MPI
      CALL MPI_Type_Free (ip_y, ierr)
      Call MPI_Comm_Free (comm1d, ierr)
!
      CALL MPI_Finalize (ierr)
!
      stop
      end program laplace

