      program laplace_hyb
!
!     USE MPI
!
!$    USE OMP_LIB
!
      IMPLICIT NONE
!
      include "mpif.h"
!
      Integer,  Parameter  ::  rp = Kind (1.0d0)
!
      Real(rp),  Allocatable, Dimension(:)  ::  xm
      Real(rp),  Allocatable, Dimension(:)  ::  ym
!
      Real(rp),  Allocatable, Dimension(:)  ::  x
      Real(rp),  Allocatable, Dimension(:)  ::  y
!
      Real(rp),  Allocatable, Dimension(:,:)  ::  V, V_old
      Real(rp),  Allocatable, Dimension(:,:)  ::  eps
      Real(rp),  Allocatable, Dimension(:,:)  ::  vw, vs, vn, ve
      Real(rp),  Allocatable, Dimension(:,:)  ::  vc, sou
!
!$    Integer,   Dimension(0:64,-1:11)  ::  itab_synchro
!
      Real(rp)  ::  distx, disty, dxx, dyy, omega1, t1, t2
      Real(rp)  ::  eps0, epsR, uc, ua, erreur_tol, vv
      Real(rp)  ::  omega, erreur_max, erreur_loc
!
      Integer   ::  nx, ny
      Integer   ::  i, j, k, iter_max
      Integer   ::  idielg, idield, nxdg, nxdd
!
      Integer   ::  iMPI_imin , iMPI_imax, iMPI_jmin, iMPI_jmax
      Integer   ::  iOMP_imin , iOMP_imax, iOMP_jmin, iOMP_jmax
      Integer   ::  iOMP_imin1, iOMP_imax1
      Integer   ::  iMPI_NbProc, iMPI_Rank, ierr
      Integer   ::  iOMP_NbThd,  iOMP_Rank
      Integer   ::  ip_y, comm1d
      Integer,  Dimension(3)  ::  itab
      Integer,  Dimension(1)  ::  idims
      Integer,  Dimension(1)  ::  icoords
      Logical,  Dimension(1)  ::  periods
      Integer,  Dimension(2)  ::  voisins
      Integer,  Parameter     ::  ivw = 1, ive = 2
      Logical  ::  reorg = .TRUE.
      Integer  ::  isendtag, irecvtag, ireq1, ireq2
      Integer,  Allocatable,  dimension(:)  ::  ireq3
      INTEGER,  DIMENSION(MPI_STATUS_SIZE)  ::  status
      Integer  :: iProvided
!
      Character (3)  ::  str
      Character(19)  ::  filename
      Character(80)  ::  nom
!
!-------------------
! Debut du programme
!-------------------
!
!Initialisation  de  MPI
      CALL MPI_Init_Thread (MPI_THREAD_MULTIPLE, iProvided, ierr)
!
!Savoir  quel  processeur  je  suis
      CALL  MPI_Comm_Rank (MPI_COMM_WORLD, iMPI_Rank, ierr)
!
!Connaitre le nombre total de processeurs
      CALL  MPI_Comm_Size (MPI_COMM_WORLD, iMPI_NbProc, ierr)

      if ( iMPI_Rank == 0) then
         read (5,*) itab(1), itab(2), itab(3), erreur_tol
      end if
      CALL MPI_BCast (itab, 3, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
      nx = itab(1)
      ny = itab(2)
      iter_max = itab(3)
      CALL MPI_BCast (erreur_tol, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)

!--------------
! topologie MPI
!--------------
      idims(1:1) = (/0/)
      CALL MPI_Dims_Create (iMPI_NbProc, 1, idims, ierr)
!
!Creation de la grille de processeurs 1D sans periodicite
      periods(1:1) = (/ .FALSE. /)
!
      CALL MPI_Cart_Create (MPI_COMM_WORLD, 1, idims, periods, reorg, comm1d, ierr)
      CALL  MPI_Comm_Rank (comm1d, iMPI_Rank, ierr)
!
!Recherche de ses 4 voisins pour chaque processeur
      voisins(1:2) = (/ MPI_PROC_NULL, MPI_PROC_NULL /)
!
!Recherche des voisins Ouest et Est
      CALL  MPI_Cart_Shift (comm1d, 0, 1, voisins(ivw), voisins(ive), ierr)
!
! Recherche des coordonnees dans la grille
      CALL MPI_CART_GET( comm1d, 1, idims, periods, icoords, ierr)

      CALL MPI_Get_Processor_Name (nom, i, ierr)
!
      WRITE (6,'(I4,1x,A,1x,4I4)') iMPI_Rank, nom(1:i), voisins(1:2), idims(1), icoords(1)
!
!Determiner les indices de chaque sous-domaine
      iMPI_imin = 1  +(icoords(1)*nx) / idims(1)
      iMPI_imax = ((icoords(1)+1)*nx) / idims(1)
!
!Envoi de donnees regulierement espacees au voisin W et reception du voisin  E
!extraction des "x" dans les "y"
      CALL MPI_TYPE_VECTOR (
     &       ny,  ! nombre de blocs
     &       1,                      ! longueur d'un bloc
     &       iMPI_imax-iMPI_imin+3,  ! pas entre le debut de
     &       MPI_DOUBLE_PRECISION,   ! deux blocs consecutifs
     &       ip_y, ierr)
      CALL MPI_TYPE_COMMIT (ip_y, ierr)

!Allocation des tableaux
      ALLOCATE (xm (iMPI_imin-1:iMPI_imax), ym(0:ny), x(iMPI_imin:iMPI_imax), y(ny), 
     &          V  (iMPI_imin-1:iMPI_imax+1,0:ny+1), V_old(iMPI_imin-1:iMPI_imax+1,0:ny+1), 
     &          eps(iMPI_imin-1:iMPI_imax+1,0:ny+1), vn(iMPI_imin:iMPI_imax,1:ny),
     &          ve (iMPI_imin:iMPI_imax,1:ny),       vs(iMPI_imin:iMPI_imax,1:ny),
     &          vw (iMPI_imin:iMPI_imax,1:ny),       vc(iMPI_imin:iMPI_imax,1:ny),
     &          sou(iMPI_imin:iMPI_imax,1:ny), ireq3(ny), STAT=ierr)
      IF (ierr /= 0) THEN
         Write (6,'(A)') 'ERREUR ALLOCATION DYNAMIQUE'
         CALL MPI_Abort (MPI_COMM_WORLD, -1, ierr)
         STOP
      END IF
!
      erreur_max = 0.0d0
!$OMP PARALLEL DEFAULT(NONE)
!$OMP&SHARED (xm, ym, x, y, V, V_old, eps, vn, ve, vs, vw, vc, sou, erreur_max, erreur_tol, nx, ny)
!$OMP&SHARED (itab_synchro, iMPI_Rank, iMPI_NbProc, iMPI_imin, iMPI_imax, t1, t2)
!$OMP&SHARED (idims, comm1d, voisins, icoords, ip_y, iter_max)
!$OMP&SHARED (ireq1, ireq2, ireq3, MPI_IN_PLACE)
!$OMP&PRIVATE (i, j, k, distx, disty, dxx, dyy, eps0, epsR, ierr)
!$OMP&PRIVATE (uc, ua, erreur_loc, iOMP_Rank, iOMP_NbThd, omega1, vv, status, isendtag, irecvtag)
!$OMP&PRIVATE (omega, idielg, idield, nxdg, nxdd, iOMP_imin, iOMP_imax, iOMP_jmin, iOMP_jmax)
!$OMP&PRIVATE (iOMP_imin1, iOMP_imax1)
      iOMP_Rank = 0
!$    iOMP_Rank = OMP_GET_THREAD_NUM()
      iOMP_NbThd = 1
!$    iOMP_NbThd  = OMP_GET_NUM_THREADS()
!
!* Initialisation OpenMP
      if (iMPI_Rank == 0) then
!$OMP SINGLE
!$       IF (.TRUE.) THEN
!$          WRITE (6,'(A)')    'Execution en parallele hybride MPI + OpenMP'
!$          WRITE (6,'(A,I2)') 'Nombre de threads OpenMP par processus MPI : ', iOMP_NbThd
!$       ELSE
            WRITE (6,'(A)') 'Execution en parallele MPI simple'
!$       END IF
!$OMP END SINGLE NOWAIT
      end if
!
!------------------------------------
! discretisation du domaine de calcul
!------------------------------------
      iOMP_imin = iMPI_imin     + ( iOMP_Rank   *(iMPI_imax-iMPI_imin+1) ) / iOMP_Nbthd
      iOMP_imax = iMPI_imin - 1 + ((iOMP_Rank+1)*(iMPI_imax-iMPI_imin+1) ) / iOMP_Nbthd
!
      if (iOMP_imin == 1) then
         iOMP_imin1 = 0
      else
         iOMP_imin1 = iOMP_imin
      end if
      if (iOMP_imax == nx) then
         iOMP_imax1 = nx+1
      else
         iOMP_imax1 = iOMP_imax
      end if
!
      distx = 1.0d0
      disty = 1.0d0
!    
      dxx = distx / Real(nx,rp)
      dyy = disty / Real(ny,rp)
!$OMP SINGLE
      xm(iMPI_imin-1) = real(iMPI_imin-1,rp) * dxx
      do i = iMPI_imin, iMPI_imax
         xm(i) = xm(i-1) + dxx
         x (i) = (xm(i) + xm(i-1) ) * 0.5d0
      end do
!$OMP END SINGLE NOWAIT
!
! initialisation du drapeau sur les colonnes
!$OMP SINGLE
!$      itab_synchro(1,-1) = ny+1
!$      DO j = 0, iOMP_NbThd-1
!$         itab_synchro(1,j) = 0
!$      END DO
!$OMP END SINGLE NOWAIT
!
!$OMP SINGLE
      ym(0) = 0.0d0
      do j = 1, ny
         ym(j) = ym(j-1) + dyy
         y(j) = (ym(j) + ym(j-1) ) * 0.5d0
      end do
!$OMP END SINGLE
!
!------------------------------------------------        
! initialisation des parametres et des variables
!------------------------------------------------
      eps0 = 8.85d-14  !permittivite (F/cm)   
!
      do j = 0, ny+1
         do i = iOMP_imin1, iOMP_imax1
            V  (i,j) = 0.0d0
            eps(i,j) = eps0
         end do
      end do
!
! choix des valeurs des potentiels fixes a gauche (uc) et a droite (ua)
      uc = 0.0d0
      ua = 1.0d0
!
! prise en compte des dielectriques 1 = oui ou 0 = non
      idielg = 1
      idield = 1
!
      nxdg = 11   ! epaisseur a gauche nxdg-1 mailles
      nxdd = 90   ! epaisseur a droite nx-nxdd mailles
      epsR = 5.0d0
!
      if (iMPI_imin < nxdg) then
         if (idielg == 1) then
!$OMP SINGLE
            if (iMPI_imax > nxdg) write (6,*) 'dielectrique a gauche jusqu a ', xm(nxdg-1)
!$OMP END SINGLE NOWAIT
!$OMP DO
            do j = 1, ny
               do i = iMPI_imin, min(iMPI_imax,nxdg-1)
                  eps(i,j) = eps0 * epsR
               end do
            end do
!$OMP END DO NOWAIT
         end if
      end if
!
      if (iMPI_imax > nxdd) then
         if (idield == 1) then
!$OMP SINGLE
            if (iMPI_imin <nxdd) write(6,*)'dielectrique a droite depuis ', xm(nxdd)
!$OMP END SINGLE NOWAIT
!$OMP DO
            do j = 1, ny
               do i = max (iMPI_imin,nxdd+1), iMPI_imax
                  eps(i,j) = eps0 * epsR
               end do
            end do
!$OMP END DO NOWAIT
         end if
      end if
!
!-------------------------------------
! Resolution de l'equation de Laplace
!-------------------------------------
! 
! choix du critere de convergence
!      erreur_tol = 1.0d-5
!
! choix de la methode de resolution
! omega = 1 : Gauss-Seidel
! 1<omega<2 : Sur-relaxation
      omega  = 1.5d0
      omega1 = 1.0d0 - omega
!
! Definition des coefficients de l'equation de Laplace
! cas general
      do j = 1, ny
         do i = iOMP_imin, iOMP_imax
            ve(i,j) = 2.0d0 * eps(i,j) * eps(i+1,j  ) / ( (eps(i,j) + eps(i+1,j) ) * dxx**2)
            vn(i,j) = 2.0d0 * eps(i,j) * eps(i  ,j+1) / ( (eps(i,j) + eps(i,j+1) ) * dyy**2)
            vs(i,j) = 2.0d0 * eps(i,j) * eps(i  ,j-1) / ( (eps(i,j) + eps(i,j-1) ) * dyy**2)
            vw(i,j) = 2.0d0 * eps(i,j) * eps(i-1,j  ) / ( (eps(i,j) + eps(i-1,j) ) * dxx**2)
            vc(i,j) = omega / ( vw(i,j) + vn(i,j) + vs(i,j) + ve(i,j) )
            sou(i,j) = 0.0d0
         end do
      end do
!
! Conditions aux limites de Neumann en j = 1 et j = ny
      do i = iOMP_imin, iOMP_imax
         j = 1
         vs(i,j) = 0.0d0
         vc(i,j) = omega / ( vw(i,j) + vn(i,j)           + ve(i,j) )
         j = ny
         vn(i,j) = 0.0d0
         vc(i,j) = omega / ( vw(i,j)           + vs(i,j) + ve(i,j) )
      end do
!
! Conditions aux limites de Dirichlet en i = 1 et i = nx
      if (icoords(1) == 0) then
!$       if (iOMP_Rank == 0) then
            i = 1
            do j = 1, ny
               vv       = 2.0d0 * eps(i,j) / (dxx)**2
               vc (i,j) = omega / ( vv      + vn(i,j) + vs(i,j) + ve(i,j) )
               sou(i,j) = vv * uc
               vw (i,j) = 0.0d0
            end do
!$       end if
      end if
!
      if (icoords(1) == idims(1)-1) then
!$       if (iOMP_Rank == iOMP_Nbthd-1) then
            i = nx
            do j = 1, ny
               vv       = 2.0d0 * eps(i,j) / (dxx)**2
               vc (i,j) = omega / ( vw(i,j) + vn(i,j) + vs(i,j) + vv      )
               sou(i,j) = vv * ua
               ve (i,j) = 0.0d0
            end do
!$       end if
      end if
!
!$OMP SINGLE
      t1 = MPI_WTime ()
!$OMP END SINGLE NOWAIT
!
! resolution sur k iterations
      do k = 1, iter_max
!
!$OMP SINGLE
         erreur_max = 0.0d0
!$OMP FLUSH (erreur_max)
!$OMP END SINGLE
!
! Mise en memoire des potentiels a chaque iteration
         do j = 1, ny
            do i = iOMP_imin, iOMP_imax
               V_old(i,j) = V(i,j)
            end do
         end do
!
! Attente reception bord est, envoi bord sud et ouest
         IF (k /= 1) THEN
!$OMP SINGLE
            CALL MPI_Wait (ireq1, status, ierr)
!$OMP END SINGLE NOWAIT
!
!$OMP SINGLE
            CALL MPI_Wait (ireq2, status, ierr)
!$OMP END SINGLE !! NOWAIT
!
         END IF
!
! Resolution de l'equation de Laplace
         do j = 1, ny
!
! Reception point fantome bord ouest
!$          if (iOMP_Rank == 0) then
               CALL MPI_Recv (V(iMPI_imin-1,j), 1, MPI_DOUBLE_PRECISION, voisins(ivw), j, comm1d, status, ierr)
!$          end if
!
! Validation envoi point bord est
!$          if (iOMP_Rank == iOMP_nbThd-1) then
               IF (k /= 1) Call MPI_Wait (ireq3(j), status, ierr)
!$          end if
!
! attente sur le drapeau sur les colonnes
!$          synchro: do
!$OMP FLUSH(itab_synchro)
!$             if ( itab_synchro (1, iOMP_Rank) < itab_synchro (1, iOMP_Rank-1)  ) exit
!$          end do synchro

            do i = iOMP_imin, iOMP_imax
               V(i,j) = omega1 * V_old(i,j)
     &                + ( sou(i,j)
     &                  + ve(i,j) * V(i+1,j  ) + vw(i,j) * V(i-1,j  )
     &                  + vn(i,j) * V(i  ,j+1) + vs(i,j) * V(i  ,j-1) ) * vc(i,j)
            end do
!
! on met a jour son drapeau pour liberer du synchro la thread de rang suivant
!$OMP FLUSH(itab_synchro,V)
!$           itab_synchro (1, iOMP_Rank) = j
!$OMP FLUSH(itab_synchro)
!
! Envoi point bord est
!$          if (iOMP_Rank == iOMP_NbThd-1) then
               CALL MPI_ISend (V(iMPI_imax,j), 1, MPI_DOUBLE_PRECISION, voisins(ive), j, comm1d, ireq3(j), ierr)
!$          end if
!
         end do
!
! Rafraichissement interface bord est
!$OMP SINGLE
         irecvtag = 3000+iMPI_Rank
         CALL MPI_IRecv (V(iMPI_imax+1,1), 1, ip_y, voisins(ive), irecvtag, comm1d, ireq1, ierr)
!$OMP END SINGLE NOWAIT
!
! Rafraichissement interface bord ouest
!$OMP SINGLE
         isendtag = 3000+voisins(ivw)
         CALL MPI_ISend (V(iMPI_imin,1), 1, ip_y, voisins(ivw), isendtag, comm1d, ireq2, ierr)
!$OMP END SINGLE NOWAIT
!
! calcul du critere de convergence
!
! erreur locale a chaque thread OpenMP d'un processus MPI
         erreur_loc = 0.0d0
         do j = 1, ny
            do i = iOMP_imin, iOMP_imax
               erreur_loc = max (erreur_loc, abs (V(i,j) - V_old(i,j) ) )
            end do
         end do
!
! erreur locale (max des threads) d'un processus MPI
!$OMP ATOMIC
         erreur_max = max (erreur_max, erreur_loc)
!$OMP BARRIER

! erreur globale (max des processus MPI)
!$OMP SINGLE
         CALL MPI_AllReduce (MPI_IN_PLACE, erreur_max, 1, MPI_DOUBLE_PRECISION, MPI_MAX, comm1d, ierr)
!$OMP FLUSH (erreur_max)
!$OMP END SINGLE

! remise a zero du drapeau sur les colonnes
!$       itab_synchro (1, iOMP_Rank) = 0
!$OMP FLUSH(itab_synchro)

!
! critere d'arret 
         if (erreur_max < erreur_tol) then
!$OMP SINGLE
            if (iMPI_Rank == 0) write (6,*) 'erreur_max ', erreur_max, ' it ', k
!$OMP END SINGLE NOWAIT
            go to 10
         end if

!$OMP SINGLE
         if ( ( MOD(k, 100) == 0) .and. (iMPI_Rank == 0) ) write (6,*) 'k = ', k, ' erreur = ', erreur_max
!$OMP END SINGLE
!
      end do

!$OMP SINGLE
      if (iMPI_Rank == 0) write (6,*) 'erreur_max finale : ', erreur_max, ' it ', iter_max
!$OMP END SINGLE NOWAIT

10    continue 

!$OMP SINGLE
      CALL MPI_Wait (ireq1, status, ierr)
!$OMP END SINGLE NOWAIT
!
!$OMP SINGLE
      CALL MPI_Wait (ireq2, status, ierr)
!$OMP END SINGLE NOWAIT
!
!$    if (iOMP_Rank == iOMP_NbThd-1) then
            do j = 1, ny
               call MPI_Wait (ireq3(j), status, ierr)
            end do
!$    end if

!$OMP SINGLE
      t2 = MPI_WTime () - t1
      if (iMPI_Rank == 0) write (6,'(A, F8.2)') 'Temps ecoule : ', t2
!$OMP END SINGLE NOWAIT

!$OMP END PARALLEL

!---------------------
! sortie des resultats
!---------------------
!     write (str, '(i3.3)') myrank
!     filename = 'Potentiel2D_' // str // '.dat'
!     open (unit=41, file=filename, status='unknown', form='formatted')
!     write (41,'(3ES12.5)') ( ( x(i), y(j), V(i,j), j=jmin, jmax), i=imin, imax)
!     close (41)

!     if ( (nx/2 >= jmin) .AND. (nx/2 <= jmax) ) then
!        write (str, '(i3.3)') myrank
!        filename = 'Potentiel1D_' // str // '.dat'
!        open (unit=44,file=filename, status='unknown', form='formatted')
!        write (44,*) (x(i), V(i,j), i=imin, imax)
!        close (44)
!     end if
!
!Liberation des ressources
      DEALLOCATE (xm, ym, x, y, V, V_old, eps, vn, ve, vs, vw, vc, sou, ireq3)

!Desactivation  de  MPI
      CALL MPI_Type_Free (ip_y, ierr)
      Call MPI_Comm_Free (comm1d, ierr)
!
      CALL MPI_Finalize (ierr)
!
      stop
      end program laplace_hyb
