!    This program is free software: you can redistribute it and/or modify
!    it under the terms of the GNU General Public License as published by
!    the Free Software Foundation, either version 3 of the License, or
!    (at your option) any later version.
!
!    This program is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with this program.  If not, see <http://www.gnu.org/licenses/>.


      program laplace
!
      USE MPI
!
      IMPLICIT NONE
!
      Integer,  Parameter  ::  rp = Kind (1.0d0)
      Real(rp),  Allocatable, Dimension(:)  ::  xm
      Real(rp),  Allocatable, Dimension(:)  ::  ym
      Real(rp),  Allocatable, Dimension(:)  ::  x
      Real(rp),  Allocatable, Dimension(:)  ::  y
      Real(rp),  Allocatable, Dimension(:,:)  ::  V, V_old
      Real(rp),  Allocatable, Dimension(:,:)  ::  eps
      Real(rp),  Allocatable, Dimension(:,:)  ::  vw, vs, vn, ve
      Real(rp),  Allocatable, Dimension(:,:)  ::  vc, sou
      Real(rp)  ::  distx, disty, dxx, dyy, omega1, t1, t2
      Real(rp)  ::  eps0, epsR, uc, ua, erreur_tol, vv
      Real(rp)  ::  omega, erreur, erreur_max, erreur_loc
      Integer   ::  nx, ny
      Integer   ::  i, j, k, iter_max
      Integer   ::  idielg, idield, nxdg, nxdd
      Integer   ::  imin, imax, jmin, jmax
      Integer   ::  nbproc, myrank, ierr
      Integer   ::  ip_x, ip_y, comm2d
      Integer,  Dimension(3)  ::  itab
      Integer,  Dimension(2)  ::  idims
      Integer,  Dimension(2)  ::  icoords
      Logical,  Dimension(2)  ::  periods
      Integer,  Dimension(4)  ::  voisins
      Integer,  Parameter     ::  ivn = 1, ive = 2, ivs = 3, ivw = 4
      Logical  ::  reorg = .TRUE.
      Integer  ::  isendtag, irecvtag, ireq1, ireq2, ireq3, ireq4, ireq5
      INTEGER,  DIMENSION(MPI_STATUS_SIZE)  ::  status
      Integer  :: nb_req
!
      Character (3)  ::  str
      Character(19)  ::  filename
      Character(len=8) :: value

      CALL  MPI_Init (ierr)
      CALL  MPI_Comm_Rank (MPI_COMM_WORLD, myrank, ierr)
      CALL  MPI_Comm_Size (MPI_COMM_WORLD, nbproc, ierr)

      if ( myrank == 0 ) then
 
         call GET_COMMAND_ARGUMENT(1, value)
         read (value ,'(I10)') itab(1) ! Convert a string to a numeric value
         call GET_COMMAND_ARGUMENT(2, value)
         read (value ,'(I10)') itab(2) 
         call GET_COMMAND_ARGUMENT(3, value)
         read (value ,'(I10)') itab(3)
         call GET_COMMAND_ARGUMENT(4, value)
         read (value ,'(F8.3)') erreur_tol

      end if

      CALL MPI_BCast (itab, 3, MPI_INTEGER, 0, MPI_COMM_WORLD, ierr)
      nx = itab(1)
      ny = itab(2)
      iter_max = itab(3)
      CALL MPI_BCast (erreur_tol, 1, MPI_DOUBLE_PRECISION, 0, MPI_COMM_WORLD, ierr)
#ifdef _VERBOSE
      print*, myrank, itab(1),itab(2),itab(3),erreur_tol
#endif

!--------------
! topologie MPI
!--------------
      idims(1:2) = 0
      CALL MPI_Dims_Create (nbproc, 2, idims, ierr)
!
!Creation de la grille de processeurs 2D sans periodicite
      periods(1:2) = .FALSE.
!
      CALL MPI_Cart_Create (MPI_COMM_WORLD, 2, idims, periods, reorg, comm2d, ierr)
!
!Recherche de ses 4 voisins pour chaque processeur
      voisins(1:4) = (/ MPI_PROC_NULL, MPI_PROC_NULL, MPI_PROC_NULL, MPI_PROC_NULL /)
!
!Recherche des voisins Ouest et Est
      CALL  MPI_Cart_Shift (comm2d, 0, 1, voisins(ivw), voisins(ive), ierr)
!
!Recherche des voisins Nord  et Sud
      CALL  MPI_Cart_Shift (comm2d, 1, 1, voisins(ivs), voisins(ivn), ierr)
!
! Recherche des coordonnees dans la grille
      CALL MPI_CART_GET( comm2d, 2, idims, periods, icoords, ierr)

!
!Determiner les indices de chaque sous-domaine
      imin = 1  +(icoords(1)*nx) / idims(1)
      imax = ((icoords(1)+1)*nx) / idims(1)
      jmin = 1  +(icoords(2)*ny) / idims(2)
      jmax = ((icoords(2)+1)*ny) / idims(2)

#ifdef _VERBOSE
      WRITE (6,'(13I4)') myrank, voisins(1:4), idims(1:2), icoords(1:2), imin, imax, jmin, jmax
#endif
!
!Envoi de donnees contigues au voisin E et reception du voisin O
      CALL MPI_Type_Contiguous (imax-imin+1,          & ! longueur d'un bloc
                               MPI_DOUBLE_PRECISION, & ! type initial
                               ip_x, ierr)
      CALL MPI_Type_Commit (ip_x, ierr)
!
!Envoi de donnees regulierement espacees au voisin W et reception du voisin  E
!extraction des "x" dans les "y"
      CALL MPI_TYPE_VECTOR ( &
             jmax-jmin+1,          & ! nombre de blocs
             1,                    & ! longueur d'un bloc
             imax-imin+3,          & ! pas entre le debut de
             MPI_DOUBLE_PRECISION, & ! deux blocs consecutifs
             ip_y, ierr)
      CALL MPI_TYPE_COMMIT (ip_y, ierr)

!Allocation des tableaux
      ALLOCATE (xm(imin-1:imax), ym(jmin-1:jmax), x(imin:imax), y(jmin:jmax), &
                V(imin-1:imax+1,jmin-1:jmax+1), V_old(imin-1:imax+1,jmin-1:jmax+1), &
                eps(imin-1:imax+1,jmin-1:jmax+1), vn(imin:imax,jmin:jmax), ve(imin:imax,jmin:jmax), &
                vs(imin:imax,jmin:jmax), vw(imin:imax,jmin:jmax), vc(imin:imax,jmin:jmax), &
                sou(imin:imax,jmin:jmax), STAT=ierr)
      IF (ierr /= 0) THEN
         Write (6,'(A)') 'ERREUR ALLOCATION DYNAMIQUE'
         CALL MPI_Abort (MPI_COMM_WORLD, -1, ierr)
         STOP
      END IF

!------------------------------------
! discretisaton du domaine de calcul
!------------------------------------
      erreur_max = 0.0d0

      distx = 1.0d0
      disty = 1.0d0
!    
      dxx = distx / Real(nx,rp)
      xm(imin-1) = real(imin-1,rp) * dxx
      do i = imin, imax
         xm(i) = xm(i-1) + dxx
         x (i) = (xm(i) + xm(i-1) ) * 0.5d0
      end do
! 
      dyy = disty / Real(ny,rp)
      ym(jmin-1) = real(jmin-1,rp) * dyy
      do j = jmin, jmax
         ym(j) = ym(j-1) + dyy
         y (j) = (ym(j) + ym(j-1) ) * 0.5d0
      end do
      
!------------------------------------------------        
! initialisation des parametres et des variables
!------------------------------------------------
      eps0 = 8.85d-14  !permittivite (F/cm)   
!
      do j = jmin-1, jmax+1
         do i = imin-1, imax+1
            V  (i,j) = 0.0d0
            eps(i,j) = eps0
         end do
      end do

! choix des valeurs des potentiels fixes a gauche (uc) et a droite (ua)
      uc = 0.0d0
      ua = 1.0d0

! prise en compte des dielectriques 1 = oui ou 0 = non
      idielg = 1
      idield = 1

      nxdg = 11   ! epaisseur a gauche nxdg-1 mailles
      nxdd = 90   ! epaisseur a droite nx-nxdd mailles
      epsR = 5.0d0
!
      if (imin < nxdg) then
         if (idielg == 1) then
#ifdef _VERBOSE
            if (imax > nxdg) write (6,*) 'dielectrique a gauche jusqu a ', xm(nxdg-1)
#endif
            do j = jmin, jmax
               do i = imin, min(imax,nxdg-1)
                  eps(i,j) = eps0 * epsR
               end do
            end do
         end if
      end if
!
      if (imax > nxdd) then
         if (idield == 1) then
#ifdef _VERBOSE
            if (imin <nxdd) write(6,*)'dielectrique a droite depuis ', xm(nxdd)
#endif
            do j = jmin, jmax
               do i = max (imin,nxdd+1), imax
                  eps(i,j) = eps0 * epsR
               end do
            end do
         end if
      end if

! SOLVER
 
      call init_solver(dxx,dyy, &
                 eps, &
                 ve, &
                 vn, &
                 vs, &
                 vw, &
                 vc, &
                 sou, &
                 imin,imax,jmin,jmax, &
                omega, omega1)

! Conditions aux limites de Neumann en j = 1 et j = ny
      if (icoords(2) == 0) then
         j = 1
         do i = imin, imax
            vs(i,j) = 0.0d0
            vc(i,j) = omega / ( vw(i,j) + vn(i,j)           + ve(i,j) )
         end do
      end if

      if (icoords(2) == idims(2)-1) then
         j = ny
         do i = imin, imax
            vn(i,j) = 0.0d0
            vc(i,j) = omega / ( vw(i,j)           + vs(i,j) + ve(i,j) )
         end do
      end if

! Conditions aux limites de Dirichlet en i = 1 et i = nx
      if (icoords(1) == 0) then
         i = 1
         do j = jmin, jmax
            vv       = 2.0d0 * eps(i,j) / (dxx)**2
            vc (i,j) = omega / ( vv      + vn(i,j) + vs(i,j) + ve(i,j) )
            sou(i,j) = vv * uc
            vw (i,j) = 0.0d0
         end do
      end if

      if (icoords(1) == idims(1)-1) then
         i = nx
         do j = jmin, jmax
            vv       = 2.0d0 * eps(i,j) / (dxx)**2
            vc (i,j) = omega / ( vw(i,j) + vn(i,j) + vs(i,j) + vv      )
            sou(i,j) = vv * ua
            ve (i,j) = 0.0d0
         end do
      end if

      t1 = MPI_WTime ()

! resolution sur k iterations
      do k = 1, iter_max

         if ( MOD(k, 100) == 0) then
            if (myrank == 0) write (6,*) 'k = ', k, ' err = ', erreur_max
         end if
!
! Mise en memoire des potentiels a chaque iteration
         do j = jmin, jmax
            do i = imin, imax
               V_old(i,j) = V(i,j)
            end do
         end do
!
! Reception bord sud
         irecvtag = 1000+myrank
         CALL MPI_Recv (V(imin,jmin-1), 1, ip_x, voisins(ivs), irecvtag, comm2d, status, ierr)
!
! Resolution de l'equation de Laplace
         do j = jmin, jmax
!
! Reception point bord ouest
            CALL MPI_Recv (V(imin-1,j), 1, MPI_DOUBLE_PRECISION, voisins(ivw), j, comm2d, status, ierr)

            call solver(omega1, V,V_old, sou,ve,vw,vn,vs,vc, imin,imax,jmin,jmax,j)

! Envoi point bord est
            CALL MPI_ISend (V(imax,j), 1, MPI_DOUBLE_PRECISION, voisins(ive), j, comm2d, ireq4, ierr)
            CALL MPI_Request_Free (ireq4, ierr)
!
            if (j == jmin) then
! Envoi bord sud
               isendtag = 2000+voisins(ivs)
               CALL MPI_ISend (V(imin,jmin  ), 1, ip_x, voisins(ivs), isendtag, comm2d, ireq2, ierr)
               CALL MPI_Request_Free (ireq2, ierr)
            end if
!
         end do
!
! Rafraichissement interface bord est
         irecvtag = 3000+myrank
         CALL MPI_IRecv (V(imax+1,jmin), 1, ip_y, voisins(ive), irecvtag, comm2d, ireq3, ierr)
         CALL MPI_Request_Free (ireq3, ierr)

! Rafraichissement interface bord nord
         irecvtag = 2000+myrank
         CALL MPI_IRecv (V(imin,jmax+1), 1, ip_x, voisins(ivn), irecvtag, comm2d, ireq2, ierr)
         CALL MPI_Request_Free (ireq2, ierr)
!
! Envoi bord nord
         isendtag = 1000+voisins(ivn)
         CALL MPI_ISend (V(imin,jmax  ), 1, ip_x, voisins(ivn), isendtag, comm2d, ireq1, ierr)
         CALL MPI_Request_Free (ireq1, ierr)
!
! Rafraichissement interface bord ouest
         isendtag = 3000+voisins(ivw)
         CALL MPI_ISend (V(imin,jmin), 1, ip_y, voisins(ivw), isendtag, comm2d, ireq5, ierr)
         CALL MPI_Request_Free (ireq5, ierr)



! calcul du critere de convergence
         erreur_loc = 0.0d0
         do j = jmin, jmax
            do i = imin, imax
               erreur = abs (V(i,j) - V_old(i,j) )
               erreur_loc = max (erreur_loc, erreur)
            end do
         end do
         CALL MPI_AllReduce (erreur_loc, erreur_max, 1, MPI_DOUBLE_PRECISION, MPI_MAX, comm2d, ierr)

! critere d'arret 
         if (erreur_max < erreur_tol) then
            if (myrank == 0) write (6,*) 'erreur_max ', erreur_max, ' it ', k
            go to 10
         end if
  
      end do

      if (myrank == 0) write (6,*) 'erreur_max finale : ', erreur_max, ' it ', k

10    continue 

      t2 = MPI_WTime () - t1
      if (myrank == 0) write (6,'(A, F8.2)') 'Time (seconds) : ', t2

!
!Liberation des ressources
      DEALLOCATE (xm, ym, x, y, V, V_old, eps, vn, ve, vs, vw, vc, sou)

!Desactivation  de  MPI
      CALL MPI_Type_Free (ip_x, ierr)
      CALL MPI_Type_Free (ip_y, ierr)
      Call MPI_Comm_Free (comm2d, ierr)
      CALL MPI_Finalize (ierr)

end program laplace

subroutine init_solver(dxx,dyy, eps,ve,vn,vs,vw,vc,sou, imin,imax,jmin,jmax, omega,omega1)                      
                                                                                                                
        implicit none                                                                                           
        Integer,  Parameter  ::  rp = Kind (1.0d0)                                                              
        Integer, INTENT(IN)   :: imin,imax,jmin,jmax                                                            
        Real(rp), INTENT(IN)  :: dxx, dyy                                                                       
        Real(rp), INTENT(OUT) :: omega                                                                          
        Real(rp), INTENT(OUT) :: omega1                                                                         
        Real(rp), INTENT(IN),    Dimension(:,:) :: eps
        Real(rp), INTENT(INOUT), Dimension(:,:) :: ve, vn, vs, vw, vc, sou        
        Integer :: i,j                                                                                          
!-------------------------------------                                                                          
! Resolution de l'equation de Laplace                                                                           
!-------------------------------------                                                                          
                                                                                                                
! choix de la methode de resolution                                                                             
! omega = 1 : Gauss-Seidel
! 1<omega<2 : Sur-relaxation
      omega = 1.5d0
      omega1 = 1.0d0 - omega
 
! Definition des coefficients de l'equation de Laplace
! cas general
      do j = jmin, jmax
         do i = imin, imax
            ve(i,j) = 2.0d0 * eps(i,j) * eps(i+1,j  ) / ( (eps(i,j) + eps(i+1,j) ) * dxx**2)
            vn(i,j) = 2.0d0 * eps(i,j) * eps(i  ,j+1) / ( (eps(i,j) + eps(i,j+1) ) * dyy**2)
            vs(i,j) = 2.0d0 * eps(i,j) * eps(i  ,j-1) / ( (eps(i,j) + eps(i,j-1) ) * dyy**2)
            vw(i,j) = 2.0d0 * eps(i,j) * eps(i-1,j  ) / ( (eps(i,j) + eps(i-1,j) ) * dxx**2)
            vc(i,j) = omega / ( vw(i,j) + vn(i,j) + vs(i,j) + ve(i,j) )
            sou(i,j) = 0.0d0
         enddo
      enddo

end subroutine init_solver

subroutine solver(omega1, V,V_old, sou,ve,vw,vn,vs,vc, imin,imax,jmin,jmax, j)

        implicit none
        Integer,  Parameter  ::  rp = Kind (1.0d0)
        Real(rp), INTENT(INOUT), Dimension(imin-1:imax+1,jmin-1:jmax+1) :: V
        Real(rp), INTENT(IN), Dimension(imin-1:imax+1,jmin-1:jmax+1) :: V_old 
        Real(rp), INTENT(IN) :: omega1
        Real(rp), INTENT(IN), Dimension(imin:imax,jmin:jmax) :: sou, ve, vw, vn, vs, vc 
        Integer, INTENT(IN)   :: imin,imax,j,jmin,jmax
        Integer :: i, istat

        do i = imin, imax
        
                V(i,j) = omega1 * V_old(i,j) &
                      + ( sou(i,j)              &
                      + ve(i,j) * V(i+1,j  ) + vw(i,j) * V(i-1,j  ) &
                      + vn(i,j) * V(i  ,j+1) + vs(i,j) * V(i  ,j-1) ) * vc(i,j) 
                if ( abs(V(i,j) ) < 1d-20 ) V(i,j) = 0.0d0

        enddo

end subroutine solver
