! Point d'entree du programme. Cette fonction contient
! essentiellement la boucle d'integration 
program orego
  
  implicit none

  ! Nombre d'equations
  integer, parameter :: neq = 3
  ! Parametres des equations
  real(kind=kind(1.d0)),parameter  ::  p=77.27,q=0.161,r=8.375e-06
  ! tableau contenant la solution du systeme
  real(kind=kind(1.d0))  ::  s(neq)
  ! intervalle d'integration [t,tmax] et pas d'integration
  real(kind=kind(1.d0))  ::  t, tmax, dt
  ! compteur de pas, nombre de pas d'integration
  integer   ::  n,nmax,i

  ! Initialisation
  t=0
  tmax=360
  dt=0.01
  nmax = (tmax-t)/dt

  call init(neq,s)

  ! integration par rk4
  do n = 0,nmax
     t = n*dt
     ! afficher l'état actuel
     write(*,*) t,(s(i),i=1,neq)
     ! integrer d'un pas de temps dt
     call rk4(neq,s,dt,p,q,r)
  end do

  ! affichage de l'état final
  t = n*dt
  write(*,*) t,(s(i),i=1,neq)

end program orego

! Conditions initiales
subroutine init (neq,s)

  implicit none

  integer  ::  neq
  real(kind=kind(1.d0))  :: s(neq)

  s(1) = 1.
  s(2) = 2.
  s(3) = 3.

end subroutine init

! Calcul du second membre du systeme
subroutine derive(neq,s,ds,p,q,r)
  
  implicit none

  integer  ::  neq
  real(kind=kind(1.d0))  :: s(neq),ds(neq),p,q,r
  
  ds(1) = p*(s(2)-s(1)*s(2)+s(1)-r*s(1)*s(1))
  ds(2) = 1/p*(-s(2)-s(1)*s(2)+s(3))
  ds(3) = q*(s(1)-s(3))

end subroutine derive

! Cette fonction integre les equations d'evolution sur un pas de
!   temps dt, selon un schéma de Runge-Kutta d'ordre 4, en partant de
!   l'etat contenu dans le tableau u. Le nouvel etat du systeme
!   est finalement stocke dans le meme tableau. 
subroutine rk4(neq,u,dt,p,q,r)

  implicit none
  
  integer  ::  neq
  real(kind=kind(1.d0))  :: u(neq),dt,p,q,r

  real(kind=kind(1.d0))  :: utmp(neq),k1(neq),k2(neq),k3(neq),k4(neq)
  integer   :: i

  ! k1 = f(u)
  call derive(neq,u,k1,p,q,r)

  ! k2 = f(u+0.5*dt*k1)
  do i = 1,neq
     utmp(i) = u(i) + 0.5*dt*k1(i)
  end do
  call derive(neq,utmp,k2,p,q,r)

  ! k3 = f(u+0.5*dt*k2)
  do i = 1,neq
     utmp(i) = u(i) +0.5*dt*k2(i)
  end do
  call derive(neq,utmp,k3,p,q,r)

  ! k4 = f(u+dt*k3)
  do i = 1,neq
     utmp(i) = u(i) + dt*k3(i)
  end do
  call derive(neq,utmp,k4,p,q,r)

  ! u = utmp + dt/6 * (k1 + 2*k2 + 2*k3 + k4)
  do i = 1,neq
     u(i) = u(i) + dt/6.*(k1(i) + 2.*k2(i) + 2.*k3(i) + k4(i));
  end do

end subroutine rk4

