module CSR_mod
  implicit none
  
  !> CSR sparse structure
  !! @param n row matrix dimension
  !! @param nnz non zero entries
  !! @param val non zero values
  !! @param col_ind non zero column indices
  !! @param row_ptr 
  type CSR
     INTEGER :: n, nnz
     REAL(kind=8), POINTER, DIMENSION(:) :: val
     INTEGER, POINTER, DIMENSION(:) :: col_ind
     INTEGER, POINTER, DIMENSION(:) :: row_ptr
  end type CSR
contains
  !> matrix product y = A*x
  !! @param A CSR matrix
  !! @param x vector
  !! @param y vector
  subroutine matmult(A, x, y)
    TYPE(CSR) :: A
    REAL(kind=8), DIMENSION(A%n) :: x, y
    INTEGER :: i,j
    
    if (size(x) /= size(y) .or. size(x) /= A%n) then
       print*, 'incompatible size'
    end if
    
    y = 0.d0
    do i=1, A%n
       do j=A%row_ptr(i), A%row_ptr(i+1)-1
          y(i) = y(i) + A%val(j)*x(A%col_ind(j)) 
       end do
    end do
  end subroutine matmult
  
  !> print CSR matrix information
  !! @param A CSR matrix
  subroutine printCSR(A)
    TYPE(CSR) :: A
    INTEGER :: i,j
    
    do i=1, A%n
       do j=A%row_ptr(i), A%row_ptr(i+1)-1
          print*, '(', i, ',', A%col_ind(j),')  ',A%val(j)
       end do
    end do
  end subroutine printCSR
  
  !> create CSR matrix
  !! @param A CSR matrix
  !! @param n
  !! @param nnz
  subroutine createCSR(A, n, nnz)
      TYPE(CSR) :: A
      integer :: n, nnz

      A%n = n
      A%nnz = nnz
      allocate(A%row_ptr(n+1))
      allocate(A%val(nnz))
      allocate(A%col_ind(nnz))
  end subroutine createCSR

  !> destroy CSR matrix
  !! @param A CSR matrix
  subroutine destroyCSR(A)
      TYPE(CSR) :: A

      deallocate(A%val)
      deallocate(A%col_ind)
      deallocate(A%row_ptr)
  end subroutine destroyCSR
    
end module CSR_mod
  
