import numpy as np

def cg(A, b, x0, ldir, xdir, tol = 1e-6, itmax = 500, withInfo=False, verbose=False):
    """
    Gradient conjugue sequentiel

    Parametres d'entree:
    --------------------

    A: matrice du systeme a resoudre
    b: second membre
    x0: solution initiale
    ldir: liste des points Dirichlet
    xdir: conditions de Dirichlet
    tol: tolerance relative
          Defaut: 1e-6
    itmax: nombre d'iterations maximal
    verbose: affichage des iterations
             par defaut: False

    """

    x = x0.copy()
    if ldir is not None:
        x[ldir] = xdir

    g = A*x - b
    if ldir is not None:
        g[ldir] = 0.

    g0 = np.sqrt(np.dot(g, g))
    if g0 == 0.:
        g0 = 1.

    if verbose:
        print '|| Ax0 - b || =', g0

    w = g.copy()

    it = 0
    residual = []

    while it < itmax:
        Aw = A*w
        if ldir is not None:
            Aw[ldir] = 0.

        rho = -np.dot(g,w) / np.dot(Aw, w)

        x[:] = x + rho*w
        g[:] = g + rho*Aw

        res = np.sqrt(np.dot(g, g)) / g0
        residual.append(res)
        if verbose:
            print 'iteration', it, 'residu ->', res
        if res < tol:
            break

        gamma = -np.dot(g, Aw) / np.dot(Aw, w)
        w[:] = g + gamma*w

        it += 1

    if withInfo:
        return x, residual, it
    else:
        return x

def cgPara(A, b, x0, ldir, xdir, tol = 1e-6, itmax = 500, withInfo=False, verbose=False):
    """
    Gradient conjugue parallele

    Parametres d'entree:
    --------------------

    A: matrice du systeme a resoudre
    b: second membre
    x0: solution initiale
    ldir: liste des points Dirichlet
    xdir: conditions de Dirichlet
    tol: tolerance relative
          Defaut: 1e-6
    itmax: nombre d'iterations maximal
    verbose: affichage des iterations
             par defaut: False

    """

    x = x0.copy()
    if ldir is not None:
        x[ldir] = xdir

    g = A*x - b
    if ldir is not None:
        g[ldir] = 0.

    g0 = np.sqrt(np.dot(g, g))
    if g0 == 0.:
        g0 = 1.

    if verbose:
        print '|| Ax0 - b || =', g0

    w = g.copy()

    it = 0
    residual = []

    while it < itmax:
        Aw = A*w
        if ldir is not None:
            Aw[ldir] = 0.

        rho = -np.dot(g,w) / np.dot(Aw, w)

        x[:] = x + rho*w
        g[:] = g + rho*Aw

        res = np.sqrt(np.dot(g, g)) / g0
        residual.append(res)
        if verbose:
            print 'iteration', it, 'residu ->', res
        if res < tol:
            break

        gamma = -np.dot(g, Aw) / np.dot(Aw, w)
        w[:] = g + gamma*w

        it += 1

    if withInfo:
        return x, residual, it
    else:
        return x

