import scipy.sparse as sp
import numpy as np

def matElemMass(dx, dy):
      """
      Calcul de la matrice de masse elementaire sur un quadrangle

      Parametres d'entree:
      --------------------

      dx: pas d'espace suivant x
      dy: pas d'espace suivant y

      Sortie:
      -------

      matrice de rigidite elementaire

      """
      s = dx*dy
      return np.array([[s/9, s/18, s/18, s/36],
                       [s/18, s/9, s/36, s/18],
                       [s/18, s/36, s/9, s/18],
                       [s/36, s/18, s/18, s/9]])

def matElemStiff(dx, dy):
      """
      Calcul de la matrice de rigidite elementaire sur un quadrangle

      Parametres d'entree:
      --------------------

      dx: pas d'espace suivant x
      dy: pas d'espace suivant y

      Sortie:
      -------

      matrice de rigidite elementaire

      """
      s = dx*dy
      d = s*((1./(dx*dx*3.))+(1./(dy*dy*3.)))
      dh = s*(-(1./(dx*dx*3.))+(1./(dy*dy*6.)))
      dv = s*((1./(dx*dx*6.))-(1./(dy*dy*3.)))
      dd = s*(-(1./(dx*dx*6.))-(1./(dy*dy*6.)))

      return np.array([[d, dh, dd, dv],
                       [dh, d, dv, dd],
                       [dd, dv, d, dh],
                       [dv, dd, dh, d]])

def stiffAssemble(m):
      """
      Assemblage de la matrice de rigidite sur le maillage de calcul represente par des quadrangles

      Parametre d'entree:
      -------------------

      m: maillage

      Sortie:
      -------
      
      matrice de rigidite

      """
      size = m.numberOfNodes()
      
      e = m.elements[2][1]
      dx = m.nodes[e[:, 2], 0] - m.nodes[e[:, 0], 0] 
      dy = m.nodes[e[:, 2], 1] - m.nodes[e[:, 0], 1] 
      ae = matElemStiff(dx, dy)
      
      ind = np.indices((4,4))
      elem =  m.elements[2][1][:, ind]
      line = elem[:, 0].flatten()
      col = elem[:, 1].flatten()

      return sp.coo_matrix((ae.transpose().flatten(),[line, col])).tocsr()

def massAssemble(m):
      """
      Assemblage de la matrice de masse sur le maillage de calcul represente par des quadrangles

      Parametre d'entree:
      -------------------

      m: maillage

      Sortie:
      -------
      
      matrice de masse

      """
      size = m.numberOfNodes()
    
      e = m.elements[2][1]
      dx = m.nodes[e[:, 2], 0] - m.nodes[e[:, 0], 0] 
      dy = m.nodes[e[:, 2], 1] - m.nodes[e[:, 0], 1] 
      ae = matElemMass(dx, dy)
      
      ind = np.indices((4,4))
      elem =  m.elements[2][1][:, ind]
      line = elem[:, 0].flatten()
      col = elem[:, 1].flatten()

      return sp.coo_matrix((ae.transpose().flatten(),[line, col])).tocsr()

def buildEFLaplaciandMatrix(mesh):
    return stiffAssemble(mesh)

def buildRhs(mesh, b):
    mass = massAssemble(mesh)
    return mass*b

def buildDirichletBc(m, bc, border=[0, 1, 1, 0]):
      """
      Recherche des points Dirichlet et mise en place d'un vecteur contenant les valeurs aux bords de meme taille

      Parametres d'entree:
      --------------------

      m: un maillage
      bc: liste de 4 elements representant les conditions de Dirichlet pour les 4 bords du domaine rectangulaire [bas, droite, haut, gauche]
      border: lignes des bords [bas, droite, haut, gauche]
              par defaut:
                 y = 0 
                 x = 1 
                 y = 1
                 x = 0
      
      Sorties:
      --------

      xdir: vecteur des valeurs Dirichlet aux bords
      ldir: liste des indices des points Dirichlet du bord

      """
      ldir = None
      xdir = None
      
      c = [1, 0, 1, 0]

      for i in xrange(4):
            ind = np.where(m.nodes[:,c[i]] == border[i])[0]
    
            if ind.size != 0:
                  if ldir is None:
                        ldir = ind
                        xdir = bc[i]*np.ones(ldir.shape)
                  else:
                        ldir = np.concatenate((ldir, ind))
                        xdir = np.concatenate((xdir, bc[i]*np.ones(ind.shape)))

      return xdir, ldir

if __name__ == '__main__':
    import mesh
    import time

    x = np.linspace(0., 1., 101)
    m = mesh.Mesh() 
    m.fromCoords(x, x)

    t1 = time.time()
    A = assemble(m)

    u = m.nodes[:, 0] + m.nodes[:, 1]

    print 'execution time' , time.time() - t1
    print np.dot(u, A*u)


    
    
