#include <mpi.h>
#include "vtkMPICommunicator.h"
#include "vtkMPIController.h"
#include "vtkSynchronizedRenderWindows.h"
#include "vtkSynchronizedRenderers.h"
#include "vtkTestUtilities.h"
#include "vtkRegressionTestImage.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkOpenGLRenderWindow.h"
#include "vtkRenderer.h"
#include "vtkActor.h"
#include "vtkSphereSource.h"
#include "vtkPolyDataMapper.h"
#include "vtkLookupTable.h"
#include "vtkCamera.h"
#include "vtkPieceScalars.h"
#include "vtkProcess.h"
#include "vtkObjectFactory.h"

class MyProcess : public vtkProcess
{
public:
  static MyProcess *New();
  vtkTypeRevisionMacro(MyProcess, vtkProcess);
  
  /** Execute method (implemented below) */
  virtual void Execute();

  void SetArgs(int anArgc,
               char *anArgv[])
    {
    this->Argc=anArgc;
    this->Argv=anArgv;  
    }

  void CreatePipeline(vtkRenderer* renderer)
    {
    int num_procs = this->Controller->GetNumberOfProcesses();
    int my_id = this->Controller->GetLocalProcessId();

    vtkSphereSource* sphere = vtkSphereSource::New();
    sphere->SetPhiResolution(100);
    sphere->SetThetaResolution(100);
    
    // TODO: Add a vtkPieceScalars filter (piecescalars)
    vtkPieceScalars *piecescalars = vtkPieceScalars::New();
    piecescalars->SetInputConnection(sphere->GetOutputPort());

    // TODO: SetScalarModeToCellData()
    piecescalars->SetScalarModeToCellData();

    vtkPolyDataMapper* mapper = vtkPolyDataMapper::New();
    mapper->SetInputConnection(piecescalars->GetOutputPort());
    mapper->SetScalarModeToUseCellFieldData();
    mapper->SelectColorArray("Piece");
    mapper->SetScalarRange(0, num_procs-1);
    mapper->SetPiece(my_id);
    mapper->SetNumberOfPieces(num_procs);
    mapper->Update();

    vtkActor* actor = vtkActor::New();
    actor->SetMapper(mapper);
    renderer->AddActor(actor);

    actor->Delete();
    mapper->Delete();
    piecescalars->Delete();
    sphere->Delete();
    }
  
protected:
  MyProcess() { this->Argc = 0; this->Argv = NULL; }
  
  int Argc;
  char **Argv;
};

// Standard VTK Macros
vtkCxxRevisionMacro(MyProcess, "1.0");
vtkStandardNewMacro(MyProcess);

void MyProcess::Execute()
{
  int my_id = this->Controller->GetLocalProcessId();

  // Create a render window and a renderer
  vtkRenderWindow* renWin = vtkRenderWindow::New();
  renWin->DoubleBufferOn();
  vtkRenderer* renderer = vtkRenderer::New();
  renWin->AddRenderer(renderer);

  // TODO: Create a vtkSynchronizedRenderWindows
  vtkSynchronizedRenderWindows* syncWindows =
    vtkSynchronizedRenderWindows::New();

  syncWindows->SetIdentifier(1);
  // TODO: Added the render window (renWin) to the vtkSynchronizedRenderWindows

  syncWindows->SetRenderWindow(renWin);

  // TODO: Set the parallel controler to the vtkSynchronizedRenderWindows
  syncWindows->SetParallelController(this->Controller);

  vtkSynchronizedRenderers* syncRenderers = vtkSynchronizedRenderers::New();
  syncRenderers->SetRenderer(renderer);
  syncRenderers->SetParallelController(this->Controller);
  //syncRenderers->SetImageReductionFactor(3);

  // TODO: Call the create pipeline to do the rendering
  this->CreatePipeline(renderer);

  if (my_id == 0)
    {
    vtkRenderWindowInteractor* iren = vtkRenderWindowInteractor::New();
    iren->SetRenderWindow(renWin);
    iren->Start();
    iren->Delete();

    this->Controller->TriggerBreakRMIs();
    this->Controller->Barrier();
    }
  else
    {
    this->Controller->ProcessRMIs();
    this->Controller->Barrier();
    }

  renderer->Delete();
  renWin->Delete();
  syncWindows->Delete();
  syncRenderers->Delete();
}


int main(int argc, char **argv)
{
  // This is here to avoid false leak messages from vtkDebugLeaks when
  // using mpich. It appears that the root process which spawns all the
  // main processes waits in MPI_Init() and calls exit() when
  // the others are done, causing apparent memory leaks for any objects
  // created before MPI_Init().
  MPI_Init(&argc, &argv);

  // Note that this will create a vtkMPIController if MPI
  // is configured, vtkThreadedController otherwise.
  vtkMPIController *contr = vtkMPIController::New();
  contr->Initialize(&argc, &argv, 1);

  int numProcs = contr->GetNumberOfProcesses();

  if (numProcs < 2 && false)
    {
    cout << "This test requires at least 2 processes" << endl;
    contr->Delete();
    return 1;
    }

  vtkMultiProcessController::SetGlobalController(contr);

  // Create my process 
  MyProcess *p = MyProcess::New();
  p->SetArgs(argc,argv);
  
  // Add my process to the multiprocess controller
  contr->SetSingleProcessObject(p);
  contr->SingleMethodExecute();

  // Delete the objects
  p->Delete();
  contr->Finalize();
  contr->Delete();
  vtkMultiProcessController::SetGlobalController(0);
  return 0;
}
