#include "vtkMPIController.h"
#include "vtkSmartPointer.h"
#include "vtkRTAnalyticSource.h"
#include "vtkContourFilter.h"
#include "vtkImageData.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkXMLPPolyDataWriter.h"
#include "vtkCommunicator.h"

int main(int argc, char** argv)
{
  vtkSmartPointer<vtkMPIController> controller =
    vtkSmartPointer<vtkMPIController>::New();
  controller->Initialize(&argc, &argv);
  
  int myid = controller->GetLocalProcessId();
  int numProcs = controller->GetNumberOfProcesses();
  
  // Create a vtkRTAnalyticSource -> vtkContourFilter pipeline.
  // Set the contour value to 157.
  vtkSmartPointer<vtkRTAnalyticSource> source =
    vtkSmartPointer<vtkRTAnalyticSource>::New();
    
  vtkSmartPointer<vtkContourFilter> contour =
    vtkSmartPointer<vtkContourFilter>::New();
  contour->SetInputConnection(source->GetOutputPort());
  contour->SetNumberOfContours(1);
  contour->SetValue(0, 157);
  
  // Update the contour filter and print the number of output
  // polygons process 0.
  contour->Update();
  
  cerr << "Process (" << myid << "): " 
    << "Cells in output of contour on process (without change in request): "
    << contour->GetOutput()->GetNumberOfCells() << endl;

  // Update the contour filter in parallel and print the number of
  // output polygons on process 0.
  // Hint: Use myid, numProcs.
  // Hint: Use one of the SetUpdateExtent() methods on
  // vtkStreamingDemandDrivenPipeline.
  vtkStreamingDemandDrivenPipeline* exec =
    vtkStreamingDemandDrivenPipeline::SafeDownCast(contour->GetExecutive());
  exec->SetUpdateExtent(0, myid, numProcs, 0);
  
  contour->Update();

  vtkIdType numCells = contour->GetOutput()->GetNumberOfCells();
  cerr << "Process (" << myid << "): " 
    << "Cells in output of contour on process after requesting"
    " 1 piece from " << numProcs << " pieces: " << numCells << endl;
  
  // Print the total (from all processes) number of polygons.
  // Hint: Use vtkMPIController::Reduce()
  vtkIdType totalNumCells = 0;
  controller->Reduce(&numCells, &totalNumCells, 1, vtkCommunicator::SUM_OP, 0);
  cerr << "Process (" << myid << "): " 
    << "Cell count after reduction: " << totalNumCells << endl;
  
  // Write the output of the contour with a vtkXMLPPolyDataWriter.
  // Hint: don't forget to set NumberOfPieces, StartPiece and
  // EndPiece on the writer.
  vtkSmartPointer<vtkXMLPPolyDataWriter> writer =
    vtkSmartPointer<vtkXMLPPolyDataWriter>::New();
  writer->SetNumberOfPieces(numProcs);
  writer->SetStartPiece(myid);
  writer->SetEndPiece(myid);
  writer->SetFileName("contour.pvtp");
  writer->SetInputConnection(contour->GetOutputPort());
  writer->Write();

  controller->Finalize();
  return 0;
}
