/*
this file uses original content I found on the Web. Orginal author is K.T (see below)

*/
/********** U T I L I T Y ************************************
 * Utility functions for use with the Jacobi and SOR solvers *
 * Kadin Tseng, Boston University, November 1999             *
 *************************************************************/

#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#ifdef PARALLEL
#include <mpi.h>
#endif

#ifdef ADIOS
#include "adios.h"
#include "adios_types.h"
#endif

#include "VisItInterfaceTypes_V2.h"
#include "VisItControlInterface_V2.h"
#include "VisIt_NameList.h"
#include "VisIt_OptionList.h"

#include "solvers.h"
void set_initial_bc(simulation_data *sim)
{
/*********** Boundary Conditions ****************************************
 *  PDE: Laplacian u = 0;      0<=x<=1;  0<=y<=1                        *
 *  B.C.: u(x,0)=sin(pi*x); u(x,1)=sin(pi*x)*exp(-pi); u(0,y)=u(1,y)=0  *
 *  SOLUTION: u(x,y)=sin(pi*x)*exp(-pi*y)                               *
 ************************************************************************/
  int i;
  double x;
  memset(sim->Temp, 0, sizeof(double)*(sim->bx+2)*(sim->by+2));
  sim->iter = 0; sim->gdel = 1.0;
#ifdef PARALLEL
    if (sim->ranky == 0)
      {
      for (i = 0; i < (sim->bx+2); i++)
        {
        sim->Temp[i] = sin(M_PI*(i+ sim->rankx * sim->bx)/(sim->m+1));              /* at y = 0; all x */
        }
      }
    if (sim->ranky == (sim->cart_dims[1]-1)) {
      for (i = 0; i < (sim->bx+2); i++) {
        sim->Temp[i+(sim->bx+2)*(sim->by+1)] = sin(M_PI*sim->cx[i])*exp(-M_PI);   // at y = 1; all x
      }
    }
#else
    for (i = 0; i < (sim->m+2); i++)
      {
      x = sin(M_PI*i/(sim->m+1));    /* at y = 0; all x */
      sim->Temp[i] = x;    /* at y = 0; all x */
      sim->Temp[i+(sim->m+2)] = x*exp(-M_PI);   /* at y = 1; all x */
      }
#endif
  memset(sim->oldTemp, 0, sizeof(double)*(sim->bx+2)*(sim->by+2));
#ifdef _VISIT_
  VisItTimeStepChanged();
  VisItUpdatePlots();
#endif
}

void CopyTempValues_2_OldValues(simulation_data *sim)
{
  //save current solution array to buffer
  memcpy(sim->oldTemp, sim->Temp, sizeof(double)*(sim->bx+2)*(sim->by+2));
}

void simulate_one_timestep(simulation_data *sim)
{
  int savedFile = 0, exportedFile = 0;
  CopyTempValues_2_OldValues(sim);
/* compute Temp solution according to the Jacobi scheme */
  double del = update_jacobi(sim);
  /* find global max error */
#ifdef PARALLEL
  MPI_Allreduce( &del, &sim->gdel, 1, MPI_DOUBLE, MPI_MAX, sim->topocomm );
#else
  sim->gdel = del;
#endif
  
  if(sim->iter%INCREMENT == 0)  // only print global error every N steps
    {
#ifdef PARALLEL
    if(!sim->par_rank)
      {
      //fprintf(stdout,"iter,del,gdel: %6d, %lf %lf\n",sim->iter, del, sim->gdel);
      }
#else
    fprintf(stdout,"iter,del,gdel: %6d, %lf %lf\n", sim->iter, del, sim->gdel);
#endif
    }
#ifdef PARALLEL
  exchange_ghost_lines(sim); // update lowest and uppermost grid lines
#endif
  sim->iter++;
#ifdef _VISIT_
  VisItTimeStepChanged();
  VisItUpdatePlots();

  if(sim->iter%INCREMENT == 0)
    {
    if(sim->savingFiles)
    {
        char filename[100];
        sprintf(filename, "jacobi.%04d.png", sim->saveCounter);
        if(VisItSaveWindow(filename, 800, 800, VISIT_IMAGEFORMAT_PNG) == VISIT_OKAY)
        {
            savedFile = 1;
            if(sim->par_rank == 0)
                printf("Saved %s\n", filename);
        }
        else if(sim->par_rank == 0)
            printf("The image could not be saved to %s\n", filename);
    }

    if(sim->export)
    {
        char filename[100];
        visit_handle vars = VISIT_INVALID_HANDLE;
        VisIt_NameList_alloc(&vars);
        VisIt_NameList_addName(vars, "default");
#define TEST_VTK_OPTIONS 1
#ifdef TEST_VTK_OPTIONS
        {
            /* Create an option list that tells the VTK export to
             * blah blah blah.
             */
            visit_handle options = VISIT_INVALID_HANDLE;
            VisIt_OptionList_alloc(&options);
            VisIt_OptionList_setValueB(options, "Binary format", 1);
            VisIt_OptionList_setValueB(options, "XML format", 1);
            sprintf(filename, "updateplots_export%04d", sim->saveCounter);
            if(VisItExportDatabaseWithOptions(filename, "VTK_1.0", 
                                              vars, options) &&
               sim->par_rank == 0)
            {
                 printf("Exported %s\n", filename);
            }

            VisIt_OptionList_free(options);
        }
#else
        sprintf(filename, "updateplots_export%04d", sim->saveCounter);
        if(VisItExportDatabase(filename, "VTK_1.0", vars) &&
           sim->par_rank == 0)
        {
            printf("Exported %s\n", filename);
        }
#endif
        VisIt_NameList_free(vars);

        exportedFile = 1;
    }

    if(savedFile  || exportedFile)
        sim->saveCounter++;
    }
#endif
}

double update_jacobi(simulation_data *sim)
{
  int i, j;
  double del = 0.0;

  for (j = 1; j < sim->by+1; j++)
    {
    for (i = 1; i < sim->bx+1; i++)
      {
      sim->Temp[i+(sim->bx+2)*j] = ( sim->oldTemp[i+(sim->bx+2)*(j+1)] +
                           sim->oldTemp[(i+1)+(sim->bx+2)*j] +
                           sim->oldTemp[(i-1)+(sim->bx+2)*j] +
                           sim->oldTemp[i+(sim->bx+2)*(j-1)] )*0.25;
      del += fabs(sim->Temp[i+(sim->bx+2)*j] - sim->oldTemp[i+(sim->bx+2)*j]); /* find local max error */
      }
    }

  return del;
}

#ifdef PARALLEL
void exchange_ghost_lines(simulation_data *sim)
{
  MPI_Status status;

// send my last computed row north and receive from south my south boundary wall
  MPI_Sendrecv(&sim->Temp[0+sim->by*(sim->bx+2)], 1, rowtype, sim->south, 0,
               &sim->Temp[0+0*(sim->bx+2)], 1, rowtype, sim->north, 0,
                sim->topocomm, &status );
// send my first computed row south and receive from north my north boundary wall
  MPI_Sendrecv(&sim->Temp[0 + 1*(sim->bx+2) ], 1, rowtype, sim->north, 1,
               &sim->Temp[0+(sim->by+1)*(sim->bx+2)], 1, rowtype, sim->south, 1,
                sim->topocomm, &status );

// send my last computed column east and receive from west my west boundary wall
  MPI_Sendrecv(&sim->Temp[sim->bx + 1*(sim->bx+2)], 1, coltype, sim->east, 2,
               &sim->Temp[0  + 1*(sim->bx+2)], 1, coltype, sim->west, 2,
                sim->topocomm, &status );
// send my first computed column west and receive from east my east boundary wall
  MPI_Sendrecv(&sim->Temp[1  + 1*(sim->bx+2)], 1, coltype, sim->west, 3,
               &sim->Temp[(sim->bx+1) + 1*(sim->bx+2)], 1, coltype, sim->east, 3,
                sim->topocomm, &status );

}

void neighbors(simulation_data *sim)
{
  MPI_Cart_shift(sim->topocomm, 0, 1,  &sim->west, &sim->east); // in the 0-th dimension
  MPI_Cart_shift(sim->topocomm, 1, 1,  &sim->north, &sim->south);// in the 1-th dimension
}

void MPIIOWriteData(const char *filename, simulation_data *sim)
{
  // global size of array on disk
  char fname[256];
  strcpy(fname, filename);
  strcpy(&fname[strlen(filename)], ".bin");
  int dimuids[2]={sim->m+2, sim->m+2};
  int ustart[2], ucount[2];
  int disp = 0, offset=0;

  MPI_File      filehandle;
  MPI_Datatype  filetype;

  MPI_File_open(sim->topocomm, fname,
                          MPI_MODE_CREATE | MPI_MODE_WRONLY,
                          MPI_INFO_NULL, &filehandle);
  MPI_File_set_size(filehandle, disp);

// write the grid dimensions to allow a restart
/*
  if(sim->par_rank == 0)
    MPI_File_write(filehandle, dimuids, 2, MPI_INT, MPI_STATUS_IGNORE);

  disp = 2 * sizeof(int); // offset because we just wrote 2 integers
*/
  ustart[1] = sim->rankx * (sim->bx);
  ustart[0] = sim->ranky * (sim->by);
  ucount[1] = sim->bx+2;
  ucount[0] = sim->by+2;

 // Create the subarray representing the local block
  MPI_Type_create_subarray(2, dimuids, ucount, ustart,
                           MPI_ORDER_C, MPI_DOUBLE, &filetype);
  MPI_Type_commit(&filetype);

  MPI_File_set_view(filehandle, disp, MPI_DOUBLE,
                    filetype, "native", MPI_INFO_NULL);
  MPI_File_write_all(filehandle, sim->Temp, ucount[0]*ucount[1], MPI_DOUBLE, MPI_STATUS_IGNORE);
  MPI_File_close(&filehandle);
  MPI_Type_free(&filetype);
}

#ifdef ADIOS
void ADIOSWriteData(const char *filename, simulation_data *sim)
{
  char fname[256];
  strcpy(fname, filename);
  strcpy(&fname[strlen(filename)], ".bp");
  int ustart[2], ucount[2], gdims[2];
  uint64_t    adios_groupsize, adios_totalsize;
  int64_t     adios_handle;

  ustart[1] = sim->rankx * (sim->bx);
  ustart[0] = sim->ranky * (sim->by);
  ucount[1] = sim->bx+2;
  ucount[0] = sim->by+2;
  gdims[0] = gdims[1] = sim->m+2; 

  adios_init ("/users/jfavre/Projects/InSitu/Jacobi/pjacobi.xml");
  adios_open (&adios_handle, "writer2D", fname, "w", &sim->topocomm);

#include "gwrite_writer2D.ch"
  adios_close (adios_handle);
  MPI_Barrier (sim->topocomm);
  adios_finalize (sim->par_rank);
}
#endif

#endif
