# -*- coding: utf-8 -*-

from Scientific.Geometry import Vector
import numpy as N
import itertools
import time
import sys
import getopt

# Analyser la ligne de commande
opts, args = getopt.getopt(sys.argv[1:], 'vi:')
visualization = False
iterations = 1
for o, a in opts:
    if o == '-v':
        visualization = True
    elif o == '-i':
        iterations = int(a)

# Unités internes utilisées dans les calculs:
# longueur: 1e9 m
# temps: 1e5 s  (1.16 jours)
# masse: 1e24 kg
# force: 1e23 N (f = m*a reste valable sans préfacteurs)

# Quelques constantes en unités internes
AU = 149.6                             # Unité astronomique
h = 60*60/1.e5                         # Heure
G = 6.67428e-11/(1e9**3/1.e24/1.e5**2) # Constante gravitationnelle

MU = 0.45359237e28/1.e24      # Unité de masse dans le fichier input
                              # input (1e28 livres US)

delta_t = 0.05  # Pas d'intégration

# Unités dans le fichier systeme_solaire:
# longueur: AU
# temps: heure
# masse: 1028 livres US = 466.29296 kg

def group(seq, n):
    return itertools.izip(*[itertools.islice(seq, i, None, n)
                            for i in range(n)])
names = []
masses = []
radii = []
positions = []
velocities = []
for data in group(file('systeme_solaire').readlines(), 5):
    names.append(data[0].strip())
    positions.append([float(x)*AU for x in data[1].split()])
    velocities.append([float(x)*AU/h for x in data[2].split()])
    masses.append(float(data[3].strip())*MU)
    radii.append(float(data[4].strip())*AU/2.)

masses = N.array(masses)
positions = N.array(positions)
velocities = N.array(velocities)
radii = N.array(radii)

if visualization:
    import visual
    spheres = [visual.sphere(pos=visual.vector(p[0], p[1], p[2]),
                             radius = 10.,
                             color = visual.color.yellow)
               for p in positions]
    curves = [visual.curve(color = visual.color.green, radius=3.)
              for i in range(len(masses))]

def calc_forces(masses, positions):
    n = masses.shape[0]
    assert positions.shape == (n, 3)
    r_ij = positions[:, N.newaxis, :] - positions[N.newaxis, :, :]
    r_ij_abs = N.sqrt(N.sum(r_ij**2, axis=-1))
    r_ij_abs += r_ij_abs == 0.
    f_ij = G * masses[:, N.newaxis, N.newaxis] \
             * masses[N.newaxis, :, N.newaxis] \
             * r_ij/r_ij_abs[:, :, N.newaxis]**3
    return N.sum(f_ij, axis=0)

def move(masses, positions, velocities, time):
    n = len(masses)
    assert positions.shape == (n, 3)
    assert velocities.shape == (n, 3)
    assert time >= 0
    steps = int(time/delta_t)
    forces = calc_forces(masses, positions)
    for k in range(steps):
        v_midstep = velocities + 0.5*delta_t*forces/masses[:, N.newaxis]
        positions[:] = positions + delta_t*v_midstep
        if visualization:
            for i in range(n):
                spheres[i].pos = positions[i]
                curves[i].append(pos = positions[i])
        forces = calc_forces(masses, positions)
        velocities[:] = v_midstep + 0.5*delta_t*forces/masses[:, N.newaxis]

def center_of_mass(masses, positions):
    n = len(masses)
    assert positions.shape == (n, 3)
    return N.sum(masses[:, N.newaxis]*positions, axis=0)/N.sum(masses)

start = time.clock()
print center_of_mass(masses, positions)
for i in range(iterations):
    move(masses, positions, velocities, 365*24*h)
print center_of_mass(masses, positions)
end = time.clock()

print "CPU time: %d s" % (end-start)
