# -*- coding: utf-8 -*-

from Scientific.Geometry import Vector
from support import analyzeCommandLine, readData, \
     setupVisualization, updateVisualization
import time

# Unités internes utilisées dans les calculs:
# longueur: 1e9 m
# temps: 1e5 s  (1.16 jours)
# masse: 1e24 kg
# force: 1e23 N (f = m*a reste valable sans préfacteurs)

# Quelques constantes en unités internes
h = 60*60/1.e5                         # Heure
G = 6.67428e-11/(1e9**3/1.e24/1.e5**2) # Constante gravitationnelle

delta_t = 0.05  # Pas d'intégration

#
# Initialisation
#
visualization, iterations = analyzeCommandLine()

masses, positions, velocities, radii = readData('systeme_solaire')

if visualization:
    graphics = setupVisualization(positions)

#
# Simulation
#
def calc_forces(masses, positions):
    n = len(masses)
    assert n == len(positions)
    forces = [Vector(0., 0., 0.) for i in range(n)]
    for i in range(n):
        for j in range(i+1, n):
            r_ij = positions[i]-positions[j]
            r_ij_abs = r_ij.length()
            f_ij = G*masses[i]*masses[j]*r_ij/r_ij_abs**3
            forces[i] = forces[i] - f_ij
            forces[j] = forces[j] + f_ij
    return forces

def move(masses, positions, velocities, time):
    n = len(masses)
    assert n == len(positions)
    assert n == len(velocities)
    assert time >= 0
    steps = int(time/delta_t)
    forces = calc_forces(masses, positions)
    v_midstep = n*[None]
    for k in range(steps):
        for i in range(n):
            v_midstep[i] = velocities[i] + 0.5*delta_t*forces[i]/masses[i]
            positions[i] = positions[i] + delta_t*v_midstep[i]
            if visualization:
                updateVisualization(graphics[i], positions[i])
        forces = calc_forces(masses, positions)
        for i in range(n):
            velocities[i] = v_midstep[i] + 0.5*delta_t*forces[i]/masses[i]

def center_of_mass(masses, positions):
    n = len(masses)
    assert n == len(positions)
    s = Vector(0., 0., 0.)
    m = 0.
    for i in range(n):
        s = s + masses[i]*positions[i]
        m = m + masses[i]
    return s/m


print center_of_mass(masses, positions)
start = time.clock()
for i in range(iterations):
    move(masses, positions, velocities, 365*24*h)
end = time.clock()
print center_of_mass(masses, positions)

print "CPU time: %d s" % (end-start)
