# -*- coding: utf-8 -*-

import numpy as N
from support import analyzeCommandLine, readData, \
     setupVisualization, updateVisualization
import time

# Unités internes utilisées dans les calculs:
# longueur: 1e9 m
# temps: 1e5 s  (1.16 jours)
# masse: 1e24 kg
# force: 1e23 N (f = m*a reste valable sans préfacteurs)

# Quelques constantes en unités internes
h = 60*60/1.e5                         # Heure
G = 6.67428e-11/(1e9**3/1.e24/1.e5**2) # Constante gravitationnelle

delta_t = 0.05  # Pas d'intégration


#
# Initialisation
#
visualization, iterations = analyzeCommandLine()

masses, positions, velocities, radii = map(N.array, readData('systeme_solaire_complet'))

if visualization:
    graphics = setupVisualization(positions)

#
# Simulation
#
def calc_forces(masses, positions):
    n = masses.shape[0]
    assert positions.shape == (n, 3)
    r_ij = positions[:, N.newaxis, :] - positions[N.newaxis, :, :]
    r_ij_abs = N.sqrt(N.sum(r_ij**2, axis=-1))
    r_ij_abs += r_ij_abs == 0.
    f_ij = G * masses[:, N.newaxis, N.newaxis] \
             * masses[N.newaxis, :, N.newaxis] \
             * r_ij/r_ij_abs[:, :, N.newaxis]**3
    return N.sum(f_ij, axis=0)

def move(masses, positions, velocities, time):
    n = len(masses)
    assert positions.shape == (n, 3)
    assert velocities.shape == (n, 3)
    assert time >= 0
    steps = int(time/delta_t)
    forces = calc_forces(masses, positions)
    for k in range(steps):
        v_midstep = velocities + 0.5*delta_t*forces/masses[:, N.newaxis]
        positions[:] = positions + delta_t*v_midstep
        if visualization:
            for i in range(n):
                updateVisualization(graphics[i], positions[i])
        forces = calc_forces(masses, positions)
        velocities[:] = v_midstep + 0.5*delta_t*forces/masses[:, N.newaxis]

def center_of_mass(masses, positions):
    n = len(masses)
    assert positions.shape == (n, 3)
    return N.sum(masses[:, N.newaxis]*positions, axis=0)/N.sum(masses)

print center_of_mass(masses, positions)
start = time.clock()
for i in range(iterations):
    move(masses, positions, velocities, 365*24*h)
end = time.clock()
print center_of_mass(masses, positions)

print "CPU time: %d s" % (end-start)
