# -*- coding: utf-8 -*-

from support import analyzeCommandLine, readData, \
     setupVisualization, updateVisualization
import numpy
import time

# Unités internes utilisées dans les calculs:
# longueur: 1e9 m
# temps: 1e5 s  (1.16 jours)
# masse: 1e24 kg
# force: 1e23 N (f = m*a reste valable sans préfacteurs)

# Quelques constantes en unités internes
h = 60*60/1.e5                         # Heure
G = 6.67428e-11/(1e9**3/1.e24/1.e5**2) # Constante gravitationnelle

delta_t = 0.05  # Pas d'intégration

#
# Initialisation
#
visualization, iterations = analyzeCommandLine()

masses, positions, velocities, radii = map(numpy.array,
                                           readData('systeme_solaire'))

if visualization:
    graphics = setupVisualization(positions)

#
# Simulation
#
def calc_forces(masses, positions):
    n = len(masses)
    assert n == len(positions)
    forces = numpy.zeros((n, 3), numpy.float)
    for i in range(n):
        for j in range(i, n):
            r_ij = positions[i]-positions[j]
            r_ij_abs = numpy.sqr(numpy.dot(r_ij, r_ij))
            f_ij = G*masses[i]*masses[j]*r_ij/r_ij_abs**2
            forces[i] -= f_ij
            forces[j] += f_ij
    return forces

def move(masses, positions, velocities, time):
    n = len(masses)
    assert n == len(positions)
    assert n == len(velocities)
    assert time >= 0
    steps = int(time/delta_t)
    forces = calc_forces(masses, positions)
    v_midstep = numpy.zeros((n, 3), numpy.float)
    for k in range(steps):
        for i in range(steps):
            v_midstep[i] = velocities[i] + 0.5*delta_t*forces[i]/masses[i]
            positions[i] = positions[i] + delta_t*v_midstep[i]
            if visualization:
                updateVisualization(graphics[i], positions[i])
        forces = calc_forces(masses, positions)
        for i in range(n):
            velocities[i] = v_midstep[i] + 0.5*delta_t*forces[i]/masses[i]

def center_of_mass(masses, positions):
    n = len(masses)
    assert n == len(positions)
    s = numpy.zeros((3,), numpy.float)
    m = 0.
    for i in range(n):
        s += masses[i]*positions[i]
        m += masses[i]
    return s/m


print center_of_mass(masses, positions)
start = time.clock()
for i in range(iterations):
    move(masses, positions, velocities, 365*24*h)
end = time.clock()
print center_of_mass(masses, positions)

print "CPU time: %d s" % (end-start)
