/************************************************************************/
/** Interface python pour effectuer des opérations géométriques sur un **/
/** ensemble de points.                                                **/
/************************************************************************/
# include <Python.h>
# include <math.h>
# include <numpy/arrayobject.h>
# include "rotate.h"

static char translate_doc[] = "\
Translate un ensemble de points P suivant un vecteur T dans IR³, IR² ou IR. Range le résultat\
dans un troisième tableau TrP ";
static PyObject* PyTranslate( PyObject* self, PyObject* args )
{
  PyObject *ob1, *ob2, *ob3;
  PyArrayObject *pyTr;
  PyArrayObject *pyNodes1, *pyNodes2;
  double *tr, *pt1, *pt2;
  int dim[2];
  int n;

  pyTr = NULL; pyNodes1 = NULL; pyNodes2 = NULL;
  ob3 = NULL;
  if ( !PyArg_ParseTuple( args, "OO|O", &ob1, &ob2, &ob3 ) )
    {
      	PyErr_SetString(PyExc_TypeError,
			"Usage : translate(Nodes,vector, transNodes) ou transNodes = translate(Nodes,vector)");
	goto _fail;
    }
  /* On regarde si ob1 contient bien un tableau numpy à   deux dimensions */
  pyNodes1 = (PyArrayObject *)PyArray_ContiguousFromObject(ob1, PyArray_DOUBLE, 2, 2);
  // La fonction renvoie nulle si elle n'arrive pas à transformer ob1 en tableau
  // ( contenant ici des réels double précision ).
  if ( pyNodes1 == NULL ) 
    {
      PyErr_SetString(PyExc_TypeError, "Usage : translate(Nodes,vector, transNodes) ou transNodes = translate(Nodes,vector)");
      goto _fail;
    }
  // On regarde le nombre de noeuds : 
  dim[0] = PyArray_DIM(pyNodes1,0);
  // On regarde la dimension des noeuds :
  dim[1] = PyArray_DIM(pyNodes1,1);
  /* On regarde si ob2 contient bien un tableau à deux dimensions  contenant des
   réels double précision */
  pyTr = (PyArrayObject *)PyArray_ContiguousFromObject(ob2, PyArray_DOUBLE, 1, 1);
  if ( pyTr == NULL )
    {
      PyErr_SetString(PyExc_TypeError, "Usage : translate(Nodes,vector, transNodes) ou transNodes = translate(Nodes,vector)");
      goto _fail;
    }
  // On vérifie si Tr est bien un vecteur de même dimension que les noeuds :
  if ( PyArray_DIM(pyTr,0) != dim[1] )
    {
      PyErr_SetString(PyExc_TypeError,
		      "Le vecteur de translation doit être de dimension 3 !");
      goto _fail;
    }
  tr = (double*)PyArray_DATA(pyTr);
  if ( ob3 != NULL ) // Si on a passé le vecteur où stocker les résultats :
  {
    /* On regarde si ob3 contient  un tableau numpy à   deux dimensions */
    pyNodes2 = (PyArrayObject *)PyArray_ContiguousFromObject(ob3, PyArray_DOUBLE, 2, 2);
    // Vérification des dimensions de pyNodes2 :
    if ( PyArray_DIM(pyNodes2,0) != dim[0] )
      {
	PyErr_SetString(PyExc_TypeError,
			"Le tableau résultant doit avoir le même nombre de points que le tableau en entrée !");
	goto _fail;
      }
    if ( PyArray_DIM(pyNodes2,1) != dim[1] )
      {
	PyErr_SetString(PyExc_TypeError,
			"Le tableau résultant doit avoir des points de même dimensions que le tableau en entrée !");
	goto _fail;
      }
  }
  else // Sinon, on crée le tableau où stocker les résultats pour le renvoyer à Python :
  {
    pyNodes2 = (PyArrayObject*)PyArray_FromDims( 2, dim, PyArray_DOUBLE);
    if ( pyNodes2 == NULL )
    {
      PyErr_SetString(PyExc_MemoryError,
		      "L'allocation du tableau contenant les noeuds translatés a échoué !");
      goto _fail;
    }
  }
  // On définit ici la partie du code qui pourra marcher un multi--threads
  // sans avoir a prendre un lock global.
  Py_BEGIN_ALLOW_THREADS;
  // On calcul les noeuds translatés dans pyNodes2 :
  for ( n = 0; n < dim[0]; n++ )
    {
      int d;
      // On pointe pt1 sur le n ième noeud de pyNodes1 :      
      pt1 = (double*)PyArray_GETPTR2(pyNodes1,n,0);
      // On pointe pt2 sur le n ième noeud de pyNodes2 :
      pt2 = (double*)PyArray_GETPTR2(pyNodes2,n,0);
      // Calcul de la translatée :
      for ( d = 0; d < dim[1]; d++ ) pt2[d] = pt1[d] + tr[d];
    }
  // Fin de la section marchant en multi--threadings.
  Py_END_ALLOW_THREADS;
  // On décrémente les compteurs de références pour éviter des problèmes de mémoire :
  Py_DECREF(pyNodes1);
  Py_DECREF(pyTr);
  // Cette fonction ne retourne rien...
  if ( ob3 != NULL ) // Dans ce cas, on ne retourne rien car le tableau résultant a été donné en paramêtre
  {
    Py_DECREF(pyNodes2);// On décrément également le compteur pour le tableau résultat
    Py_RETURN_NONE;
  }
  else
    return (PyObject*)pyNodes2; // On ne le décrémente pas car passé à Python...
 _fail:
  Py_XDECREF(pyNodes1);
  Py_XDECREF(pyNodes2);
  Py_XDECREF(pyTr);
  return NULL;
}
// ========================================================================
static char rotate_doc[] = "\
Effectue une rotation des points P autour d'un axe axis avec un angle ang \
et range le résultat dans le tableau rotP.";
static PyObject* PyRotate( PyObject* self, PyObject* args )
{
  PyObject *ob1, *ob2, *ob3;
  double ang;
  PyArrayObject *pyAxis;
  PyArrayObject *pyNodes1, *pyNodes2;
  double *axis;
  int dim[2];

  pyAxis = NULL; pyNodes1 = NULL; pyNodes2 = NULL;
  ob3 = NULL;
  if ( !PyArg_ParseTuple( args, "OOd|O", &ob1, &ob2, &ang, &ob3 ) )
    {
      	PyErr_SetString(PyExc_TypeError,
			"Usage : rotate(Nodes, axis, angle, RotNodes)");
	goto _fail;
    }
  /* On regarde si ob1 contient bien un tableau numpy à   deux dimensions */
  pyNodes1 = (PyArrayObject *)PyArray_ContiguousFromObject(ob1, PyArray_DOUBLE, 2, 2);
  // La fonction renvoie nulle si elle n'arrive pas à transformer ob1 en tableau
  // ( contenant ici des réels double précision ).
  if ( pyNodes1 == NULL ) 
    {
      PyErr_SetString(PyExc_TypeError, "Usage : rotNodes = rotate(Nodes,axis,angle) ou rotate(Nodes,axis,angle,rotNodes)");
      goto _fail;
    }
  // On regarde le nombre de noeuds : 
  dim[0] = PyArray_DIM(pyNodes1,0);
  // On regarde la dimension des noeuds :
  dim[1] = PyArray_DIM(pyNodes1,1);
  /* On regarde si ob2 contient bien un tableau à deux dimensions  contenant des
   réels double précision */
  pyAxis = (PyArrayObject *)PyArray_ContiguousFromObject(ob2, PyArray_DOUBLE, 1, 1);
  if ( pyAxis == NULL )
    {
      PyErr_SetString(PyExc_TypeError, "Usage : rotNodes = rotate(Nodes,axis,angle) ou rotate(Nodes,axis,angle,rotNodes)");
      goto _fail;
    }
  // On vérifie si Tr est bien un vecteur de même dimension que les noeuds :
  if ( PyArray_DIM(pyAxis,0) != dim[1] )
    {
      PyErr_SetString(PyExc_TypeError,
		      "L'axe de rotation doit être de même dimension que les noeuds !");
      goto _fail;
    }
  axis = (double*)PyArray_DATA(pyAxis);
  if ( ob3 != NULL ) // Si le tableau résultant a été passé en paramêtre :
  {
    /* On regarde si ob3 contient bien un tableau numpy à   deux dimensions */
    pyNodes2 = (PyArrayObject *)PyArray_ContiguousFromObject(ob3, PyArray_DOUBLE, 2, 2);
    // Vérification des dimensions de pyNodes2 :
    if ( PyArray_DIM(pyNodes2,0) != dim[0] )
      {
	PyErr_SetString(PyExc_TypeError,
			"Le tableau résultant doit avoir le même nombre de points que le tableau en entrée !");
	goto _fail;
      }
    if ( PyArray_DIM(pyNodes2,1) != dim[1] )
      {
	PyErr_SetString(PyExc_TypeError,
			"Le tableau résultant doit avoir des points de même dimensions que le tableau en entrée !");
	goto _fail;
      }
  }
  else
  {
    pyNodes2 = (PyArrayObject*)PyArray_FromDims( 2, dim, PyArray_DOUBLE);
    if ( pyNodes2 == NULL )
    {
      PyErr_SetString(PyExc_MemoryError,
		      "L'allocation du tableau contenant les noeuds translatés a échoué !");
      goto _fail;
    }
  }
  // On définit ici la partie du code qui pourra marcher un multi--threads
  // sans avoir a prendre un lock global.
  Py_BEGIN_ALLOW_THREADS;
  rotate(dim[0], (const double*)PyArray_DATA(pyNodes1), axis, ang, 
	 (double*)PyArray_DATA(pyNodes2) );
  // Fin de la section marchant en multi--threadings.
  Py_END_ALLOW_THREADS;
  // On décrémente les compteurs de références pour éviter des problèmes de mémoire :
  Py_DECREF(pyNodes1);
  Py_DECREF(pyAxis);
  // Cette fonction ne retourne rien...
  if ( ob3 != NULL ) // Dans ce cas, on ne retourne rien car le tableau résultant a été donné en paramêtre
  {
    Py_DECREF(pyNodes2);// On décrément également le compteur pour le tableau résultat
    Py_RETURN_NONE;
  }
  else
    return (PyObject*)pyNodes2; // On ne le décrémente pas car passé à Python...
 _fail:
  Py_XDECREF(pyNodes1);
  Py_XDECREF(pyNodes2);
  Py_XDECREF(pyAxis);
  return NULL;
}
// ========================================================================
static PyMethodDef PyGeometry_Methods [] =
  {
    {"translate", PyTranslate, METH_VARARGS, translate_doc},
    {"rotate", PyRotate, METH_VARARGS, rotate_doc},
    {NULL,NULL}
  };
// ========================================================================
// Initialisation du module :
static char geometry_doc[] = "Quelques fonctions utiles en géométrie";

PyMODINIT_FUNC initCGeometry()
{
    PyObject *m;
    m = Py_InitModule4("CGeometry",PyGeometry_Methods, geometry_doc,
		       (PyObject*)NULL,PYTHON_API_VERSION);

    /* Très important : initialise numpy afin de pouvoir l'utiliser ici !!!! */
    import_array();

    /** XXXX Add constants here */
}
