import scipy.sparse as sp
from scipy import array

def laplacian(dx, dy, nx, ny, bc):
    """
    Compute the sparse matrix for a - laplacian term
    on a rectangular domain

    Parameters:
    -----------
        dx : mesh size in x direction

        dy : mesh size in y direction

        nx : number of point in x direction without Dirichlet points

        ny : number of point in y direction without Dirichlet points

        bc : array with boundary conditions on a rectangular domain
            bc[0] -> left boundary
            bc[1] -> bottom boundary
            bc[2] -> right boundary
            bc[3] -> top boundary
            
            boundary type:
            --------------
                - 'd': Dirichlet condition
                - 'n': Neumann condition
                - 'i': Interface condition

    Output:
    -------
        sparse matrix for a reaction term

    """
    # Dirichlet conditions
    coefLeft = [-1./dx**2]
    coefBottom = [-1./dy**2]
    coefRight = [-1./dx**2]
    coefTop = [-1./dy**2]

    adiag = [2./dx**2 + 2./dy**2]*nx*ny

    axinf = ([-1./dx**2]*(nx - 2) + coefRight + [0.])*ny

    axsup = ([0.] + coefLeft + [-1./dx**2]*(nx - 2))*ny

    ayinf = [-1./dy**2]*nx*(ny - 2) + coefTop*2*nx
    
    aysup = 2*coefBottom*nx + [-1./dy**2]*nx*(ny - 2)

    return sp.spdiags(array([adiag, axsup, axinf, aysup, ayinf]), 
                      [0, 1, -1, nx, -nx], 
                      nx*ny, nx*ny)

def grad_dx(dx, nx, ny, bc):
    """
    Compute the sparse matrix for a dx term
    on a rectangular domain

    Parameters:
    -----------
        dx : mesh size in x direction

        nx : number of point in x direction without Dirichlet points

        ny : number of point in y direction without Dirichlet points

        bc : array with boundary conditions on a rectangular domain
            bc[0] -> left boundary
            bc[1] -> bottom boundary
            bc[2] -> right boundary
            bc[3] -> top boundary

            boundary type:
            --------------
                - 'd': Dirichlet condition
                - 'n': Neumann condition
                - 'i': Interface condition

    Output:
    -------
        sparse matrix for a reaction term

    """

    coefCenterL = [0.]
    coefLeft = [1./(2.*dx)]

    coefCenterR = [0.]
    coefRight = [-1./(2.*dx)]

    adiag = (coefCenterL + [0.]*(nx - 2) + coefCenterR)*ny

    axinf = ([-1./(2.*dx)]*(nx - 2) + coefRight + [0.])*ny

    axsup = ([0.] + coefLeft + [1./(2.*dx)]*(nx - 2))*ny

    return sp.spdiags(array([axinf, adiag, axsup]), 
                      [-1, 0, 1], 
                      nx*ny, nx*ny)

def grad_dy(dy, nx, ny, bc):
    """
    Compute the sparse matrix for a dy term
    on a rectangular domain

    Parameters:
    -----------
        dy : mesh size in y direction

        nx : number of point in x direction without Dirichlet points

        ny : number of point in y direction without Dirichlet points

        bc : array with boundary conditions on a rectangular domain
            bc[0] -> left boundary
            bc[1] -> bottom boundary
            bc[2] -> right boundary
            bc[3] -> top boundary
            
            boundary type:
            --------------
                - 'd': Dirichlet condition
                - 'n': Neumann condition
                - 'i': Interface condition

    Output:
    -------
        sparse matrix for a reaction term

    """

    coefCenterB = [0.]
    coefBottom = [1./(2.*dy)]

    coefCenterT = [0.]
    coefTop = [-1./(2.*dy)]

    adiag = coefCenterB*nx + [0.]*nx*(ny - 2) + coefCenterT*nx

    ayinf = [-1./(2.*dy)]*nx*(ny - 2) + coefTop*2*nx

    aysup = coefBottom*2*nx + [1./(2.*dy)]*nx*(ny - 2)

    return sp.spdiags(array([aysup, adiag, ayinf]), 
                      [nx, 0, -nx], 
                      nx*ny, nx*ny)

def reac(nx, ny, bc):
    """
    Compute the sparse matrix for a reaction term
    on a rectangular domain

    Parameters:
    -----------
        nx : number of point in x direction without Dirichlet points

        ny : number of point in y direction without Dirichlet points

        bc : array with boundary conditions on a rectangular domain
            bc[0] -> left boundary
            bc[1] -> bottom boundary
            bc[2] -> right boundary
            bc[3] -> top boundary
            
            boundary type:
            --------------
                - 'd': Dirichlet condition
                - 'n': Neumann condition
                - 'i': Interface condition

    Output:
    -------
        sparse matrix for a reaction term

    """

    a = [1.]*nx*ny

    return sp.spdiags(array([a]), 
                      [0], 
                      nx*ny, nx*ny)
