import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
from torchvision import datasets, transforms, utils
from torch.autograd import Variable
import itertools

# Performance monitoring
from time import process_time
import matplotlib.pyplot as plt
import numpy as np


class AttrDict(dict):
    def __init__(self, *args, **kwargs):
        super(AttrDict, self).__init__(*args, **kwargs)
        self.__dict__ = self
        

args = AttrDict(
    batch_size      = 250,
    test_batch_size = 250,
    epochs          = 10,
    learning_rate   = .01,
    momentum        = .5,
    log_interval    = 10
)

imgsize = (28,28) # Default MNIST dataset

# Initialize the random seed, decide whether or not to use the GPU
SEED = 1
use_cuda = torch.cuda.is_available() 
torch.manual_seed(SEED)
if use_cuda: torch.cuda.manual_seed(SEED)

train_dataset = None
test_dataset = None
train_loader = None
test_loader  = None
class_names  = None

def display_parameters() :
    print('Optimizations will be made using the following arguments:\n', args)
    print('Using CUDA ? ', 'Yes' if use_cuda else 'No')
    print('Random seed : ', SEED)
    print('Remember that you can change those values dynamically!')
    print('(MU.args.batch_size = ...)')
    
def show(img):
    npimg = img.numpy()
    fig = plt.figure() # (figsize=(5,10))
    plt.imshow(np.transpose(npimg, (1,2,0)), interpolation='nearest')
    plt.tight_layout() ; fig.canvas.draw()

def generate_image(model, seed = 'random', mode = "class",
                   target_class = None, target_operator = None,
                   target_layer = None, target_neuron   = None,
                   optim_scheme = "SGD", verbose=True, lr = .1, nits = 200) :
    """
    Computes, naively, an image which triggers a strong reaction from the model.
    There are three modes available :
    - 'class', with target_class a class id (between 0 and 9 for MNIST dataset)
    - 'neuron', with target_operator a network operator (neuron at its output), 
                 and target_neuron the index of the neuron: 
                 choose an (index,) if you're targetting a Linear layer,
                 or a (Layer, Y, X) triple for a convolution layer.
    - 'layer', with target_operator a convolution layer,
               and target_layer a Layer index.
    
    The image is generated by gradient ascent from a noisy texture, a zero image
    or an image from the dataset.
    """
    if   seed == "random"   : image = torch.from_numpy(np.random.normal(size=(1,1,imgsize[0],imgsize[1]))).float()
    elif seed == "database" : image = test_dataset[0][0].view(1,1,imgsize[0],imgsize[1])
    elif seed == "zero"     : image = torch.zeros(1,1,imgsize[0],imgsize[1]).float()
    image = image.cuda() if use_cuda else image
    image = Variable(image, requires_grad=True)
    # Define optimizer for the image
    if   optim_scheme == "SGD"  : optimizer = optim.SGD( [image], lr = lr, momentum= .5)
    elif optim_scheme == "Adam" : optimizer = optim.Adam([image])
    
    
    generate_image.objective = 0.
    if   mode == "neuron" :
        if isinstance(target_neuron, int) :
            target_neuron = (target_neuron,)
        def hook_function(module, inp, out):
            generate_image.objective = -out[(0, *target_neuron)]
        hook_handle = target_operator.register_forward_hook(hook_function)
    elif mode == "layer" :
        if isinstance(target_layer, int) :
            target_layer = (target_layer,)
        def hook_function(module, inp, out):
            generate_image.objective = -out[(0, *target_layer)].sum()
        hook_handle = target_operator.register_forward_hook(hook_function)
        
    for i in range(1, nits):
        output = model(image) # Forward
        if   mode == "class" : objective = -output[0,target_class] # Target specific class
        elif mode == "neuron": objective = generate_image.objective
        elif mode == "layer" : objective = generate_image.objective
        model.zero_grad()    # Zero grads
        objective.backward() # Backward
        optimizer.step()     # Update image
    
    if mode == "neuron" or mode == "layer" : hook_handle.remove()
    
    if verbose :
        print('Loss', "{0:.2f}".format(objective.data.cpu().numpy()[0]))   
        
        softmax = torch.nn.Softmax()
        probas  = softmax(output).data.cpu().numpy()[0]
        print('Probas per class : ', ', '.join( '{0:.0f}%'.format(100*p) for p in probas))
    return image.view(imgsize[0],imgsize[1]).data.cpu().numpy()

# Confusion matrix code, from the scikit documentation

from sklearn.metrics import confusion_matrix

def plot_confusion_matrix(cm, classes,
                          normalize=False,
                          title='Confusion matrix',
                          cmap=plt.cm.Blues):
    """
    This function prints and plots the confusion matrix.
    Normalization can be applied by setting `normalize=True`.
    """
    if normalize:
        cm = cm.astype('float') / cm.sum(axis=1)[:, np.newaxis]
        print("Normalized confusion matrix")
    else:
        print('Confusion matrix, without normalization')

    #print(cm)

    plt.imshow(cm, interpolation='nearest', cmap=cmap)
    plt.title(title)
    plt.colorbar()
    tick_marks = np.arange(len(classes))
    plt.xticks(tick_marks, classes, rotation=45)
    plt.yticks(tick_marks, classes)

    fmt = '.2f' if normalize else 'd'
    thresh = cm.max() / 2.
    for i, j in itertools.product(range(cm.shape[0]), range(cm.shape[1])):
        plt.text(j, i, format(cm[i, j], fmt),
                 horizontalalignment="center",
                 color="white" if cm[i, j] > thresh else "black")

    plt.tight_layout()
    plt.ylabel('True label')
    plt.xlabel('Predicted label')

def confusion_diagnostic(model) :
    "Displays a synthetic matrix, which represents our classifier performances."
    y_test    = [] ; y_pred    = [] ; 
    for data, target in test_loader: # We won't load the testing dataset all at once:
        # Load the data on the GPU if needed, and wrap it into an autodiff object ----
        if use_cuda: data, target = data.cuda(), target.cuda()
        data, target = Variable(data, volatile=True), Variable(target)
        # Evaluation of our model on the test mini-batch -----------------------------
        output     = model(data) # Forward pass through the model
        pred       = output.data.max(1, keepdim=True)[1] # get the index of the max log-probability
        y_test.append(target.data.cpu().numpy()) ; y_pred.append(pred.view(-1).cpu().numpy())
    y_test = np.hstack(y_test) ; y_pred = np.hstack(y_pred)
    
    # Display --------------------------------------------------------------------   
    cnf_matrix = confusion_matrix(y_test, y_pred)
    fig = plt.figure()
    plot_confusion_matrix(cnf_matrix, classes=class_names,
                      title='Confusion matrix, without normalization')
    fig.canvas.draw()

def display_classified_images(model, true_class, predicted_class) :
    """
    Display the images from the test set with 'true_class' which are classified
    as 'predicted_class' by the model.
    """
    images = []
    for data, target in test_loader: # We won't load the testing dataset all at once:
        # Load the data on the GPU if needed, and wrap it into an autodiff object ----
        if use_cuda: data, target = data.cuda(), target.cuda()
        data, target = Variable(data, volatile=True), Variable(target)
        # Evaluation of our model on the test mini-batch -----------------------------
        output     = model(data) # Forward pass through the model
        pred       = output.data.max(1, keepdim=True)[1] # get the index of the max log-probability
        for (tc, pc, dat) in zip(target, pred, data) :
            if tc.eq(true_class).data.cpu().numpy() and pc.eq(predicted_class).cpu().numpy() :
                images.append(dat.data.cpu())
    show(utils.make_grid( images, nrow=4 ))







Loss = F.nll_loss # Use a standard classifier loss

def train(optimizer, model, epoch, fig, loss_line, time_0):
    "Makes a pass through the training dataset."
    model.train() # Tell the model to pass in "training" mode
   
    for batch_idx, (data, target) in enumerate(train_loader):   # Training loop :
        # Load the data on the GPU if needed, and wrap it into an autodiff object ----
        if use_cuda : data, target = data.cuda(), target.cuda()
        data, target = Variable(data), Variable(target)
        # "Gradient" descent step ----------------------------------------------------
        optimizer.zero_grad()  # Set to zero the gradient accumulation tensors
        output  = model(data)  # Forward pass through the model
        loss    = Loss(output, target) # Use a standard classifier loss
        loss.backward()        # Backpropoagation through the model
        optimizer.step()       # As gradients have been accumulated, we can make a step
        
        # Display --------------------------------------------------------------------   
        xd = loss_line.get_xdata() ; yd = loss_line.get_ydata()
        xd = np.append(xd, batch_idx / len(train_loader) + (epoch-1))
        yd = np.append(yd, loss.data[0])
        loss_line.set_xdata(xd)    ; loss_line.set_ydata(yd)
        fig.canvas.draw()
        """
        if batch_idx % args.log_interval == 0:
            print('Train Epoch: {} [{}/{} ({:.0f}%)]\tLoss: {:.6f}'.format(
                epoch, batch_idx * len(data), len(train_loader.dataset),
                100. * batch_idx / len(train_loader), loss.data[0]))
        """

def test(model, epoch, fig, accuracy_line, time_0):
    "Makes a pass through the testing dataset."
    model.eval()  # Tell the model to pass in "production" mode
    test_loss = 0 # Accumulate the cost which is minimized
    correct   = 0 # Accumulate the training success
    for data, target in test_loader: # We won't load the testing dataset all at once:
        # Load the data on the GPU if needed, and wrap it into an autodiff object ----
        if use_cuda: data, target = data.cuda(), target.cuda()
        data, target = Variable(data, volatile=True), Variable(target)
        
        # Evaluation of our model on the test mini-batch -----------------------------
        output     = model(data) # Forward pass through the model
        test_loss += Loss( output, target, size_average=False).data[0] # sum up batch loss
        pred       = output.data.max(1, keepdim=True)[1] # get the index of the max log-probability
        correct   += pred.eq(target.data.view_as(pred)).cpu().sum()
    # Display --------------------------------------------------------------------   
    test_loss /= len(test_loader.dataset)
    percentage = 100. * correct / len(test_loader.dataset)
    print('Time Elapsed: {:>7}s, Average test loss: {:.4f}, Test accuracy: {}/{} ({:.0f}%)'.format(
        '{0:.2f}'.format(process_time() - time_0), 
        test_loss, correct, len(test_loader.dataset),
        percentage))
    xd = accuracy_line.get_xdata() ; yd = accuracy_line.get_ydata()
    xd = np.append(xd, epoch)
    yd = np.append(yd, percentage)
    accuracy_line.set_xdata(xd)    ; accuracy_line.set_ydata(yd)
    fig.canvas.draw()

def evaluate_model(model) :
    "Trains and tests a classifier model."
    x = np.array([0.])
    y = np.array([0.])

    fig = plt.figure()
    ax1  = fig.add_subplot(111)
    ax1.set_xlim([0, args.epochs])
    ax1.set_ylim([0, 2])
    lossline, = ax1.plot(x, y, 'b-') 

    ax1.set_xlabel('Epochs')
    ax1.set_ylabel('Loss', color ='b')
    ax1.tick_params('y',   colors='b')

    ax2 = ax1.twinx()
    ax2.set_ylabel('Accuracy', color='r')
    ax2.tick_params('y', colors='r')
    ax2.set_ylim([50, 100])
    accuracyline, = ax2.plot(x, y, 'r-') 

    fig.tight_layout()
    
    time_0 = process_time()
    test( model, 0, fig, accuracyline, time_0)
    
     # Define an optimizer routine on the parameters of the model
    to_optimize = []
    for par in model.parameters() :
        if par.requires_grad : to_optimize.append(par)
    # optimizer = optim.SGD(to_optimize, lr = args.learning_rate, momentum=args.momentum)
    optimizer = optim.Adam(to_optimize)
    
    for epoch in range(1, args.epochs + 1):
        ax1.set_xlim([0, epoch]) ; ax2.set_xlim([0, epoch])
        train(optimizer, model, epoch, fig, lossline,     time_0)
        test(            model, epoch, fig, accuracyline, time_0)
    confusion_diagnostic(model)









